' TreeWalk.Bas - Support module for the TreeWalk routines.

CONST FALSE = 0, TRUE = NOT FALSE

' $INCLUDE: 'TREEWALK.BI'
' $INCLUDE: 'DIRFUN.BI'
' $INCLUDE: 'QUALNAME.BI'
' $INCLUDE: 'PARSPATH.BI'
' $INCLUDE: 'CURDIR.BI'
' $INCLUDE: 'CHDRIVE.BI'

' The ProcessDirectoryEntry routine declared below is contained in 
' the main module of the program using the TreeWalk support module.
' Each time you write a program to use TreeWalk, you'll need to write
' your own version of ProcessDirectoryEntry.
DECLARE SUB ProcessDirectoryEntry (PathSpec$, EntrySpec$, _
            DirEnt AS DirectoryRecord, Level%)


SUB TreeWalk (Spec$, ActionKey%)
' TreeWalk - Iterate all the files in the tree
'            starting with the path Spec$

WhereIWasWhenIStarted$ = CurDir$    ' Save current directory

' Qualify and parse the user's input specification

ParsePathname Spec$, JustDrive$, JustPath$, JustSpec$  ' parse input spec
ChDrive JustDrive$                    ' Change to specified drive
WhereDriveWasWhenItStarted$ = CurDir$ ' save cwd on specified drive
CHDIR JustPath$                       ' enter the starting directory

' Since we pass the "\" separator into the real tree walking code,
' we need to make sure we're not starting with the root directory.
' If we are, the path spec already ends with a "\".
FirstSep$ = ""
IF RIGHT$(JustPath$, 1) <> "\" THEN FirstSep$ = "\"

' Begin the treewalk.
TreeWalkHelper JustPath$ + FirstSep$, JustSpec$, ActionKey%, 1

CHDIR WhereDriveWasWhenItStarted$     ' reset cwd on specified drive

' Parse the original working directory into drive and path,
' and return there.
ParsePathname WhereIWasWhenIStarted$, JustDrive$, JustPath$, JustSpec$
ChDrive JustDrive$: CHDIR WhereIWasWhenIStarted$
' End of TreeWalk
END SUB

SUB TreeWalkHelper (PathSpec$, EntrySpec$, ActionKey%, Level%)
' TreeWalkHelper - Subprogram to perform the essential details
'                  of a recursive directory treewalk.

DIM twDTA AS DataTransferArea
DIM DirEnt AS DirectoryRecord
DIM Stat AS INTEGER

' Set up the data transfer area for this level of TreeWalkHelper
SetDTA twDTA

' Find the first entry in the current directory that matches
' the input specification. You can modify fAttr% below to filter
' out entries you don't want to process, (such as hidden & system files),
' Attribute value     Meaning
'         &H00         Normal file
'         &H01         Read-only
'         &H02         Hidden
'         &H04         System
'         &H08         Volume label
'         &H10         Directory/subdirectory
'         &H20         Archived
fAttr% = 2 OR 4 OR &H10

Stat = FindFirst(fAttr%, "*.*", DirEnt, twDTA)
DO WHILE Stat = 0
  ' Extract the filename from the directory entry record.
  EntryName$ = _
       MID$(DirEnt.FileName, 1, INSTR(DirEnt.FileName, CHR$(0)) - 1)
  ' If we're doing a top-down traversal, then we'll process the
  ' directory entry name here and now. Ideally, you'd use the
  ' top-down traversal to print all the names of files, count the
  ' number of files, accumulate a total size in bytes, etc.
  IF (ActionKey% AND TopDown) THEN
    ' However, if the directory entry is the one for either
    ' either the current or parent directories, ignore it. If
    ' you wre creating a TREE-oriented version of the DIR command,
    ' then you might NOT want to ignore these two special entries.
    IF EntryName$ <> "." AND EntryName$ <> ".." THEN
      ' Process a directory entry here. Or, rather, invoke
      ' the subprogram ProcessDirectoryEntry, which resides
      ' in your program's main source module.      
      ProcessDirectoryEntry PathSpec$, EntrySpec$, DirEnt, Level%
    END IF
  END IF
  ' If this directory entry is a subdirectory instead of a file,
  ' then it's time to recurse to the next level of the tree.
  IF DirEnt.FileAttb AND 16 THEN
    ' However, we'll get stuck in an infinite loop if we try
    ' to enter the current directory or the parent directory this way.
    IF EntryName$ <> "." AND EntryName$ <> ".." THEN
      CHDIR EntryName$    ' Enter the subdirectory
      TreeWalkHelper PathSpec$ + EntryName$ + "\", _
                     EntrySpec$, ActionKey%, Level% + 1
      CHDIR ".."          ' Exit from the subdirectory
    END IF
  END IF
  ' When we recursed to the next level, it's activation changed the
  ' data transfer area to a new one for it's level. Now we have to
  ' change the DTA back to our own.
  SetDTA twDTA
  ' If we're performing a bottoms-up traversal, then we process this
  ' entry after having "gone into it" (if it was a subdirectory).
  IF (ActionKey% AND BottomsUp) THEN
    IF EntryName$ <> "." AND EntryName$ <> ".." THEN
      ProcessDirectoryEntry PathSpec$, EntrySpec$, DirEnt, Level%
    END IF
  END IF
  ' Lastly, we need to see if there's another entry in this directory
  ' that requires processing.
  Stat = FindNext(DirEnt, twDTA)
LOOP
END SUB

