with Automobile_Interface; use Automobile_Interface;
-- **************************************************************
-- *                                                            *
-- *  Cruise                  									*  SPEC
-- *															*
-- **************************************************************
package Cruise is -- package specification
--|  Purpose
--|  Cruise provides a set of functions and procedures which
--|  implements a cruise control program.  Input function 
--|  Fetch_Key requires the use of the "stty raw" command.  
--|  After this command is entered, the "Return" key must be
--|  implemented using "Cntl-J."  The terminal may be reset 
--|  with the "stty -raw" command.  
--|  This package implements a cruise control program using the
--|  standard engage/disengage, set/resume cruise control 
--|  functions.

type CC_SET_RESUME_ACTION is (SET, RESUME, UNSET);
type ENGINE_ACTION is (ENGINE_OFF, ENGINE_ON);
type CC_ENGAGED_ACTION is (ENGAGE, DISENGAGE);
type CONTROLLER is (USER, CRUISE_CONTROL, USER_CRUISE);
type PEDAL_ACTION is (RELEASE_PEDALS, DEPRESS_BRAKE, RELEASE_BRAKE, DEPRESS_ACCEL, HOLD_ACCEL, RELEASE_ACCEL);

subtype BRAKE_ACTION is PEDAL_ACTION range DEPRESS_BRAKE..RELEASE_BRAKE;
subtype ACCEL_ACTION is PEDAL_ACTION range DEPRESS_ACCEL..RELEASE_ACCEL;
subtype SET_RESUME is CC_SET_RESUME_ACTION range SET..RESUME;
subtype AUTO_CC_FUNCTION is CHARACTER;   

--|  Global variables with initialization
cc_set_resume_status : CC_SET_RESUME_ACTION := UNSET;
cc_engaged_status : CC_ENGAGED_ACTION := DISENGAGE;
engine_status : ENGINE_ACTION := ENGINE_OFF;
auto_controller : CONTROLLER := USER;
pedal_operation : PEDAL_ACTION := RELEASE_PEDALS;
brake_status : BRAKE_ACTION := RELEASE_BRAKE;
accel_status : ACCEL_ACTION := RELEASE_ACCEL;
desired_speed : SPEED := 0.0;
key : CHARACTER := ' ';
quit : BOOLEAN := FALSE;

--...............................................................
--.																.
--.  Cruise.Decode_Function										.
--.																.
--...............................................................
procedure Decode_Function (func : AUTO_CC_FUNCTION);
--|  Purpose
--|  Decode input function requested by user.  This procedure
--|  calls the appropriate update procedure.

--...............................................................
--.																.
--.  Cruise.Update_Pedal_Status									.
--.																.
--...............................................................
procedure Update_Pedal_Status (pedal : PEDAL_ACTION);
--|  Purpose
--|  Update global variables pedal_operation, brake_status, and
--|  accel_status to the current action requested.

--...............................................................
--.																.
--.  Cruise.Update_Engine_Status								.
--.																.
--...............................................................
procedure Update_Engine_Status (engine : ENGINE_ACTION);
--|  Purpose
--|  Update global variable engine_status to the engine action
--|  requested.

--...............................................................
--.																.
--.  Cruise.Update_CC_Set_Resume_Status							.
--.																.
--...............................................................
procedure Update_CC_Set_Resume_Status (CC_set_action : CC_SET_RESUME_ACTION);
--|  Purpose
--|  Update global variable cc_set_resume_status to the 
--|  set/resume action  

--...............................................................
--.																.
--.  Cruise.Update_CC_Engaged_Status	 						.
--.																.
--...............................................................
procedure Update_CC_Engaged_Status (CC_engage_action : CC_ENGAGED_ACTION);
--|  Purpose
--|  Update global variable cc_engaged_status to the engaged
--|  action.

--...............................................................
--.																.
--.  Cruise.Print_Menu				 							.
--.																.
--...............................................................
procedure Print_Menu;
--|  Purpose
--|  Print user menu once when program begins.

--...............................................................
--.																.
--.  Cruise.Format_Display										.
--.																.
--...............................................................
procedure Format_Display	(speed_value    : SPEED; 
							engine_text     : in STRING;
							brake_text      : in STRING;
							accel_text      : in STRING;
							engaged_text    : in STRING;
							set_resume_text : in STRING);
--|  Purpose
--|  Format output for display.  

--...............................................................
--.																.
--.  Cruise.Determine_Control									.
--.																.
--...............................................................
procedure Determine_Control;
--|  Purpose
--|  Determine if cruise control or user is regulating the
--|  automobile speed

--...............................................................
--.																.
--.  Cruise.Fetch_Key											.
--.																.
--...............................................................
function Fetch_Key return BOOLEAN;
--|  Purpose
--|  Fetch user input

--...............................................................
--.																.
--.  Cruise.Maintain_Speed										.
--.																.
--...............................................................
procedure Maintain_Speed;
--|  Purpose
--|  Maintain automobile speed at desired speed.

--...............................................................
--.																.
--.  Cruise.Perform_Auto_Function								.
--.																.
--...............................................................
procedure Perform_Auto_Function;
--|  Purpose
--|  Apply the desired automobile function using package
--|  Automobile_Interface.

--...............................................................
--.																.
--.  Cruise.Update_Dashboard									.
--.																.
--...............................................................
procedure Update_Dashboard;
--|  Purpose
--|  Update output text to correspond to current automobile state.


end Cruise;
