-- *******************************************
-- *                                         *
-- *  Automobile_Interface                   *  BODY
-- *                                         *
-- *******************************************
package body Automobile_Interface is

  type AUTO_STATE is (STOPPED, DECELERATING, ACCELERATING, HOLDING);

  subtype ACCELERATION is FLOAT;

  Engine_Is_On      : BOOLEAN := FALSE;
  Brake_Is_On       : BOOLEAN := FALSE;
  Current_State     : AUTO_STATE := STOPPED;

  Current_Speed     : SPEED := 0.0;         -- MPH
  Starting_Speed    : SPEED := 0.0;         -- MPH
  Accel_Rate        : constant ACCELERATION := 2.0;  -- MPH/S
  Decel_Rate        : constant ACCELERATION := 1.0;  -- MPH/S
  Braking_Rate      : constant ACCELERATION := 3.0;  -- MPH/S
  Elapsed_Time      : NATURAL := 0;         -- Seconds

  -- ...........................................
  -- .                                         .
  -- .  Automobile_Interface.Turn_On_Engine    .  BODY
  -- .                                         .
  -- ...........................................
  procedure Turn_On_Engine is
  begin
    Engine_Is_On := TRUE;
  end Turn_On_Engine;

  -- ...........................................
  -- .                                         .
  -- .  Automobile_Interface.Turn_Off_Engine   .  BODY
  -- .                                         .
  -- ...........................................
  procedure Turn_Off_Engine is
  begin
    Engine_Is_On      := FALSE;
    Brake_Is_On       := FALSE;
    Current_State     := STOPPED;
    Current_Speed     := 0.0;
    Starting_Speed    := 0.0;
  end Turn_Off_Engine;

  -- ....................................................
  -- .                                                  .
  -- .  Automobile_Interface.Depress_Accelerator_Pedal  .  BODY
  -- .                                                  .
  -- ....................................................
  procedure Depress_Accelerator_Pedal is
  begin
    Elapsed_Time   := 0;
    Current_State  := ACCELERATING;
    Starting_Speed := Current_Speed;
    Brake_Is_On    := FALSE;
  end Depress_Accelerator_Pedal;

  -- ....................................................
  -- .                                                  .
  -- .  Automobile_Interface.Hold_Accelerator_Pedal     .  BODY
  -- .                                                  .
  -- ....................................................
  procedure Hold_Accelerator_Pedal is
  begin
    Current_State     := HOLDING;
    Brake_Is_On       := FALSE;
  end Hold_Accelerator_Pedal;

  -- ....................................................
  -- .                                                  .
  -- .  Automobile_Interface.Release_Accelerator_Pedal  .  BODY
  -- .                                                  .
  -- ....................................................
  procedure Release_Accelerator_Pedal is
  begin
    Elapsed_Time      := 0;
    Current_State     := DECELERATING;
    Starting_Speed    := Current_Speed;
    Brake_Is_On       := FALSE;
  end Release_Accelerator_Pedal;

  -- ..............................................
  -- .                                            .
  -- .  Automobile_Interface.Depress_Brake_Pedal  .  BODY
  -- .                                            .
  -- ..............................................
  procedure Depress_Brake_Pedal is
  begin
    Elapsed_Time      := 0;
    Current_State     := DECELERATING;
    Starting_Speed    := Current_Speed;
    Brake_Is_On       := TRUE;
  end Depress_Brake_Pedal;

  -- ..............................................
  -- .                                            .
  -- .  Automobile_Interface.Release_Brake_Pedal  .  BODY
  -- .                                            .
  -- ..............................................
  procedure Release_Brake_Pedal is
  begin
    Current_State := DECELERATING;
    Brake_Is_On       := FALSE;
  end Release_Brake_Pedal;

  -- ...........................................
  -- .                                         .
  -- .  Automobile_Interface.Sensed_Speed      .  BODY
  -- .                                         .
  -- ...........................................
  function Sensed_Speed return SPEED is
  begin
    return Current_Speed;
  end Sensed_Speed;

  -- ...........................................
  -- .                                         .
  -- .  Automobile_Interface.Update            .  BODY
  -- .                                         .
  -- ...........................................
  procedure Update is
  begin
    if Engine_Is_On then
      Elapsed_Time := Elapsed_Time + 1;
      case Current_State is
        when ACCELERATING =>
          begin
            Current_Speed := Starting_Speed +
                             SPEED(Accel_Rate * FLOAT(Elapsed_Time));
          exception
            when others =>
              Current_Speed := Maximum_Speed;
              Current_State := HOLDING;
          end;
        when DECELERATING =>
          begin
            if Brake_Is_On then
              Current_Speed := Starting_Speed -
                               SPEED(Braking_Rate * FLOAT(Elapsed_Time));
            else
              Current_Speed := Starting_Speed -
                               SPEED(Decel_Rate * FLOAT(Elapsed_Time));
            end if;
          exception
            when others =>
              Current_Speed := 0.0;
              Current_State := STOPPED;
          end;
        when HOLDING | STOPPED =>
          null;
      end case;
    end if;
  end Update;

begin -- Initialization
  Turn_Off_Engine;
end Automobile_Interface;
