/*********************************************************************
 *      common.c - common code
 * 
 *      Copyright (C) 1999-2000 Rui Sousa
 *********************************************************************
 *     This program is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU General Public License as
 *     published by the Free Software Foundation; either version 2 of
 *     the License, or (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 *     USA.
 *********************************************************************
*/

#include <stdlib.h>
#include <stdio.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <math.h>
#include <fcntl.h>
#include "common.h"

int setup(char *device, mode_t mode)
{
	int audio_fd;
	audio_buf_info ospace_info, ispace_info;
	int val, ret;

	if ((audio_fd = open(device, mode, 0)) == -1) {
		perror("open");
		exit(EXIT_FAILURE);
	}

	printf("\nSELECTED FORMAT\n");

	val = FORMAT;
	if ((ret = ioctl(audio_fd, SNDCTL_DSP_SETFMT, &val)) == -1)
		perror("SNDCTL_DSP_SETFMT");

	if (val != FORMAT)
		fprintf(stderr, "format %d not supported\n", FORMAT);

	printf("(SNDCTL_DSP_SETFMT)\nformat: %d ret: %d\n\n", val, ret);

	val = STEREO;
	if ((ret = ioctl(audio_fd, SNDCTL_DSP_STEREO, &val)) == -1)
		perror("SNDCTL_DSP_STEREO");

	if (val != STEREO)
		fprintf(stderr, "stereo mode %d not supported\n", STEREO);

	printf("(SNDCTL_DSP_STEREO)\nstereo: %d ret: %d\n\n", val, ret);

	val = SPEED;
	if ((ret = ioctl(audio_fd, SNDCTL_DSP_SPEED, &val)) == -1)
		perror("SNDCTL_DSP_SPEED");

	if ((double) val / SPEED - 1.0 > 0.05)
		fprintf(stderr, "speed %d not supported\n", SPEED);

	printf("(SNDCTL_DSP_SPEED)\nspeed: %d ret: %d\n\n", val, ret);

	printf("\nSELECTED BUFFER\n");

	val = (FRAGSHIFT & 0xffff) | ((FRAGS & 0xffff) << 16);
	if ((ret = ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &val)))
		perror("SNDCTL_DSP_SETFRAGMENT");

	printf("(SNDCTL_DSP_SETFRAGMENT)\nsize: %d number: %d ret: %d\n\n", 1 << FRAGSHIFT, FRAGS, ret);

	printf("\nACTUAL BUFFER\n");

	if ((ret = ioctl(audio_fd, SNDCTL_DSP_GETBLKSIZE, &val)) == -1)
		perror("SNDCTL_DSP_GETBLKSIZE");

	printf("(SNDCTL_DSP_GETBLKSIZE)\nfragsize: %d ret: %d\n\n", val, ret);

	if ((mode & O_WRONLY) || (mode & O_RDWR)) {

		if ((ret = ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &ospace_info)) == -1)
			perror("SNDCTL_DSP_GETOSPACE");

		printf
		    ("(SNDCTL_DSP_GETOSPACE)\nfragments: %d fragstotal: %d fragsize: %d bytes: %d ret: %d\n\n",
		     ospace_info.fragments, ospace_info.fragstotal, ospace_info.fragsize, ospace_info.bytes, ret);

		printf("\nPLAYING\n");

		printf("Left Channel\n");
		printf("Frequency     %7d Hz\nVolume            %3d %%\n", FREQL, VL);

		if (STEREO) {
			printf("\nRight Channel\n");
			printf("Frequency     %7d Hz\nVolume            %3d %%\n\n", FREQR, VR);
		}
	}

	if (!(mode & O_WRONLY) || (mode & O_RDWR)) {
		if ((ret = ioctl(audio_fd, SNDCTL_DSP_GETISPACE, &ispace_info)) == -1)
			perror("SNDCTL_DSP_GETISPACE");

		printf
		    ("(SNDCTL_DSP_GETISPACE)\nfragments: %d fragstotal: %d fragsize: %d bytes: %d ret: %d\n\n",
		     ispace_info.fragments, ispace_info.fragstotal, ispace_info.fragsize, ispace_info.bytes, ret);

	}

	fflush(stdout);

	getchar();

	return audio_fd;
}

static void generate_wave(struct wave *wave, double vol, int size)
{
	int i;

	if (vol > 100.0)
		vol = 100.0;
	else if (vol < 0.0)
		vol = 0.0;

	for (i = 0; i < size; i++)
		wave->buf[i] = vol / 100.0 * sin(wave->w * (i + wave->t0));

	wave->t0 += size;

	return;
}

void output_devbuf(__u8 * devbuf)
{
	int i;
	__s8 lt8, rt8;
	__s16 lt16, rt16;
	FILE *fp;

	if ((fp = fopen("output.dat", "a")) == NULL) {
		perror("output.dat");
		exit(EXIT_FAILURE);
	}

	switch (FORMAT) {

	case AFMT_S8:
		if (STEREO) {
			for (i = 0; i < BUF_SIZE / 2; i++)
				fprintf(fp, "%d %d\n", (__s8) devbuf[2 * i], (__s8) devbuf[2 * i + 1]);
		} else {
			for (i = 0; i < BUF_SIZE; i++)
				fprintf(fp, "%d\n", (__s8) devbuf[i]);
		}
		break;

	case AFMT_U8:
		if (STEREO) {
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt8 = devbuf[2 * i] ^ 0x80;
				rt8 = devbuf[2 * i + 1] ^ 0x80;
				fprintf(fp, "%d %d\n", lt8, rt8);
			}
		} else {
			for (i = 0; i < BUF_SIZE; i++) {
				lt8 = devbuf[i] ^ 0x80;
				fprintf(fp, "%d\n", (__s8) lt8);
			}
		}
		break;

	case AFMT_S16_LE:
		if (STEREO) {
			for (i = 0; i < BUF_SIZE / 4; i++) {
				lt16 = (devbuf[4 * i + 1] << 8) | devbuf[4 * i];
				rt16 = (devbuf[4 * i + 3] << 8) | devbuf[4 * i + 2]; 
				fprintf(fp, "%d %d\n", lt16, rt16);
			}
		} else {
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt16 = (devbuf[2 * i + 1] << 8) | devbuf[2 * i];
				fprintf(fp, "%d\n", lt16);
			}
		}
		break;

	case AFMT_S16_BE:
		if (STEREO) {
			for (i = 0; i < BUF_SIZE / 4; i++) {
				lt16 = (devbuf[4 * i] << 8) | devbuf[4 * i + 1]; 
				rt16 = (devbuf[4 * i + 2] << 8) | devbuf[4 * i + 3]; 
				fprintf(fp, "%d %d\n", lt16, rt16);
			}
		} else {
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt16 = (devbuf[2 * i] << 8) | devbuf[2 * i + 1];
				fprintf(fp, "%d\n", lt16);
			}
		}
		break;

	case AFMT_U16_LE:
		if (STEREO) {
			for (i = 0; i < BUF_SIZE / 4; i++) {
				lt16 = ((devbuf[4 * i + 1] << 8) | devbuf[4 * i]) ^ 0x8000;
				rt16 = ((devbuf[4 * i + 3] << 8) | devbuf[4 * i + 2]) ^ 0x8000; 
				fprintf(fp, "%d %d\n", lt16, rt16);
			}
		} else {
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt16 = ((devbuf[2 * i + 1] << 8) | devbuf[2 * i]) ^ 0x8000; 
				fprintf(fp, "%d\n", lt16);
			}
		}
		break;
	case AFMT_U16_BE:
		if (STEREO) {
			for (i = 0; i < BUF_SIZE / 4; i++) {
				lt16 = ((devbuf[4 * i] << 8) | devbuf[4 * i + 1]) ^ 0x8000;
				rt16 = ((devbuf[4 * i + 2] << 8) | devbuf[4 * i + 3]) ^ 0x8000;
				fprintf(fp, "%d %d\n", lt16, rt16);
			}
		} else {
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt16 = ((devbuf[2 * i] << 8) | devbuf[2 * i + 1]) ^ 0x8000; 
				fprintf(fp, "%d\n", lt16);
			}
		}
		break;

	default:
		break;
	}

	fclose(fp);

	return;
}



void fill_devbuf(struct wave *lwave, struct wave *rwave, __u8 * devbuf)
{
	int i;
	__u8 lt8, rt8;
	__u16 lt16, rt16;
	FILE *fp;

	if (WRITE_FILE)
		if ((fp = fopen("input.dat", "a")) == NULL) {
			perror("input.dat");
			exit(EXIT_FAILURE);
		}

	switch (FORMAT) {

	case AFMT_S8:
		if (STEREO) {
			generate_wave(rwave, VL, BUF_SIZE / 2);
			generate_wave(lwave, VR, BUF_SIZE / 2);
			for (i = 0; i < BUF_SIZE / 2; i++) {
				devbuf[2 * i] = 127.0 * lwave->buf[i];
				devbuf[2 * i + 1] = 127.0 * rwave->buf[i];
				if (WRITE_FILE)
					fprintf(fp, "%d %d\n", (__s8) devbuf[2 * i], (__s8) devbuf[2 * i + 1]);
			}
		} else {
			generate_wave(lwave, VL, BUF_SIZE);
			for (i = 0; i < BUF_SIZE; i++) {
				devbuf[i] = 127.0 * lwave->buf[i];
				if (WRITE_FILE)
					fprintf(fp, "%d\n", (__s8) devbuf[i]);
			}
		}
		break;

	case AFMT_U8:
		if (STEREO) {
			generate_wave(lwave, VL, BUF_SIZE / 2);
			generate_wave(rwave, VR, BUF_SIZE / 2);
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt8 = 127.0 * lwave->buf[i];
				rt8 = 127.0 * rwave->buf[i];
				devbuf[2 * i] = lt8 ^ 0x80;
				devbuf[2 * i + 1] = rt8 ^ 0x80;
				if (WRITE_FILE)
					fprintf(fp, "%d %d\n", (__s8) lt8, (__s8) rt8);
			}
		} else {
			generate_wave(lwave, VL, BUF_SIZE);
			for (i = 0; i < BUF_SIZE; i++) {
				lt8 = 127.0 * lwave->buf[i];
				devbuf[i] = lt8 ^ 0x80;
				if (WRITE_FILE)
					fprintf(fp, "%d\n", (__s8) lt8);
			}
		}
		break;

	case AFMT_S16_LE:
		if (STEREO) {
			generate_wave(lwave, VL, BUF_SIZE / 4);
			generate_wave(rwave, VR, BUF_SIZE / 4);

			for (i = 0; i < BUF_SIZE / 4; i++) {
				lt16 = (__u16) (32767.0 * lwave->buf[i]);
				rt16 = (__u16) (32767.0 * rwave->buf[i]);
				devbuf[4 * i] = lt16 & 0x00ff;
				devbuf[4 * i + 1] = (lt16 >> 8) & 0x00ff;
				devbuf[4 * i + 2] = rt16 & 0x00ff;
				devbuf[4 * i + 3] = (rt16 >> 8) & 0x00ff;
				if (WRITE_FILE)
					fprintf(fp, "%d %d\n", (__s16) lt16, (__s16) rt16);
			}
		} else {
			generate_wave(lwave, VL, BUF_SIZE / 2);
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt16 = (__u16) (32767.0 * lwave->buf[i]);
				devbuf[2 * i] = lt16 & 0x00ff;
				devbuf[2 * i + 1] = (lt16 >> 8) & 0x00ff;
				if (WRITE_FILE)
					fprintf(fp, "%d\n", (__s16) lt16);
			}
		}
		break;

	case AFMT_S16_BE:
		if (STEREO) {
			generate_wave(lwave, VL, BUF_SIZE / 4);
			generate_wave(rwave, VR, BUF_SIZE / 4);
			for (i = 0; i < BUF_SIZE / 4; i++) {
				lt16 = (__u16) (32767.0 * lwave->buf[i]);
				rt16 = (__u16) (32767.0 * rwave->buf[i]);
				devbuf[4 * i] = (lt16 >> 8) & 0x00ff;
				devbuf[4 * i + 1] = lt16 & 0x00ff;
				devbuf[4 * i + 2] = (rt16 >> 8) & 0x00ff;
				devbuf[4 * i + 3] = rt16 & 0x00ff;
				if (WRITE_FILE)
					fprintf(fp, "%d %d\n", (__s16) lt16, (__s16) rt16);
			}
		} else {
			generate_wave(lwave, VL, BUF_SIZE / 2);
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt16 = (__u16) (32767.0 * lwave->buf[i]);
				devbuf[2 * i] = (lt16 >> 8) & 0x00ff;
				devbuf[2 * i + 1] = lt16 & 0x00ff;
				if (WRITE_FILE)
					fprintf(fp, "%d\n", lt16);
			}
		}
		break;

	case AFMT_U16_LE:
		if (STEREO) {
			generate_wave(lwave, VL, BUF_SIZE / 4);
			generate_wave(rwave, VR, BUF_SIZE / 4);

			for (i = 0; i < BUF_SIZE / 4; i++) {
				lt16 = (__u16) (32767.0 * lwave->buf[i]) ^ 0x8000;
				rt16 = (__u16) (32767.0 * rwave->buf[i]) ^ 0x8000;
				devbuf[4 * i] = lt16 & 0x00ff;
				devbuf[4 * i + 1] = (lt16 >> 8) & 0x00ff;
				devbuf[4 * i + 2] = rt16 & 0x00ff;
				devbuf[4 * i + 3] = (rt16 >> 8) & 0x00ff;
				if (WRITE_FILE)
					fprintf(fp, "%d %d\n", (__s16) (lt16 ^ 0x8000), (__s16) (rt16 ^ 0x8000));
			}
		} else {
			generate_wave(lwave, VL, BUF_SIZE / 2);
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt16 = (__u16) (32767.0 * lwave->buf[i]) ^ 0x8000;
				devbuf[2 * i] = lt16 & 0x00ff;
				devbuf[2 * i + 1] = (lt16 >> 8) & 0x00ff;
				if (WRITE_FILE)
					fprintf(fp, "%d\n", (__s16) (lt16 ^ 0x8000));
			}
		}
		break;
	case AFMT_U16_BE:
		if (STEREO) {
			generate_wave(lwave, VL, BUF_SIZE / 4);
			generate_wave(rwave, VR, BUF_SIZE / 4);
			for (i = 0; i < BUF_SIZE / 4; i++) {
				lt16 = (__u16) (32767.0 * lwave->buf[i]) ^ 0x8000;
				rt16 = (__u16) (32767.0 * rwave->buf[i]) ^ 0x8000;
				devbuf[4 * i] = (lt16 >> 8) & 0x00ff;
				devbuf[4 * i + 1] = lt16 & 0x00ff;
				devbuf[4 * i + 2] = (rt16 >> 8) & 0x00ff;
				devbuf[4 * i + 3] = rt16 & 0x00ff;
				if (WRITE_FILE)
					fprintf(fp, "%d %d\n", (__s16) (lt16 ^ 0x8000), (__s16) (rt16 ^ 0x8000));
			}
		} else {
			generate_wave(lwave, VL, BUF_SIZE / 2);
			for (i = 0; i < BUF_SIZE / 2; i++) {
				lt16 = (__u16) (32767.0 * lwave->buf[i]) ^ 0x8000;
				devbuf[2 * i] = (lt16 >> 8) & 0x00ff;
				devbuf[2 * i + 1] = lt16 & 0x00ff;
				if (WRITE_FILE)
					fprintf(fp, "%d\n", (__s16) (lt16 ^ 0x8000));
			}
		}
		break;

	default:
		break;
	}

	if (WRITE_FILE)
		fclose(fp);

	return;
}

void print_info(int audio_fd, mode_t mode)
{
	audio_buf_info ospace_info, ispace_info;
	count_info optr_info, iptr_info;
	int ret, val;

	if ((mode & O_WRONLY) || (mode & O_RDWR)) {

		if ((ret = ioctl(audio_fd, SNDCTL_DSP_GETOPTR, &optr_info)) == -1) {
			perror("SNDCTL_DSP_GETOPTR");
		}

		printf("(SNDCTL_DSP_GETOPTR)\nbytes: %d blocks: %d ptr: %d ret: %d\n\n", optr_info.bytes, optr_info.blocks, optr_info.ptr, ret);

		if ((ret = ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &ospace_info)) == -1) {
			perror("SNDCTL_DSP_GETOSPACE");
		}

		printf
		    ("(SNDCTL_DSP_GETOSPACE)\nfragments: %d fragstotal: %d fragsize: %d bytes: %d ret: %d\n\n",
		     ospace_info.fragments, ospace_info.fragstotal, ospace_info.fragsize, ospace_info.bytes, ret);

		if ((ret = ioctl(audio_fd, SNDCTL_DSP_GETODELAY, &val)) == -1) {
			perror("SNDCTL_DSP_GETODELAY");
		}

		printf("(SNDCTL_DSP_GETODELAY)\npending: %d ret: %d\n\n", val, ret);
	}

	if (!(mode & O_WRONLY) || (mode & O_RDWR)) {
		if ((ret = ioctl(audio_fd, SNDCTL_DSP_GETIPTR, &iptr_info)) == -1) {
			perror("SNDCTL_DSP_GETIPTR");
		}

		printf("(SNDCTL_DSP_GETIPTR)\nbytes: %d blocks: %d ptr: %d ret: %d\n\n", iptr_info.bytes, iptr_info.blocks, iptr_info.ptr, ret);

		if ((ret = ioctl(audio_fd, SNDCTL_DSP_GETISPACE, &ispace_info)) == -1) {
			perror("SNDCTL_DSP_GETISPACE");
		}

		printf
		    ("(SNDCTL_DSP_GETISPACE)\nfragments: %d fragstotal: %d fragsize: %d bytes: %d ret: %d\n\n",
		     ispace_info.fragments, ispace_info.fragstotal, ispace_info.fragsize, ispace_info.bytes, ret);

	}

	return;
}
