//                              -*- Mode: C++ -*-
// avicat.cc -- combine .AVI files into 1 long movie
//
// $Id: avicat.cpp,v 1.10 2001/11/15 15:38:14 kabi Exp $
//
// Copyright (C) 2001  Tom Pavel <pavel@alum.mit.edu>
//
// Creator         : Tom Pavel  <tom>     Fri Feb  2 00:45:21 2001
// Modifier        : Tom Pavel  <tom>     Sun May 13 00:35:47 2001
// Update Count    : 31
// Status          : first working version
//
// added multiple audio+video streams,
// added switch to segment the avi for 800MB CDRs: Rainer Lay <rainer.lay@gmx.de>
//               currently deactivated due to synch problems
// removed error in sync (GetFrameTime)
//
//
// open issues: every audio stream syncs to 1st video stream
//
// TODO
// kabi: - properly handle VBR stream (headers should be read via StreamInfo)

#include "avifile.h"
#include "aviplay.h"
#include "except.h"
#include "version.h"

#include <stdio.h>
#include <stdlib.h> // exit
#include <unistd.h> // for getopt()

// hold on to audio samples until we have a chunk at least this big:
static const int MinAudChunk = 1;
static char* buf;
static uint_t bufsize = 512000;
static int debug = 0;

static IAviWriteStream* CreateVidOutStream(const IAviReadStream* inStr,
					   IAviWriteFile* outFile)
{
    uint_t sz = inStr->GetVideoFormat();
    BITMAPINFOHEADER* bh = (BITMAPINFOHEADER*) new char[sz];
    inStr->GetVideoFormatInfo(bh, sz);
    int ftm = (int) (1000000. * inStr->GetFrameTime());
    IAviWriteStream* outStr = outFile->AddStream(AviStream::Video, bh, sz,
						 bh->biCompression, ftm);
    delete[] (char*)bh;
    return outStr;
}

static IAviWriteStream* CreateAudOutStream(const IAviReadStream* inStr,
					   IAviWriteFile* outFile)
{
    uint_t sz = inStr->GetAudioFormat();
    WAVEFORMATEX* wf = (WAVEFORMATEX*) new char[sz];
    inStr->GetAudioFormat(wf, sz);
    IAviWriteStream* outStr = outFile->AddStream(AviStream::Audio, wf, sz,
						 wf->wFormatTag,
						 wf->nAvgBytesPerSec,
						 wf->nBlockAlign);
    delete[] (char*)wf;
    return outStr;
}

static void ConcatFile(const char* fname, IAviWriteFile* outFile,
		       avm::vector<IAviWriteStream*> *outStreams)
{
    // Open input file by name, and create Video/Audio streams.
    IAviReadFile* aviFile = CreateIAviReadFile (fname);
    uint_t videoInStreams = aviFile->VideoStreamCount();
    uint_t audioInStreams = aviFile->AudioStreamCount();
    avm::vector<IAviReadStream*> inStreams;

    for (uint_t num = 0; num < videoInStreams; num++)
    {
	IAviReadStream *stream = aviFile->GetStream(num, AviStream::Video);
	if (stream == 0)
	{
	    fprintf(stderr, "Error opening video stream #$d\n");
	    return;
	}
	inStreams[num]=stream;
	// Create the output streams (if they don't exist yet), using the formats
	// of the current (first) file.
	if ((*outStreams)[num] == 0)
	{
	    if (debug) fprintf(stderr, "create video out#%d\n", num);
	    (*outStreams)[num] = CreateVidOutStream(inStreams[num], outFile);
	} else
	    // check that formats match...
	    ;
    }

    for (uint_t num = videoInStreams; num < (audioInStreams + videoInStreams); num++)
    {
	inStreams[num] = aviFile->GetStream(num-videoInStreams, AviStream::Audio);
	if (inStreams[num] == 0)
	{
	    fprintf(stderr, "Error opening audio stream #%d\n", num);
	    return;
	}
	// Create the output streams (if they don't exist yet), using the formats
	// of the current (first) file.
	if ((*outStreams)[num] == 0)
	{
	    if (debug) fprintf(stderr, "create audio out#%d\n", num);
	    (*outStreams)[num] = CreateAudOutStream(inStreams[num], outFile);
	} else
	    // check that formats match...
	    ;
    }

    WAVEFORMATEX wf;
    avm::vector<double> bpf;
    avm::vector<int64_t> written_audio;
    avm::vector<int64_t> written_frames;

    for (uint_t num = 0; num < videoInStreams; num++)
	written_frames[num] = 0;

    for (uint_t num = 0; num < audioInStreams; num++)
    {
	inStreams[num + videoInStreams]->GetAudioFormat(&wf, sizeof(wf));
	// get bpf from first video stream
	bpf[num] = inStreams[0]->GetFrameTime() * wf.nAvgBytesPerSec;
	written_audio[num] = 0;
    }

    bool stop_process = false;

    while (!stop_process)
    {
	int flags = 0;
	stop_process = true;

	uint_t samp_read, bytes_read;

	for (uint_t num = 0; num < videoInStreams; num++)
	{
	    if (! inStreams[num]->Eof())
	    {
		inStreams[num]->ReadDirect(buf, bufsize, 1,
					   samp_read, bytes_read, &flags);
		// 	    if (samp_read<1){
		// 	      cerr << "Error reading 1 frame from video stream#" << num << endl;
		// 	    }
		if (debug) fprintf(stderr, "READ %dB from VidStr#%d\n", bytes_read, num);
		(*outStreams)[num]->AddChunk(buf, bytes_read, flags);
		written_frames[num]++;
	    }
	    stop_process = inStreams[num]->Eof();
	}

	for (uint_t num = 0;  num < audioInStreams; num++)
	{
	    // get streampos from video stream #0
	    int64_t excess = int64_t(bpf[num]*(inStreams[0]->GetPos() ) - written_audio[num]);
	    uint_t to_read = ((excess < MinAudChunk) && (!stop_process)) ? 0 : excess;

	    if (to_read > bufsize)
		to_read = bufsize;

	    if (inStreams[num + videoInStreams]->Eof())
		excess = 0;

	    if (! inStreams[num + videoInStreams]->Eof() && (to_read > 0))
	    {
		int result = inStreams[num+videoInStreams]->ReadDirect(buf, bufsize, to_read,
								       samp_read, bytes_read,
								       &flags);
		if (debug) fprintf(stderr, "READ %dB from AudStr#%d, to_read %d, "
				   "samp_read %d, bytes_read %d, Excess %Ld, bpf %f, "
				   "result %d, vidPos %d, written audio %Ld\n",
				   bytes_read, num, to_read, samp_read, bytes_read,
				   excess, bpf[num], result, inStreams[0]->GetPos(),
				   written_audio[num]);

		(*outStreams)[num + videoInStreams]->AddChunk(buf, bytes_read, flags);

		written_audio[num] += bytes_read;
		excess -= bytes_read;
	    }
	    //stop_process = inStreams[num+videoInStreams]->Eof();
	}
    }
}

static void Usage(const char* progname)
{
    printf("usage: %s [-h] [-7] [-s bytes] [-d] [-o outfile] file1.avi [ file2.avi ...]\n"
	   "  -h\t\tshow this help message\n"
           "  -7\t\t737280000 bytes long segments\n"
	   "  -s bytes\tcreate #bytes long segments\n"
	   "  -d\t\tshow lots of debug messages\n"
	   "  -o\t\toutput filename\n"
	   "WARNING: do not use with VBR streams at this moment\n",
	   progname);
    exit(1);
}

int main(int argc, char* argv[])
try
{
    const char* outFn = "out.avi";
    const char* progName=argv[0];
    uint_t length = 0x7F000000;

    int ch;
    while ((ch = getopt(argc, argv, "dh7s:o:")) != EOF)
    {
	switch ((char)ch)
	{
	case 'd':
	    ++debug;
	    break;
	case 'o':
	    outFn = optarg;
	    break;
	case '7':
	    length = 737280000;
	    break;
	case 's':
	    length = atoi(optarg);
	    break;
	case 'h':
	case '?':
	default:
	    Usage(progName);
	}
    }
    argc -= optind;
    argv += optind;
    if (argc < 1)
	Usage(progName);

    // Standard AVIlib sanity check:
    if (GetAvifileVersion() != AVIFILE_VERSION)
    {
	fprintf(stderr, "This binary was compiled for Avifile ver. %f, "
		"but the library is ver. %f. Aborting.\n",
		AVIFILE_VERSION, GetAvifileVersion());
	return 0;
    }

    buf = new char[bufsize];
    // Do the real work
    IAviWriteFile* outFile = CreateSegmentedFile(outFn, length);

    // Init to NULL.  Will get set upon doing the first inFile.
    avm::vector<IAviWriteStream*> outStreams;

    while (argc > 0)
    {
	ConcatFile(*argv, outFile, &outStreams);
	--argc;
	++argv;
    }

    // Close the outFile and write out the header, etc.
    delete outFile;
    delete[] buf;
}
catch(FatalError& error)
{
    error.Print();
}
