#include "loader.h"  // loader has to come first - some compilers have problems here
#include "ldt_keeper.h"

#include "default.h"
#include "plugin.h"
#include "registry.h"
#include "except.h"

#include "ACM_AudioDecoder.h"
#include "DS_AudioDecoder.h"
#include "VideoDecoder.h"
#include "DS_VideoDecoder.h"
#include "VideoEncoder.h"
#include "guids.h"
#include "configfile.h"
#include "fillplugins.h"

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>

extern char* def_path;

// sregistry entries
static const char* crapname = "SOFTWARE\\Microcrap\\Scrunch";
static const char* softname = "SOFTWARE\\Microsoft\\Scrunch";
static const char* videoname = "SOFTWARE\\Microsoft\\Scrunch\\Video";
static const char* loadername = "Software\\LinuxLoader\\";
static const char* divxname = "Software\\DivXNetworks\\DivX4Windows";
static const char* indeo4name = "Software\\Intel\\Indeo\\4.1";
static const char* indeo5name = "Software\\Intel\\Indeo\\5.0";
static const char* vp31name = "SOFTWARE\\ON2\\VFW Encoder/Decoder\\VP31";
static avm::vector<CodecInfo> libwin32_codecs;
static avm::string last_error;

extern "C" const avm::vector<CodecInfo>& RegisterPlugin()
{
    if (!libwin32_codecs.size())
    {
	char* wp = getenv("WIN32_PATH");
	if (wp)
	    def_path = wp;
	libwin32_FillPlugins(libwin32_codecs);
    }

    return libwin32_codecs;
}

extern "C" int GetPluginVersion()
{
    return PLUGIN_API_VERSION;
}

extern "C" IVideoEncoder* CreateVideoEncoder(const CodecInfo& info, fourcc_t compressor, const BITMAPINFOHEADER& bh)
{
    try
    {
	if (!libwin32_codecs.size())
	    RegisterPlugin();

	return new VideoEncoder(info, compressor, bh);
    }
    catch (FatalError& e)
    {
	last_error = e.GetDesc();
	e.Print();
	return 0;
    }
}

extern "C" IVideoDecoder* CreateVideoDecoder(const CodecInfo& info, const BITMAPINFOHEADER& bh, int flip)
{
    try
    {
	if (!libwin32_codecs.size())
	    RegisterPlugin();

	if (info.kind == CodecInfo::DShow_Dec)
	{
	    int maxauto = 0;
	    GetAttrInt(info, "MaxAuto", maxauto);
	    return new DS_VideoDecoder(info, bh, flip, maxauto);
	}
	return new VideoDecoder(info, bh, flip);
    }
    catch (FatalError& e)
    {
	last_error = e.GetDesc();
	e.Print();
	return 0;
    }
}

extern "C" IAudioDecoder* CreateAudioDecoder(const CodecInfo& info, const WAVEFORMATEX* fmt)
{
    try
    {
	if (!libwin32_codecs.size())
	    RegisterPlugin();

	if (info.kind != CodecInfo::Win32)
	    return new DS_AudioDecoder(info, fmt);

	return new ACM_AudioDecoder(info, fmt);
    }
    catch (FatalError& e)
    {
	last_error = e.GetDesc();
	e.Print();
	return 0;
    }
}

extern "C" avm::string GetError()
{
    return last_error;
}
/****

Here we've got attribute setting functions

****/
#define HKEY_CLASSES_ROOT       (0x80000000)
#define HKEY_CURRENT_USER       (0x80000001)
#define HKEY_LOCAL_MACHINE      (0x80000002)
#define HKEY_USERS              (0x80000003)
#define HKEY_PERFORMANCE_DATA   (0x80000004)
#define HKEY_CURRENT_CONFIG     (0x80000005)
#define HKEY_DYN_DATA           (0x80000006)
#define REG_DWORD		4	/* DWORD in little endian format */

static char* GetKeyName(char* s, const char* n, fourcc_t fcc)
{
    int i = strlen(n);

    strcpy(s, n);
    s[i++] = tolower(fcc & 0xff);
    fcc >>=8;
    s[i++] = tolower(fcc & 0xff);
    fcc >>=8;
    s[i++] = tolower(fcc & 0xff);
    fcc >>=8;
    s[i++] = tolower(fcc & 0xff);
    s[i] = 0;

    return s;
}

static int SetRegValue(const char* keyname, const char* attribute, int value, int fccHandler)
{
    int result, status, newkey;
    char full_name[100];
    const char *k = keyname;
    if (fccHandler)
        k = GetKeyName(full_name, keyname, fccHandler);

    result = RegCreateKeyExA(HKEY_CURRENT_USER, k,
			     0, 0, 0, 0, 0, &newkey, &status);
    if (result != 0)
    {
	printf("SetRegValue: registry failure\n");
	return -1;
    }

    result = RegSetValueExA(newkey, attribute, 0, REG_DWORD, &value, 4);
    if (result != 0)
    {
	printf("SetRegValue: error writing value\n");
    }
    // special registry for DivX codec
    if (result == 0 && strcmp("Current Post Process Mode", attribute) == 0)
    {
        value = -1;
	result=RegSetValueExA(newkey, "Force Post Process Mode", 0, REG_DWORD, &value, 4);
	if (result != 0)
	{
	    printf("SetRegValue: error writing value\n");
	    RegCloseKey(newkey);
	    return -1;
	}
    }
    RegCloseKey(newkey);

    //printf("*********SetregFullname %s %s   %d\n", k, attribute, value);
    return result;
}

static int GetRegValue(const char* keyname, const char* attribute, int fccHandler, int* value, int def)
{
    int result, status, newkey, count = 4;
    char full_name[100];
    const char* k = keyname;
    if (fccHandler)
    {
	k = GetKeyName(full_name, keyname, fccHandler);
	result = RegOpenKeyExA(HKEY_CURRENT_USER, k, 0, 0, &newkey);
    }
    else
	result = RegCreateKeyExA(HKEY_CURRENT_USER, keyname,
				 0, 0, 0, 0, 0, &newkey, &status);
    if (result != 0)
    {
	printf("GetRegValue: registry failure\n");
	return -1;
    }
    result = RegQueryValueExA(newkey, attribute, 0, &status, value, &count);
    if (count != 4)
	result = -1;
    RegCloseKey(newkey);
    if (result != 0)
    {
	printf("GetRegValue: no such value\n");
	*value = def;
        result = 0;
    }

    //printf("*********GetregFullname %s %s  %d\n", k, attribute, *value);
    return result;
}

extern "C" int GetAttrInt(const CodecInfo& info, const char* attribute, int& value)
{
    int result;

    bool b;
    switch (info.fourcc)
    {
    case fccDIVX:
        b = (strcmp(attribute, "Postprocessing") == 0);
	if (b || (strcmp(attribute, "Saturation") == 0)
	    || (strcmp(attribute, "Brightness") == 0)
	    || (strcmp(attribute, "Contrast") == 0))
	{
	    result = GetRegValue(divxname, attribute, 0, &value,
				 (b) ? 30 : 50);
	    if (b)
                value /= 10;
	    return result;
	}
	if (strcmp(attribute, "MaxAuto") == 0)
	{
	    value = Registry::ReadInt("win32DivX4", "MaxAuto", 6);
	    return 0;
	}
        break;
    case fccDIV3:
    case fccDIV4:
    case fccDIV5:
    case fccDIV6:
	if ((strcmp(attribute, "Crispness") == 0)
	    || (strcmp(attribute, "KeyFrames") == 0)
	    || (strcmp(attribute, "BitRate") == 0))
	    return GetRegValue(loadername, attribute, info.fourcc, &value, 0);
	/* Fall through */
    case fccWMV1:
    case fccWMV2:
	if (strcmp(attribute, "Quality") == 0)
	{
	    //return
	    GetRegValue((info.kind == CodecInfo::Win32)
			       ? crapname : softname,
			       "Current Post Process Mode",
			       0, &value, 0);
	    //printf("GETQUALITY %d\n", value);
            return value;

	}
	if ((strcmp(attribute, "Saturation") == 0)
	    || (strcmp(attribute, "Hue") == 0)
	    || (strcmp(attribute, "Contrast") == 0)
	    || (strcmp(attribute, "Brightness") == 0))
	    return GetRegValue(videoname, attribute, 0, &value, 50);
	if (strcmp(attribute, "MaxAuto") == 0)
	{
	    value = Registry::ReadInt("win32", "MaxAuto", 4);
	    return 0;
	}
	break;
    case fccIV41:
    case fccIV50:
	if (strcmp(attribute, "QuickCompress") == 0)
	    return GetRegValue((info.fourcc == fccIV50)
			       ? indeo5name : indeo4name,
			       attribute, 0, &value, 0);
	break;
    case fccVP30:
    case fccVP31:
	if (strcmp(attribute, "strPostProcessingLevel") == 0)
	    return GetRegValue(vp31name, attribute, 0, &value, 0);
        break;
    case fccMJPG:
	if (strcmp(attribute, "Mode") == 0)
	{
	    value = _GetPrivateProfileIntA("Compress", attribute, 1, "M3JPEG.INI");
	    return 0;
	}
	break;
    }
    if (strcmp(attribute, "MaxAuto") == 0)
    {
        value = 0;
	return 0;
    }
    printf("GetAttrInt(): unknown attribute '%s' for codec %s\n",
	   attribute, (const char*) info.dll);
    return -1;
}

extern "C" int SetAttrInt(const CodecInfo& info, const char* attribute, int value)
{
    int result;
    bool b;
    switch (info.fourcc)
    {
    case fccDIVX:
        b = (strcmp(attribute, "Postprocessing") == 0);
	if (b || (strcmp(attribute, "Saturation") == 0)
	    || (strcmp(attribute, "Brightness") == 0)
	    || (strcmp(attribute, "Contrast") == 0))
	    return SetRegValue(divxname, attribute, value * ((b) ? 10 : 1), 0);
	if (strcmp(attribute, "MaxAuto") == 0)
	{
	    Registry::WriteInt("win32DivX4", "MaxAuto", value);
	    return 0;
	}
        break;
    case fccDIV3:
    case fccDIV4:
    case fccDIV5:
    case fccDIV6:
	if ((strcmp(attribute, "Crispness") == 0)
	    || (strcmp(attribute, "KeyFrames") == 0)
	    || (strcmp(attribute, "BitRate") == 0))
	    return SetRegValue(loadername, attribute, value, info.fourcc);
        /* fall through */
    case fccWMV1:
    case fccWMV2:
	if (strcmp(attribute, "Quality") == 0)
	{
	    //printf("SETQUALITY %d\n", value);
	    return SetRegValue((info.kind == CodecInfo::Win32)
			       ? crapname : softname,
			       "Current Post Process Mode",
			       value, 0);
	}
	if ((strcmp(attribute, "Saturation") == 0)
	    || (strcmp(attribute, "Hue") == 0)
	    || (strcmp(attribute, "Brightness") == 0)
	    || (strcmp(attribute, "Contrast") == 0))
	    return SetRegValue(videoname, attribute, value, 0);
	if (strcmp(attribute, "MaxAuto") == 0)
	{
	    Registry::WriteInt("win32", "MaxAuto", value);
	    return 0;
	}
        break;
    case fccIV41:
    case fccIV50:
	if (strcmp(attribute, "QuickCompress") == 0)
	    return SetRegValue((info.fourcc == fccIV50)
			       ? indeo5name : indeo4name,
			       attribute, value, 0);
        break;
    case fccVP30:
    case fccVP31:
	if (strcmp(attribute, "strPostProcessingLevel") == 0)
	    return SetRegValue(vp31name, attribute, value, 0);
        break;
    case fccMJPG:
	if (strcmp(attribute, "Mode") == 0)
	{
	    char s[256];
	    sprintf(s, "%d", value);
	    _WritePrivateProfileStringA("Compress", attribute, s, "M3JPEG.INI");
	    return 0;
	}
	break;
    }

    printf("SetAttrInt(): unknown attribute '%s' for codec %s\n",
	   attribute, (const char*) info.dll);
    return -1;
}

extern "C" int SetAttrString(const CodecInfo& info, const char* attribute, const char* value)
{
    if (!attribute)
	return -1;

    switch (info.fourcc)
    {
    case fccMJPG:
	if (strcmp(attribute, "UserName") == 0
	    || strcmp(attribute, "LicenseKey") == 0)
	{
	    _WritePrivateProfileStringA("Register", attribute, value, "M3JPEG.INI");
	    return 0;
	}
	break;
    }

    return -1;
}

extern "C" int GetAttrString(const CodecInfo& info, const char* attribute, char* value, int size)
{
    if (!attribute)
	return -1;

    switch (info.fourcc)
    {
    case fccMJPG:
	if (strcmp(attribute, "UserName") == 0
	    || strcmp(attribute, "LicenseKey") == 0)
	    return _GetPrivateProfileStringA("Register", attribute, "", value, size, "M3JPEG.INI");
	break;
    }

    return -1;
}
