#include "fillplugins.h"
#include "audiodecoder.h"
#include "plugin.h"
#include "except.h"
#include "utils.h"
#include "fourcc.h"

#include "mad.h"

#include <stdio.h>
#include <ctype.h>

// code here inspired from gmerlin sources

#define _
static char const *error_str(enum mad_error error)
{
  static char str[17];

  switch (error) {
  case MAD_ERROR_BUFLEN:
  case MAD_ERROR_BUFPTR:
    /* these errors are handled specially and/or should not occur */
    break;

  case MAD_ERROR_NOMEM:		 return _("not enough memory");
  case MAD_ERROR_LOSTSYNC:	 return _("lost synchronization");
  case MAD_ERROR_BADLAYER:	 return _("reserved header layer value");
  case MAD_ERROR_BADBITRATE:	 return _("forbidden bitrate value");
  case MAD_ERROR_BADSAMPLERATE:	 return _("reserved sample frequency value");
  case MAD_ERROR_BADEMPHASIS:	 return _("reserved emphasis value");
  case MAD_ERROR_BADCRC:	 return _("CRC check failed");
  case MAD_ERROR_BADBITALLOC:	 return _("forbidden bit allocation value");
  case MAD_ERROR_BADSCALEFACTOR: return _("bad scalefactor index");
  case MAD_ERROR_BADFRAMELEN:	 return _("bad frame length");
  case MAD_ERROR_BADBIGVALUES:	 return _("bad big_values count");
  case MAD_ERROR_BADBLOCKTYPE:	 return _("reserved block_type");
  case MAD_ERROR_BADSCFSI:	 return _("bad scalefactor selection info");
  case MAD_ERROR_BADDATAPTR:	 return _("bad main_data_begin pointer");
  case MAD_ERROR_BADPART3LEN:	 return _("bad audio data length");
  case MAD_ERROR_BADHUFFTABLE:	 return _("bad Huffman table select");
  case MAD_ERROR_BADHUFFDATA:	 return _("Huffman data overrun");
  case MAD_ERROR_BADSTEREO:	 return _("incompatible block_type for JS");
  }

  sprintf(str, "error 0x%04x", error);
  return str;
}

class MAD_Decoder : public IAudioDecoder
{
    struct mad_stream stream;
    struct mad_frame frame;
    struct mad_synth synth;
    bool m_bInitialized;

public:
    MAD_Decoder(const CodecInfo&, const WAVEFORMATEX*);
    ~MAD_Decoder();
    virtual int Convert(const void*, uint_t, void*, uint_t, uint_t*, uint_t*);
};

MAD_Decoder::MAD_Decoder(const CodecInfo& info, const WAVEFORMATEX* wf)
    :IAudioDecoder(info, wf), m_bInitialized(false)
{
    mad_stream_init(&stream);
    mad_frame_init(&frame);
    mad_synth_init(&synth);
}

MAD_Decoder::~MAD_Decoder()
{
    mad_synth_finish(&synth);
    mad_frame_finish(&frame);
    mad_stream_finish(&stream);
}

#if 0
int MAD_Decoder::GetOutputFormat(WAVEFORMATEX* destfmt)
{
    if (!destfmt)
	return -1;
    *destfmt = in_fmt;

    destfmt->wBitsPerSample = 16;
    destfmt->wFormatTag = 0x2000;
    destfmt->nAvgBytesPerSec = 192000;  // after conversion
    destfmt->nBlockAlign = MAD_BLOCK_SIZE;
    destfmt->nSamplesPerSec = destfmt->nAvgBytesPerSec / destfmt->nChannels
	/ (destfmt->wBitsPerSample / 8);

    /*
     destfmt->nBlockAlign = destfmt->nChannels * destfmt->wBitsPerSample / 8;
     destfmt->nAvgBytesPerSec = destfmt->nSamplesPerSec * destfmt->nBlockAlign;
     destfmt->cbSize = 0;
     */
    char b[200];
    avm_wave_format(b, sizeof(b), &in_fmt);
    printf("src %s\n", b);
    avm_wave_format(b, sizeof(b), destfmt);
    printf("dst %s\n", b);

    return 0;
}
#endif

int MAD_Decoder::Convert(const void* in_data, uint_t in_size,
			 void* out_data, uint_t out_size,
			 uint_t* size_read, uint_t* size_written)
{
    mad_stream_buffer(&stream, (const unsigned char*) in_data, in_size);
    if (mad_frame_decode(&frame, &stream) == -1)
    {
        //printf("mad_frmae_decode: stream error: %s (harmless after seeking)\n",
	//       error_str(stream.error));
	mad_frame_mute(&frame);
	mad_stream_sync(&stream);
	mad_synth_init(&synth);
    }
    else
    {
	if (!m_bInitialized)
	{
	    printf("Mad MPEG header  Layer:%d  Bitrate:%ld  SampleRate:%d\n",
		   frame.header.layer, frame.header.bitrate, frame.header.samplerate);
            m_bInitialized = true;
	}
	mad_synth_frame(&synth, &frame);
	// Convert mad_fixed to int16_t
	int16_t* samples = (int16_t*) out_data;

#if 0
	printf("pos:%d  fsize:%d ch:%d spf:%d -> %d     PTR:%p\n",
	       out_size, frame_size, ai.num_channels, ai.samples_per_frame,
	       ai.samples_per_frame * ai.num_channels * 2, in_data);
	printf("Synth %d  ch:%d  l:%d\n", synth.pcm.samplerate, synth.pcm.channels,
	       synth.pcm.length);

	printf("Timer %d %f   mode: %d  \n",
	       frame.header.duration.seconds, 1/(double)frame.header.duration.fraction,
	       frame.header.mode);
#endif
	for(int i = 0; i < synth.pcm.channels; i++)
	{
	    for(int j = 0; j < synth.pcm.length; j++)
	    {
		int sample = synth.pcm.samples[i][j] >> (MAD_F_FRACBITS + 1 - 16);
		if (sample > 32767)
		    sample = 32767;
		else if (sample < -32768)
		    sample = -32768;

		samples[j * synth.pcm.channels + i] = (int16_t)(sample);
	    }
	}
    }
    if (size_read)
	*size_read = stream.next_frame - (const unsigned char*)in_data;
    if (size_written)
	*size_written = 2 * synth.pcm.channels * synth.pcm.length;

    return 0;
}


// PLUGIN loading part
static avm::vector<CodecInfo> audiocodecs;

extern "C" const avm::vector<CodecInfo>& RegisterPlugin()
{
    if (audiocodecs.size())
	return audiocodecs;
    libmad_FillPlugins(audiocodecs);
    return audiocodecs;
}

extern "C" int GetPluginVersion() { return PLUGIN_API_VERSION; }

#define __MODULE__ "MAD audio decoder"
extern "C" IAudioDecoder* CreateAudioDecoder(const CodecInfo& info, const WAVEFORMATEX* format)
{
    IAudioDecoder* decoder = 0;
    try
    {
	switch (info.fourcc)
	{
	case 0x50:
	case 0x55:
	    decoder = new MAD_Decoder(info, format);
	    break;
	default:
	    throw FATAL("Audio format ID %d unsupported", format->wFormatTag);
	}
	printf("Audio in %s format\n", avm_wave_format_name(info.fourcc));
    }
    catch (FatalError& e)
    {
	e.Print();
    }
    return decoder;
}
