#include "fillplugins.h"
#include "audiodecoder.h"
#include "plugin.h"
#include "except.h"
#include "utils.h"
#include "ac3-iec958.h"

#include <stdio.h>

class AC3_PassDecoder : public IAudioDecoder
{
public:
    // Note:  AudioQueue has this value hardcoded and it is used
    //        as block size when ac3 passthrough stream is played!
    static const uint_t AC3_BLOCK_SIZE = 6144;

    AC3_PassDecoder(const CodecInfo&, const WAVEFORMATEX*);
    int Convert(const void*, uint_t, void*, uint_t, uint_t*, uint_t*);
    int GetOutputFormat(WAVEFORMATEX* destfmt);
};

AC3_PassDecoder::AC3_PassDecoder(const CodecInfo& info, const WAVEFORMATEX* wf)
    :IAudioDecoder(info, wf)
{
}

int AC3_PassDecoder::GetOutputFormat(WAVEFORMATEX* destfmt)
{
    if (!destfmt)
	return -1;
    *destfmt = in_fmt;

    destfmt->wBitsPerSample = 16;
    destfmt->wFormatTag = 0x2000;
    destfmt->nAvgBytesPerSec = 192000;  // after conversion
    destfmt->nBlockAlign = AC3_BLOCK_SIZE;
    destfmt->nSamplesPerSec = destfmt->nAvgBytesPerSec / destfmt->nChannels
	/ (destfmt->wBitsPerSample / 8);

    /*
     destfmt->nBlockAlign = destfmt->nChannels * destfmt->wBitsPerSample / 8;
     destfmt->nAvgBytesPerSec = destfmt->nSamplesPerSec * destfmt->nBlockAlign;
     destfmt->cbSize = 0;
     */
    char b[200];
    avm_wave_format(b, sizeof(b), &in_fmt);
    printf("src %s\n", b);
    avm_wave_format(b, sizeof(b), destfmt);
    printf("dst %s\n", b);

    return 0;
}

int AC3_PassDecoder::Convert(const void* in_data, uint_t in_size,
			     void* out_data, uint_t out_size,
			     uint_t* size_read, uint_t* size_written)
{
    if (!in_data || !out_data)
	return -1;

    int skipped;
    struct ac3info ai;

    int data_type = 1;

    int i = ac3_iec958_parse_syncinfo((unsigned char*)in_data, in_size, &ai, &skipped);
    if (i < 0) {
	printf("AC3 stream not valid.\n");
	return -1;
    }
    if (ai.samplerate != 48000) {
	printf("Only 48000 Hz streams supported.\n");
	return -1;
    }

    // we are reading just one frame - easier to synchronize
    // speed is not ours biggest problem :) ...
    // so I don't care we call write more offten then play-ac3
    ac3_iec958_build_burst(ai.framesize, data_type, 1,
			   (unsigned char*)in_data + skipped,
			   (unsigned char*)out_data);

    if (size_read)
	*size_read = skipped + ai.framesize;
    if (size_written)
	*size_written = AC3_BLOCK_SIZE;

    return 0;
}


// PLUGIN loading part

static avm::vector<CodecInfo> audiocodecs;

extern "C" const avm::vector<CodecInfo>& RegisterPlugin()
{
    if (audiocodecs.size())
	return audiocodecs;
    libac3pass_FillPlugins(audiocodecs);
    return audiocodecs;
}

extern "C" int GetPluginVersion() { return PLUGIN_API_VERSION; }

#define __MODULE__ "AC3 Pass Through audio decoder"
extern "C" IAudioDecoder* CreateAudioDecoder(const CodecInfo& info, const WAVEFORMATEX* format)
{
    IAudioDecoder* decoder = 0;
    try
    {
	switch (info.fourcc)
	{
	case 0x2000: //AC3
	    decoder = new AC3_PassDecoder(info, format);
	    break;
	default:
	    throw FATAL("Audio format ID %d unsupported\n", format->wFormatTag);
	}
	printf("Audio in %s format\n", avm_wave_format_name(info.fourcc));
    }
    catch (FatalError& e)
    {
	e.Print();
    }
    return decoder;
}
