/********************************************************

	Main AVI player widget
	Copyright 2000 Eugene Kuznetsov (divx@euro.ru)

*********************************************************/

#include <qapp.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qslider.h>
#include <qtimer.h>
#include <qcursor.h>
#include <qpainter.h>
#include <qlayout.h>
#include <qhbox.h>
#include <qprogressbar.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qdragobject.h>

#include "playercontrol.h"
#include "configdialog_impl.h"


#include <creators.h>
#include <version.h>
#include <utils.h>
#include <configfile.h>
#include <renderer.h>
#include <StreamInfo.h>
#include <videodecoder.h>
#include <creators.h>

#ifdef USE_SDL
#include <SDL_keysym.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <cstdio>
#include <cctype>
#include <math.h> //rint

#include <ostream.h>

using namespace std;
using namespace Creators;

extern PlayerControl* m;

class QMyResizeEvent : public QEvent
{
public:
    int m_x, m_y;
    QMyResizeEvent(int x, int y)
	:QEvent(Q_MyResizeEvent), m_x(x), m_y(y) {}
};

void PlayerControl::send_event(QEvent ev)
{
    QEvent* e = new QEvent(ev);
    qApp->postEvent(m, e);
    qApp->wakeUpGuiThread();
}

void PlayerControl::send_resize_event(int w, int h)
{
    QEvent* e = new QMyResizeEvent(w, h);
    qApp->postEvent(m, e);
    qApp->wakeUpGuiThread();
}

void PlayerControl::send_key_event(int qtkey, int qtascii, int qtstate)
{
    m_bKeyProcessed = true;

    QEvent* e = new QKeyEvent(QEvent::KeyPress, qtkey, qtascii, qtstate);
    qApp->postEvent(m, e);
    qApp->wakeUpGuiThread();
}

bool PlayerControl::event(QEvent* e)
{
    m_bKeyProcessed = false;
    switch(e->type())
    {
    case Q_MenuEvent:
	menu_func();
	return true;
    case Q_PosEvent:
	updatePos();
	return true;
    case Q_MyCloseEvent:
	close();
	return true;
    case Q_MyRefreshEvent:
	Refresh();
	return true;
    case Q_MyResizeEvent:
	{
	    QMyResizeEvent* er = (QMyResizeEvent*) e;
	    Resize(er->m_x, er->m_y);
	}
	return true;
    case QEvent::KeyPress:
	keyPressEvent((QKeyEvent*)e);
        return true;
    case QEvent::Quit:
    case QEvent::Close: // Have to be checked here - for safe destruction
        endPlayer();
	break;
    default:
        break;
    }

    return QWidget::event(e);
}

void PlayerControl::key_func(int sym, int mod)
{
#ifdef USE_SDL
    if (player)
    {
	//cout << "Key " << sym << "  mod " << mod << "   " << m_bKeyProcessed << endl;
	if (m_bKeyProcessed)
	    return;
        // queue new key only if the previous one has been processed


	// SDL -> QT key translation
	struct keytr {
            int sdlkey;
	    int qtkey;
	    int qtascii;
	    int qtstate;
	} sdlqttable[] =
	{
	    { SDLK_RETURN, Qt::Key_Return },
	    { SDLK_LEFT, Qt::Key_Left },
	    { SDLK_PAGEDOWN, Qt::Key_PageDown },
	    { SDLK_RIGHT, Qt::Key_Right },
	    { SDLK_PAGEUP, Qt::Key_PageUp },
	    { SDLK_UP, Qt::Key_Up },
	    { SDLK_DOWN, Qt::Key_Down },
	    { SDLK_HOME, Qt::Key_Home },
	    { SDLK_END, Qt::Key_End },
	    { SDLK_KP_PLUS, Qt::Key_Plus, 0, Qt::Keypad },
	    { SDLK_KP_MINUS, Qt::Key_Minus, 0, Qt::Keypad },
	    { SDLK_1, Qt::Key_1, '1' },
	    { SDLK_2, Qt::Key_2, '2' },
	    { SDLK_3, Qt::Key_3, '3' },
	    { SDLK_a, Qt::Key_A, 'A' },
	    { SDLK_z, Qt::Key_A, 'Z' },
	    { SDLK_LEFTBRACKET, Qt::Key_BraceLeft, '[' },
	    { SDLK_RIGHTBRACKET, Qt::Key_BraceRight, ']' },
	    { SDLK_F1, Qt::Key_F1, 0 },
	    { SDLK_F2, Qt::Key_F2, 0 },
	    { SDLK_F3, Qt::Key_F3, 0 },
	    { SDLK_F4, Qt::Key_F4, 0 },
	    { SDLK_F5, Qt::Key_F5, 0 },
	    { SDLK_F6, Qt::Key_F6, 0 },
	    { SDLK_F7, Qt::Key_F7, 0 },
	    { SDLK_F8, Qt::Key_F8, 0 },
	    { SDLK_F9, Qt::Key_F9, 0 },
	    { SDLK_F10, Qt::Key_F10, 0 },
	    { 0 }
	};

	int i = 0;
	while (sdlqttable[i].sdlkey && sdlqttable[i].sdlkey != sym)
            i++;

	if (sdlqttable[i].sdlkey)
	{
#if 0
	    send_key_event(sdlqttable[i].qtkey,
			   sdlqttable[i].qtascii,
			   sdlqttable[i].qtstate);
#else
            // testing thread stability & safety
	    QKeyEvent* e = new QKeyEvent(QEvent::KeyPress,
					 sdlqttable[i].qtkey,
					 sdlqttable[i].qtascii,
					 sdlqttable[i].qtstate);
	    keyPressEvent(e);
	    delete e;
#endif
	}
    }
#endif
}

void PlayerControl::menu_func()
{
    if (!popup)
    {
	popup = new QPopupMenu();
#if 1
	//popup->insertTearOffHandle();
	QPopupMenu* p = new QPopupMenu();
	connect(p, SIGNAL(activated(int)), this, SLOT(zoom_mode(int)));
	if (p)
	{
	    p->insertItem("Zoom 2x    &3", this, SLOT(zoom_2x()));
	    p->insertItem("Zoom 1x    &2", this, SLOT(zoom_1x()));
	    p->insertItem("Zoom 0.5x  &1", this, SLOT(zoom_0_5x()));
	    p->insertItem("Zoom In    &A", this, SLOT(zoom_in()));
	    p->insertItem("Zoom Out   &Z", this, SLOT(zoom_out()));
            popup->insertItem("Zoom", p);
	}

	p = new QPopupMenu();
	connect(p, SIGNAL(activated(int)), this, SLOT(zoom_mode(int)));
	if (p)
	{
	    p->insertItem("Original", this, SLOT(ratio_orig()));
	    p->insertItem("16:9"    , this, SLOT(ratio_16_9()));
	    p->insertItem("4:3"     , this, SLOT(ratio_4_3()));
	    p->insertItem("Max"     , this, SLOT(ratio_max()));
            popup->insertItem("Ratio", p);
	}
	popup->insertSeparator();
	popup->insertItem("&Fullscreen", this, SLOT(fullscreen()), CTRL+Key_F );
	popup->insertItem("&Maximize", this, SLOT(maximize()));

	const avm::vector<VideoRenderer*>& vvr = player->GetVideoRenderers();
	if (vvr.size() > 0)
	{
	    modes = new QPopupMenu();

	    connect(modes, SIGNAL(activated(int)),
		    this, SLOT(maximize_mode(int)));

	    unsigned i = 0;
	    while (i < vvr.size())
	    {
		const avm::vector<VideoMode>& vvm = vvr[i]->GetVideoModes();
		unsigned j = 0;
		while (j < vvm.size())
		{
		    modes->insertItem(vvm[j].name.c_str(), (i << 8) + j);
		    j++;
		}
		i++;
	    }
	    popup->insertItem("Maximize Modes", modes);
	}
#endif
	popup->insertSeparator();
	popup->insertItem("Properties", this, SLOT(about_movie()));

	if (player->GetCodecInfo().decoder_info.size())
	{
	    popup->insertSeparator();
	    popup->insertItem("Decoder config", this, SLOT(decoder_config()));
	}
    }

    if (popup->isVisible())
        popup->hide();
    else
	popup->exec(QCursor::pos());
}

void PlayerControl::zoom(double scale)
{
    if (player)
    {
        if (scale == 2.0)
	    resize_count = 2;
	else if (scale == 1.0)
	    resize_count = 1;
        else
	    resize_count = 0;

	int w = (int) (scale * player->GetWidth() + 0.5);
	int h = (int) (scale * player->GetHeight() + 0.5);

	player->Resize(w,h);
    }
}

void PlayerControl::ratio(int r)
{
    cout << "ratio" << r << endl;
}

void PlayerControl::zoom_in()
{
    cout << "zoomin" << endl;
}

void PlayerControl::zoom_out()
{
    cout << "zoomout" << endl;
}

void PlayerControl::zoomWH(int new_w, int new_h, int ori_w, int ori_h)
{
    cout << "zoomWH " << new_w << "   " << new_h << endl;

    if (ori_w > ori_h)
	new_h = (int) (new_w * ori_h / (double) ori_w + 0.5);
    else
	new_w = (int) (new_h * ori_w / (double) ori_h + 0.5);
    player->Resize(new_w, new_h);
    fullscreen();
}

void PlayerControl::send_fullscreen()
{
    send_key_event(Qt::Key_F, 'F');
}

void PlayerControl::fullscreen()
{
    if (player)
	player->ToggleFullscreen(false);
}

void PlayerControl::send_maximize()
{
    send_key_event(Qt::Key_M, 'M');
}

void PlayerControl::maximize()
{
    if (player)
	player->ToggleFullscreen(true);
}

void PlayerControl::maximize_mode(int renderer_mode)
{
    if(renderer_mode<0)
	return;

    //cout << "maximize " << i << endl;
    const avm::vector<VideoRenderer*>& vvr = player->GetVideoRenderers();
    unsigned renderer = (renderer_mode >> 8);
    if (renderer < vvr.size())
    {
        unsigned mode = renderer_mode & 0xff;
	const avm::vector<VideoMode>& vvm = vvr[renderer]->GetVideoModes();
	if (mode < vvm.size())
	{
	    int ori_w, ori_h;
	    int new_w = vvm[mode].width;
	    int new_h = vvm[mode].height;

	    if (vvr[renderer]->GetSize(ori_w, ori_h) < 0)
	    {
		ori_w = player->GetWidth();
		ori_h = player->GetHeight();
	    }
	    //else cout << " w " << ori_w << "   h " << ori_h << endl;
	    zoomWH(new_w, new_h, ori_w, ori_h);
	}
    }
}

void PlayerControl::zoom_mode(int zoom_mode)
{

}

void PlayerControl::showconf_func()
{
    if (player)
    {
	ConfigDialog_impl dilg;
	dilg.exec();
	readConfig();
    }
}

// this method should be used for all volume changes
// for common interpretation
void PlayerControl::volume_func(int vpos)
{
    if (player)
	player->Set(IAviPlayer::AUDIO_VOLUME,
		    int(1000.0 * vpos * vpos /float(MAX_VBAR * MAX_VBAR)),
		    0);
}

void PlayerControl::mute_func()
{
    int p = vbar->value();
    int n = vollast;
    if (p != 0) {
	vollast = p;
        n = 0;
    }
    vbar->setValue(n);
}


void PlayerControl::pause_func()
{
    if (player && player->IsPlaying())
	player->Pause(!player->IsPaused());
}

void PlayerControl::paintEvent(QPaintEvent *)
{
    if (player && !player->IsStopped())
	updatePos();
}

void PlayerControl::rcreseek(double spos)
{
    if (player)
    {
	scroller->setValue((int)spos);
	player->ReseekExact(spos);
	updatePos();
    }
}

// static  - thus using m - pointer to PlayerControl
void PlayerControl::stopfunc(int)
{
    m->scroller->setValue(0);
}

void PlayerControl::stop_func()
{
    if (player)
    {
	player->Stop();
	scroller->setValue(0);
    }
}

void PlayerControl::changed_func(int cpos)
{
    if (player)
    {
	//cout << is_tracking << "diff " << last_slider - cpos << "  " << cpos << "  " << last_slider << endl;

        bool wrapAround = cpos < scroller->minValue() + 2
	    && last_slider > scroller->maxValue() - 2;

	// when paused or when mouse wheel movement...
	int minchange = (player->IsPaused()) ? 0 : 2;
	if (!m_bSettingScroller
            && (!is_tracking || player->IsPaused())
	    && !wrapAround && abs(last_slider - cpos) > minchange)
	{
	    player->ReseekExact(cpos);
	}
	last_slider = cpos;
    }
}

void PlayerControl::nextpage_func()
{
    if (player && player->PageUp() == 0)
	updatePos();
}

void PlayerControl::prevpage_func()
{
    if (player && player->PageDown() == 0)
	updatePos();
}

void PlayerControl::play_func()
{
    if (player)
    {
	if (!player->IsPlaying() && !player->IsPaused())
	    player->Start();
	else
	    player->Play();
    }
}

void PlayerControl::open_func()
{
    QString open_qs;
    avm::string path = Registry::ReadString("aviplay", "file", ".");
    //QFileDialog *fld = new QFileDialog();
    //open_qs = fld->getOpenFileName(tr(path.c_str()), tr("AVI files (*.avi *.AVI);;ASF files (*.asf *.ASF);;All files(*)"));
//    mb->setTextFormat(Qt::RichText);
    open_qs = QFileDialog::getOpenFileName(tr(path.c_str()),
					   tr("AVI files (*.avi *.AVI);;ASF files (*.asf *.ASF);;All files(*)"));

    //delete fld;
    if (open_qs.isNull())
	return;
    try
    {
	AviPlayerInitParams apip;
	char* filep = new char[open_qs.length()+1];
	strcpy(filep,open_qs.local8Bit());
	apip.filename = filep;
	initPlayer(apip);
	delete filep;
    }
    catch (FatalError& error)
    {
        QMessageBox::critical( 0,
                               tr( "Aviplay - Fatal error" ),
                               tr( "Error in %1 module:\n%s" )
                               .arg( QString::fromLatin1( error.GetModule() ) )
                               .arg( QString::fromLatin1( error.GetDesc() ) ),
                               QMessageBox::Ok, QMessageBox::NoButton );
	error.PrintAll();
	update();
    }
}

void PlayerControl::opensub_func()
{
    if (!player)
        return;

    QString open_qs;
    avm::string path = Registry::ReadString("aviplay", "subtitlefile", ".");
    open_qs = QFileDialog::getOpenFileName(path.c_str(), "SUB files (*.sub *.SUB);;SRT files (*.srt *.SRT);;All files(*)");

    if (open_qs.isNull())
	return;
    try
    {
	player->InitSubtitles(open_qs.local8Bit());
	Registry::WriteString("aviplay", "subtitlefile", (const char *)open_qs.local8Bit());
    }
    catch (FatalError& error)
    {
        QMessageBox::critical( this,
                               tr( "Aviplay - Fatal error" ),
                               tr( "Error in %1 module:\n%s" )
                               .arg( QString::fromLatin1( error.GetModule() ) )
                               .arg( QString::fromLatin1( error.GetDesc() ) ),
                               QMessageBox::Ok, QMessageBox::NoButton );
	error.PrintAll();
	update();
    }
}

void PlayerControl::openaud_func()
{
    if (!player)
        return;

    QString open_qs;
    char* astr = 0;
    player->Get(player->AUDIO_STREAM, &astr, 0);
    open_qs = QFileDialog::getOpenFileName(astr, "All files(*)");
    if (astr)
        free(astr);

    if (open_qs.isNull())
	return;
    try
    {
	char *filep = new char[open_qs.length()+1];
	strcpy(filep,open_qs.local8Bit());
	player->Set(player->AUDIO_STREAM, filep, 0);
        delete filep;
    }
    catch (FatalError& error)
    {
	QMessageBox::information(this,
				 QString("Error in ") +
				 QString(error.GetModule()) +
				 QString(" module"),
				 error.GetDesc(), QMessageBox::Ok);
	error.PrintAll();
	repaint();
    }
}

#define __MODULE__ "AVI player widget"

void PlayerControl::keyPressEvent(QKeyEvent * kevent)
{
    // well-known xmms key events: 'x' stop, 'c' pause, 'v' play.
    // for SDL events are also mirrored in EventFilter ( see renderer.cpp )
    //cout << "keyPresss: "  << kevent->key() << "  acsii: "  << kevent->ascii() << endl;

    switch(toupper(kevent->ascii()))
    {
    case 'Q':
	send_event(Q_MyCloseEvent);
	return;
    case 'X':
	stop_func();
	return;
    case 'C':
    case 'P':
    case ' ':
        // try to support most natural keystrokes
	pause_func();
	return;
    case '[':
	if (player)
	{
            double a = rint((player->GetAsync() - 0.1) * 10.0) / 10.0;
	    //player->SetAsync(int(a * 10 + 1) / 10.0);
	    player->SetAsync(a);
	    cout << "Set async " << a << endl;
	}
	return;
    case ']':
	if (player)
	{
            double a = rint((player->GetAsync() + 0.1) * 10.0) / 10.0;
	    //player->SetAsync(int(a * 10 - 1) / 10.0);
	    player->SetAsync(a);
            cout << "Set async " << a << endl;
	}
	return;
    case 'V':
	play_func();
	return;
    case 'F':
	fullscreen();
	return;
    case 'M':
	maximize();
	return;
    case '1':
	zoom(0.5);
	return;
    case '2':
	zoom(1.0);
	return;
    case '3':
	zoom(2.0);
	return;
    case 'A':
	zoom(2.0);
	return;
    case 'Z':
	zoom(2.0);
	return;
    }

    switch(kevent->key())
    {
    case Qt::Key_Return:
	player->NextFrame();
	return;
    case Qt::Key_Plus:
        vbar->addStep();
	return;
    case Qt::Key_Minus:
        vbar->subtractStep();
	return;
    case Qt::Key_Left:
	player->PageDown();
	return;
    case Qt::Key_Right:
	player->PageUp();
	return;
    case Qt::Key_Home:
	player->ReseekExact(0);
	return;
    case Qt::Key_End:
	player->ReseekExact(player->GetLengthTime() - 1.);
	return;
    case Qt::Key_Up:
    case Qt::Key_PageUp:
	{
            double sktime = (kevent->key() != Qt::Key_PageUp) ? 60. : 10.;
	    double o = player->GetPos();
	    double p = o + sktime;
	    double len = player->GetLengthTime();
	    if (p > len)
	    {
		// we are at the end - try 5 seconds before end
		p = len - 5.;
		if (p < 0)
		    // to short
		    p = 0.;
	    }
	    player->ReseekExact(p);
	    if (player->GetPos() <= (o + 1.0))
		// some movies have very long intervals between KeyFrames
		// so always skip at least to the next keyframe
		player->PageUp();
	}
	return;
    case Qt::Key_Down:
    case Qt::Key_PageDown:
	{
            double sktime = (kevent->key() != Qt::Key_PageDown) ? 60. : 10.;
	    double p = player->GetPos() - sktime;
	    if (p < 0)
		p = 0.;
	    player->ReseekExact(p);
	}
	return;
    }

    struct runtimecnf {
	int qtkey;
	const char* attr;
	int add;
    } rttable[] = {
	{ Qt::Key_F1, "Saturation", -1 },
	{ Qt::Key_F2, "Saturation", 1 },
	{ Qt::Key_F5, "Brightness", -1 },
	{ Qt::Key_F6, "Brightness", 1 },
	{ Qt::Key_F7, "Hue", -1 },
	{ Qt::Key_F8, "Hue", 1 },
	{ Qt::Key_F9, "Contrast", 1 },
	{ Qt::Key_F10, "Contrast", -1 },
	{ 0, 0, 0 }
    };

    int i = 0;
    IRtConfig* rt = 0;
    while (rttable[i].qtkey)
    {
	//cout << "KEY " << rttable[i].qtkey << "   "  << kevent->key() << endl;
	if (rttable[i].qtkey == kevent->key())
	{
	    if (rt == 0)
	    {
		rt = player->GetRuntimeConfig();
		if (rt == 0)
		    break;
	    }
            int val;
	    const CodecInfo& info = player->GetCodecInfo();
	    if (rt->GetValue(rttable[i].attr, val) == 0)
	    {
		val += rttable[i].add;
		Creators::SetCodecAttr(info, rttable[i].attr, val);
		if (rt)
		    rt->SetValue(rttable[i].attr, val);
		return;
	    }
	}
        i++;
    }
}

void PlayerControl::mouseDoubleClickEvent(QMouseEvent *)
{
    showconf_func();
}

void PlayerControl::about_func()
{
    if (!about_popup)
    {
	about_popup = new QPopupMenu();
	about_popup->insertItem(tr("Usage"), this, SLOT(help()));
	about_popup->insertItem(tr("About"), this, SLOT(about()));
	about_popup->insertItem(tr("Config"), this, SLOT(showconf_func()));

    }

    about_popup->popup(QCursor::pos());
}

void PlayerControl::Resize(int w, int h)
{
    if (player)
	player->Resize(w, h);
}

void PlayerControl::Refresh()
{
    if (player)
	player->Refresh();
}

void PlayerControl::remote_command()
{
    char command[32];
    int command_pos = 0;
    static long oldpos;
    if (! m)
	return;
    long newpos = (long) m->GetPos();
    fd_set fds; // stdin changed?
    struct timeval tv; // how long to w8
    int n; // indicating whether something happened on stdin

    if (m->isStopped())
	send_event(Q_MyCloseEvent);
    if (newpos != oldpos) {
	oldpos = newpos;
	printf ("Pos: %li s / %li s\n", (long) m->GetPos(),
		(long) m->GetPlayer()->GetLengthTime());
    }
    tv.tv_sec = 0;
    tv.tv_usec = 50000;
    FD_ZERO(&fds);
    FD_SET(STDIN_FILENO, &fds);
    if ((n = select(32, &fds, NULL, NULL, &tv)) != 0)
    {
	double cpos;
	int i;
	int complete = 0;

 	while (command_pos < 32 &&
	    (i = read (STDIN_FILENO, command + command_pos, 1)) > 0 &&
	    command[command_pos] != '\r' &&
	    command[command_pos] != '\n')
	    command_pos++;

	if (command_pos == 31 || command[command_pos] == '\r' ||
	    command[command_pos] == '\n') {
	    command[command_pos] = 0;
	    complete = 1;
	}
	if (complete == 1) {
	    if (command[0] == 'p' || command[0] == 'P')
		pause_func();
	    else if (command[0] == 'f' || command[0] == 'F')
		fullscreen();
	    else if (command[0] == 'm' || command[0] == ';')
		maximize();
	    else if (command[0] == 's' || command[0] == 'S')
		stop_func();
	    else if (command[0] == 'q' || command[0] == 'Q')
		send_event(Q_MyCloseEvent);
	    else if (command[0] == 'r' || command[0] == 'R') {
		for (i = 1; i < 32 && command[i] >= '0' && command[i] <= '9'; ++i)
		    ;
		command[i] = 0;
		cpos = atof (command + 1);
		rcreseek(cpos);
	    } else
		printf ("Unknown command: %s\n", command);
	    command_pos = 0;
	    complete = 0;
	}
	return;
    }
}

void PlayerControl::about_movie()
{
    if (!player)
	return;

    char t[128];
    avm::string s = "File ";
    s += player->GetFileName();
    s += "\n\n";
    if (player->GetAudioLengthTime() > 0)
    {
	sprintf(t, ", %.2f seconds.\n", player->GetAudioLengthTime());
	s+="Audio stream: ";
	s+=player->GetAudioFormat();
	s+=t;
        StreamInfo* info = player->GetAudioStreamInfo();
	if (info)
	{
	    avm::string sinfo = info->GetString();
	    printf("%s", sinfo.c_str());
            delete info;
	}
    }
    if (player->GetVideoLengthTime() > 0)
    {
        StreamInfo* info = player->GetVideoStreamInfo();
        double bps = 0;;
	if (info)
	{
	    avm::string sinfo = info->GetString();
	    printf("%s", sinfo.c_str());
	    bps = info->GetBps();
            delete info;
	}
	sprintf(t, "Video stream: %dx%d, %.2f fps, %.2fkbit/s (%.2fKB/s)\n",
		player->GetWidth(), player->GetHeight(), player->GetFps(),
		bps * (8 / 1000.0), bps / 1024.0);
	s+=t;
	s+=player->GetVideoFormat();
	sprintf(t, ", %.2f seconds.\n", player->GetVideoLengthTime());
	s+=t;
    }
    // latin-1 ? utf-8 ?
    QMessageBox::information(0, tr( "About file" ), QString::fromLocal8Bit( s.c_str() ));
}

void PlayerControl::middle_button()
{
    int qtkey = 0, qtascii = 0;

    int w, h;
    player->Get(IAviPlayer::QUERY_VIDEO_WIDTH, &w,
		IAviPlayer::QUERY_VIDEO_HEIGHT, &h,
		0);
    int iw = player->GetWidth();
    int ih = player->GetHeight();
    //cout << "WxH " << w << " " << h << endl;
    switch((resize_count + 1) % 3)
    {
    case 0:
        // zoom 50%
	qtkey = Qt::Key_1;
	qtascii = '1';
	break;
    case 1:
        // zoom 100%
	qtkey = Qt::Key_2;
	qtascii = '2';
	break;
    case 2:
        // zoom 200%
	qtkey = Qt::Key_3;
	qtascii = '3';
	break;
    }

    send_key_event(qtkey, qtascii);

    m_bKeyProcessed = false;
    // intentionaly setting false here !
    // this is like sync point - this will
    // reset keyboard processed status
    // just in case something wrong happened before
}

void PlayerControl::readConfig()
{
    if (player)
    {
	// FIMXE
	bool ht;
	char* vcodecs = 0;
	char* acodecs = 0;

        player->Get(player->DISPLAY_FRAME_POS, &m_bDisplayFramePos,
		    player->AUTOREPEAT, &m_bAutorepeat,
		    player->USE_HTTP_PROXY, &ht,
		    player->VIDEO_CODECS, &vcodecs,
		    player->AUDIO_CODECS, &acodecs,
		    0);
	if (ht)
	{
	    char* proxy = 0;
	    player->Get(player->HTTP_PROXY, &proxy, 0);
	    if (proxy)
	    {
		if (strlen(proxy))
		    avm_setenv("HTTP_PROXY", proxy, 1);
                free(proxy);
	    }
	}
	else avm_unsetenv("HTTP_PROXY");

	if (vcodecs)
	{
	    SortVideoCodecs(vcodecs);
            free(vcodecs);
	}
	if (acodecs)
	{
	    SortAudioCodecs(acodecs);
            free(acodecs);
	}
    }
}

void PlayerControl::reseek_func()
{
    if (!player)
        return;

    double pos = scroller->value();
    _in_reseek=1;

    player->ReseekExact(pos);
    updatePos();
    is_tracking=0;
    _in_reseek=0;
}

void killer(int z)
{
    printf("quit_flag=1\n");
}

PixmapButton::PixmapButton(const char* ficon, QWidget* parent)
    :QPushButton(parent)
{
    QString file = QString::fromLatin1( SHARE_PATH "/" ) +
                   QString::fromLatin1( ficon ) +
                   QString::fromLatin1( ".ppm" );
    m_pm.load(file);
    setPixmap(m_pm);
    setFixedSize(m_pm.width() + 4, m_pm.height() + 4);
}

void PixmapButton::drawButton(QPainter* p)
{
    // ### write complete reimplementation
    QPushButton::drawButton(p);

    p->drawPixmap(2, 2, m_pm);
}

#define __MODULE__ "AVI player widget"
int PlayerControl::initPlayer(AviPlayerInitParams& uapip)
{
    apip = uapip;

    last_slider = 0;
    if (scroller)
	scroller->setValue(0);

    try
    {
	endPlayer();

	if (popup)
	{
	    delete popup;
	    popup = 0;
	}

	player = CreateAviPlayer2(this, x11Display(), apip.filename, apip.subname, apip.vcodec, apip.acodec);

	if (player)
	{
	    Registry::WriteString("aviplay", "file", apip.filename);
	    m_filename = apip.filename;

	    if (apip.subname)
	    {
		Registry::WriteString("aviplay", "subtitlefile", apip.subname);
		m_subname = apip.subname;
	    }
	    readConfig();
	    //if (quality == 11)
	    //    player->SetDecodingMode(IVideoDecoder::REALTIME_QUALITY_AUTO);
	    l_status->setText("Opening...");
	    _start_timer->start(1000); // large timeout for debugging
	    if (vbar)
		volume_func(vbar->value());
	}
    }
    catch (FatalError& error)
    {
	endPlayer();
    }
    catch (...)
    {
	cerr<<"Unknown exception"<<endl;
    }
    return 0;
}

void PlayerControl::starter_slot()
{
    if (!player || !player->IsOpened())
	return;

    if (!player->IsValid())
    {
	_start_timer->stop();
	endPlayer();
	/* may be show a message box here? */
	l_status->setText("Failed to open");
	return;
    }
    if (player->IsRedirector())
    {
	_start_timer->stop();
	avm::vector<avm::string> urls;
	player->GetURLs(urls);
	endPlayer();
	if (!urls.size())
	{
	    cout<<"Redirector with no entries?"<<endl;
	    return;
	}
	if (urls.size() > 1)
	{
	    cout<<"Multi-URL redirectors aren't yet properly supported, sorry"<<endl;
	    cout<<"Contents of the redirector: "<<endl;
	    for (unsigned i = 0; i < urls.size(); i++)
		cout<<urls[i]<<endl;
	    cout<<"Attempting to open default first entry"<<endl;
	}
	AviPlayerInitParams ip;
        ip.filename = urls[0].c_str();
	initPlayer(ip);
	return;
    }
    _start_timer->stop();

    if (apip.width && apip.height)
        player->Resize(apip.width, apip.height);

    setCaption(QFile::decodeName( m_filename.c_str() ));
    //	Scroll bar
    double len = player->GetLengthTime();
    scroller->setMinValue(0);
    scroller->setMaxValue((int)len);

    int step = int(len / 40);
    scroller->setLineStep((step < 1) ? 1 : (step > 100) ? 100 : step);
    scroller->repaint();

    _timer->start(1000);

    play_func();

    cout<<"Player started"<<endl;

    if (apip.seek > 0.0)
	player->ReseekExact(apip.seek);

    // wait here - so all graphics is initialized & opened
    if (apip.maximize)
	send_maximize();
    else if (apip.fullscreen)
	send_fullscreen();
}

void PlayerControl::endPlayer()
{
    _timer->stop();
    delete popup;
    popup = 0;
    delete about_popup;
    about_popup = 0;
    delete player;
    player = 0;
}

void PlayerControl::updatePosDisplay(int pos)
{

    if (!player)
        return;

    double percent;
    IAviPlayer::State state = player->GetState(&percent);

    if (state == IAviPlayer::Playing || state == IAviPlayer::Paused)
    {
	char timing[50];
	double len = player->GetLengthTime();
	double stream_time = player->GetPos();
	if (pos > stream_time + 2 || pos < stream_time - 2)
	    //user is seeking - show the time for seek-bar
	    stream_time = pos;

	char* spos = timing;

	//cout << "pos " << pos << "  " << stream_time << endl;
	if (stream_time >= 3600)
	    spos += sprintf(spos, "%d:", int(stream_time/3600));

	spos += sprintf(spos, "%02d:%02d", int(stream_time/60)%60,
			int(stream_time)%60);

	if (m_bDisplayFramePos)
            // more precise timeing with frame position
	    spos += sprintf(spos, ".%03d", int(stream_time*1000)%1000);

	if (len >= 3600)
	    spos += sprintf(spos, "/%d:", int(len/3600));
	else
	    spos += sprintf(spos, "/");

	spos += sprintf(spos, "%02d:%02d", int(len/60)%60, int(len)%60);

        if (m_bDisplayFramePos)
	    spos += sprintf(spos, " (%d)", player->GetFramePos());

	l_time->setText(timing);

	if (pbar)
	{
	    int prog;
            player->Get(player->QUERY_AVG_QUALITY, &prog, 0);
	    pbar->setProgress(prog);
	}
    }
}

void PlayerControl::updatePos()
{
    if (!player)
	return;

    QPainter painter(this);

    double dpos = player->GetPos();

    if (player->IsPlaying())
    {
        bool b;
	player->Get(player->QUERY_EOF, &b, 0);
        if (b)
	{
	    player->Get(player->AUTOREPEAT, &b, 0);
	    if (!b)
		send_event(Q_MyCloseEvent);
	}
    }

    if (!is_tracking)
    {
        m_bSettingScroller = true;
	scroller->setValue((int)dpos);
	m_bSettingScroller = false;
    }
    char dropstring[100];
#ifdef USE_SDL
    int w=400;
#else
    int w=width();
    int h=height();
#endif
    double percent;
    IAviPlayer::State state = player->GetState(&percent);

    switch(state)
    {
    case IAviPlayer::Invalid:
        dropstring[0] = 0;
	break;
    case IAviPlayer::Opening:
	strcpy(dropstring, "Opening...");
	break;
    case IAviPlayer::Buffering:
	sprintf(dropstring, "Buffering: %d%%", int(100*percent));
	break;
    case IAviPlayer::Paused:
	strcpy(dropstring, "Paused");
	break;
    case IAviPlayer::Stopped:
	strcpy(dropstring, "Stopped");
	break;
    case IAviPlayer::Playing:
	{
	    int drop;
	    player->Get(player->QUERY_AVG_DROP, &drop, 0);
	    if (drop > 0)
		sprintf(dropstring, "Drop: %d%%", drop);
	    else
		strcpy(dropstring, "Playing...");
	}
	break;
    }

    l_status->setText(dropstring);

    if(w > 90 && is_tracking != 1)
    	updatePosDisplay((int)dpos);
}

PlayerControl::PlayerControl(int width, int height)
    : QWidget(0, 0, WDestructiveClose)
{
    player = 0;
    popup = 0;
    about_popup = 0;
    resize_count = 1;
    is_tracking = 0;
    vollast = -1;
    _in_reseek = 0;
    m_bKeyProcessed = false;
    resize(width, height);

    QVBoxLayout* top_vbl = new QVBoxLayout(this, 0);
    top_vbl->setMargin(2);
    top_vbl->setSpacing(2);

    QGroupBox* line1_gb = new QGroupBox(this);
    //line1_gb->setBackgroundColor(QColor(0,255,255));
    line1_gb->setFrameStyle(QFrame::NoFrame);
    top_vbl->addWidget(line1_gb);

    QBoxLayout* line1_hbl = new QHBoxLayout(line1_gb, 0);
    line1_hbl->setSpacing(4);

    QHBox* controls_gb = new QHBox(line1_gb);
    controls_gb->setFrameStyle(QFrame::NoFrame);
    line1_hbl->addWidget(controls_gb);

    b_open = new PixmapButton("open", controls_gb);
    b_pause = new PixmapButton("pause", controls_gb);
    b_play = new PixmapButton("play", controls_gb);
    b_stop = new PixmapButton("stop", controls_gb);

    scroller=new QSlider(line1_gb, "Slider1");
    scroller->setOrientation(QSlider::Horizontal);
    scroller->setTickmarks(QSlider::Both);
    scroller->setMaximumHeight(18);
    scroller->setMinValue(0);
    scroller->setMaxValue(120);
    scroller->setLineStep(10);

    line1_hbl->addWidget(scroller);

    b_about=new PixmapButton("about", line1_gb);
    line1_hbl->addWidget(b_about);

    // Here we can set an icon for the application.
    // Does anyone know how to draw an AVI player? :) No? Well,
    // then let's leave this for a while.

    QPixmap pm;
    pm.load(SHARE_PATH"/test.png");
    setIcon(pm);

    QGroupBox* line2_gb = new QGroupBox(this);
    line2_gb->setFrameStyle(QFrame::NoFrame);
    //line2_gb->setBackgroundColor(QColor(0,255,255));
    top_vbl->addWidget(line2_gb);


    QBoxLayout* line2_hbl = new QHBoxLayout(line2_gb, 0);
    line2_hbl->setSpacing(4);

    b_opensub = new PixmapButton("opens", line2_gb);
    line2_hbl->addWidget(b_opensub);

    b_openaud = new PixmapButton("opena", line2_gb);
    line2_hbl->addWidget(b_openaud);

    b_mute = new PixmapButton("mute", line2_gb);
    line2_hbl->addWidget(b_mute);

    vbar=new QSlider(line2_gb, "SliderVolume");
    vbar->setOrientation(QSlider::Horizontal);
    vbar->setMinValue(0);
    vbar->setMaxValue(MAX_VBAR);
    vbar->setPageStep(5);
    vbar->setMaximumHeight(15);
    vbar->setMaximumWidth(80);
    vbar->setValue(MAX_VBAR);
    //vbar->setTickmarks(QSlider::Above);
    line2_hbl->addWidget(vbar);

    pbar = new QProgressBar(line2_gb);
    pbar->setTotalSteps(100);
    pbar->setMaximumHeight(15);
    pbar->setMaximumWidth(50);
    pbar->setProgress(0);
    line2_hbl->addWidget(pbar);

    l_status=new QLabel(line2_gb, "LabelDrop");
    line2_hbl->addWidget(l_status);
    line2_hbl->addStretch(100);

    l_time=new QLabel(line2_gb, "LabelTime");
    line2_hbl->addWidget(l_time);

    connect(scroller, SIGNAL(sliderReleased()), this, SLOT(reseek_func()));
    connect(scroller, SIGNAL(sliderPressed()), this, SLOT(slider_func()));
    connect(scroller, SIGNAL(sliderMoved(int)), this, SLOT(moved_func(int)));
    connect(scroller, SIGNAL(valueChanged(int)), this, SLOT(changed_func(int)));

    //connect(scroller, SIGNAL(sliderPressed()), this, SLOT(slider_func()));
    //connect(scroller, SIGNAL(sliderMoved(int)), this, SLOT(moved_func(int)));
    connect(vbar, SIGNAL(valueChanged(int)), this, SLOT(volume_func(int)));

    connect(b_pause, SIGNAL(clicked()), this, SLOT(pause_func()));
    connect(b_open, SIGNAL(clicked()), this, SLOT(open_func()));
    connect(b_stop, SIGNAL(clicked()), this, SLOT(stop_func()));
    connect(b_play, SIGNAL(clicked()), this, SLOT(play_func()));
    connect(b_about, SIGNAL(clicked()), this, SLOT(about_func()));
    connect(b_mute, SIGNAL(clicked()), this, SLOT(mute_func()));
    connect(b_opensub, SIGNAL(clicked()), this, SLOT(opensub_func()));
    connect(b_openaud, SIGNAL(clicked()), this, SLOT(openaud_func()));

    _timer=new QTimer(this);
    QObject::connect(_timer, SIGNAL(timeout()), this, SLOT(updatePos()));

    _start_timer=new QTimer(this);
    QObject::connect(_start_timer, SIGNAL(timeout()),
		     this, SLOT(starter_slot()));

    setAcceptDrops(TRUE);
}

PlayerControl::~PlayerControl()
{
    // empty (player destroyed through close event
}

void PlayerControl::help()
{
    QMessageBox::information( 0, tr( "AVI play usage" ),
                              tr(
                                  "<h3>Usage:</h3>"
                                  "aviplay [options] [your-file] [subtitle-file]<br>"
                                  "<h3>Key bindings:</h3>"
                                  "Keypad '+'/'-' - volume up/down.<br>"
                                  "Cursor left/right - previous/next keyframe.<br>"
                                  "Cursor up/down - 60 seconds forward/backward.<br>"
                                  "PageUp/PageDown - 10 seconds forward/backward.<br>"
                                  "Enter - display next frame.<br>"
                                  "Home/End - movie begin/end.<br>"
                                  "F, Escape or Alt + Enter - toggles fullscreen/windowed mode.<br>"
                                  "M - toggles fullscreen/windowed mode with maximization.<br>"
                                  "Q - quit<br>"
                                  "X - stop<br>"
                                  "V - play<br>"
                                  "[ / ] - adjust a-v sync by 0.1 second.<br>"
                                  "C, P or Space - pause<br>"
                                  "1,2,3 - switch zoom 0.5x, 1x, 2x.<br>"
                                  "Left-click in player window - pause/play.<br>"
                                  "Right-click in movie window brings up zoom/fullscreen menu.<br>"
                                  "F1-F2, F5-F6, F7-F8, F9-F10 - set Saturation, Brightness, Hue, Contrast.<br>"
                                  //"Double-click in player window when paused/stopped "
                                  //"to bring up configuration menu.<br>"
                                  ),
                              QMessageBox::Ok );
}

void PlayerControl::about()
{
    QMessageBox::information( 0, tr( "About AVIPlay" ),
                              tr(
                                  "AVI player version %1<br>"
                                  "(C) 2000 - 2001 Eugene Kuznetsov, Zdenek Kabelac <br>"
                                  "Based on avifile ( <a href=\"http://avifile.sf.net\">http://avifile.sf.net/</a>).<br>"
                                  "Distributed under the GNU Public Licence (GPL) version 2.<br>"
                                  "Special thanks to: "
                                  "<ul>"
                                  "<li>Authors of Wine project for Win32 DLL loading code."
                                  "<li>Avery Lee for AVI and ASF parser and his VirtualDub."
                                  "<li>Hiroshi Yamashita for porting sources to FreeBSD."
                                  "<li>JArgen Keil for porting sources to Solaris 8 x86."
                                  "<li>ArpAd GereAffy (A'rpi/ESP-team) and the rest of Mplayer team."
                                  "<li>All people from XMPS & avifile mailing lists for their patience and support."
                                  "</ul>" ).arg( AVIFILE_VERSION ),
                              QMessageBox::Ok );
}

void PlayerControl::dropEvent(QDropEvent* e)
{
    QStrList i;
    QUriDrag::decode(e, i);
    const char* uri=i.first();

    if (!uri || strlen(uri) < 6)
	return;

    QString file;

    if (0 == qstrnicmp(uri,"file:/",6) )
    {
	uri += 6;
	if ( uri[0] != '/' || uri[1] == '/' )
	{
	    // It is local.
	    file = QUriDrag::uriToUnicodeUri(uri);
	    if ( (uri[1] == '/') && (uri[0]=='/') )
		file.remove(0,1);
	    else
		file.insert(0,'/');
	    try
	    {
		cerr<<file.ascii()<<endl;
		AviPlayerInitParams ip;
                ip.filename = file.ascii();
		initPlayer(ip);
	    }
	    catch(...)
	    {
	    }
	}
    }
}

void PlayerControl::dragEnterEvent(QDragEnterEvent* event)
{
    event->accept(QUriDrag::canDecode(event));
}

void PlayerControl::moved_func(int value)
{
    updatePosDisplay(value);
}

#include "playercontrol.moc"
