/********************************************************
 *
 *       Video renderers
 *       Copyright 2000 Eugene Kuznetsov (divx@euro.ru)
 *
 ********************************************************/

#include "default.h"
#include "fourcc.h"
#include "Locker.h"
#include "except.h"
#include "renderer.h"
#include "playerwidget.h"
#include "cpuinfo.h"
#include "utils.h"
#include "mmx.h"

#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <stdarg.h>

#ifndef WITHOUT_X

#include <errno.h>

#include <sys/ipc.h>
#include <sys/shm.h>

#include <X11/Xlib.h>
#include <X11/Xlocale.h>
#include <X11/Xutil.h>
#include <X11/extensions/XInput.h> //XGetExtensionVersion
#include <X11/extensions/XShm.h>

#ifdef HAVE_XV
#include <X11/extensions/Xv.h>
#include <X11/extensions/Xvlib.h>
#endif

#ifdef HAVE_LIBXXF86DGA
#include <X11/extensions/xf86dga.h>
#endif

#ifdef HAVE_LIBXXF86VM
#include <X11/extensions/xf86vmode.h>
#endif
#ifdef HAVE_XFT
#include <X11/Xft/Xft.h>
//#include <X11/extensions/Xrender.h>
#endif

#include "VideoDPMS.h"

/*
#ifdef USE_QT
#include <qfont.h>
#include <qfontmetrics.h>
#endif
*/

#ifdef USE_SDL
#include <SDL.h>
#include <SDL_video.h>
#include <SDL_syswm.h>
#include <SDL_mouse.h>

#ifndef SDL_DISABLE
#define SDL_DISABLE 0
#endif

#ifndef SDL_ENABLE
#define SDL_ENABLE 1
#endif

#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ostream.h>

#define __MODULE__ "VideoRenderer"

#ifdef I18N
#include <iconv.h>
#define XLoadQueryFont XLoadQueryFontSet
#define XFreeFont XFreeFontSet

XFontSet XLoadQueryFontSet(Display *,const char *);

XFontSet XLoadQueryFontSet(Display *disp,const char *fontset_name)
{
  XFontSet fontset;
  int  missing_charset_count;
  char **missing_charset_list;
  char *def_string;

  fontset = XCreateFontSet(disp, fontset_name,
                           &missing_charset_list, &missing_charset_count,
                           &def_string);
  if (missing_charset_count) {
    fprintf(stderr, "Missing charsets in FontSet(%s) creation.\n",
	 fontset_name);
    XFreeStringList(missing_charset_list);
  }
  return fontset;
}
#define XDrawString(d,w,gc,x,y,s,l)  XmbDrawString(d,w,font,gc,x,y,s,l)
#define XDrawImageString(d,w,gc,x,y,s,l)  XmbDrawImageString(d,w,font,gc,x,y,s,l)
#define XTextWidth XmbTextEscapement
#endif /*I18N*/

using namespace std;

bool VideoRenderer::allow_sw_yuv=true;
//false;
bool VideoRenderer::direct_resizes=false;

int GetPhysicalDepth(void* _dpy)
{
    Display* dpy = (Display*) _dpy;
    if (!dpy)
	return 0;

    int screen = DefaultScreen(dpy);
    int planes = DefaultDepth(dpy, screen);
    int n, pixmap_bits = 0;
    XPixmapFormatValues *pf = XListPixmapFormats(dpy, &n);

    for (int i = 0; i < n; i++)
    {
	//cout << i << " depth: " << pf[i].depth << "  bpp: " << pf[i].bits_per_pixel << endl;
	if(pf[i].depth == planes)
	{
	    pixmap_bits = pf[i].bits_per_pixel;
            break;
	}
    }
    XFree(pf);

    if (pixmap_bits == 16 && DefaultVisual(dpy, screen)->red_mask == 0x7c00)
	pixmap_bits = 15;

    return pixmap_bits;
}

static void* EventThread(void *arg);

// use our internal class for our renderers;
class VideoRendererWithLock : public VideoRenderer
{
    friend void* EventThread(void*);
protected:
    int m_w, m_h;
    int pic_w, pic_h;
    int m_sub;
    int fm_height;
    bool subtitles;
    PthreadMutex m_Mutex;
    int64_t m_lLastDrawStamp;
    avm::vector<VideoMode> modes;
    bool quit;
    PlayerWidget* g_pw;

#ifdef USE_SDL
    virtual int eventFilter(const SDL_Event* ev) = 0;
#endif

public:

    VideoRendererWithLock(int width, int height, bool _subtitles = false)
	:m_w(width), m_h(height), pic_w(width), pic_h(height),
	m_sub(0), fm_height(0), subtitles(_subtitles), quit(false),
        g_pw(0) {}
    virtual ~VideoRendererWithLock() {}
    virtual int Lock()
    {
        //cout << "lock" << endl;
	return m_Mutex.Lock();
    }
    virtual int TryLock()
    {
	//cout << "trylock" << endl;
	return m_Mutex.TryLock();
    }
    virtual int Unlock()
    {
        //cout << "unlock" << endl;
	return m_Mutex.Unlock();
    }
    virtual const avm::vector<VideoMode>& GetVideoModes() const
    {
	return modes;
    }
    virtual int GetSize(int& width, int& height) const
    {
	width = pic_w;
	height = pic_h;
	return 0;
    }
};

#ifdef USE_SDL
#undef __MODULE__
#define __MODULE__ "Fullscreen renderer"
/**
 *
 * This function processes events received by SDL window.
 *
 */
static const char event_names[][30]=
{
    "SDL_NOEVENT",
    "SDL_ACTIVEEVENT",
    "SDL_KEYDOWN",
    "SDL_KEYUP",
    "SDL_MOUSEMOTION",
    "SDL_MOUSEBUTTONDOWN",
    "SDL_MOUSEBUTTONUP",
    "SDL_JOYAXISMOTION",
    "SDL_JOYBALLMOTION",
    "SDL_JOYHATMOTION",
    "SDL_JOYBUTTONDOWN",
    "SDL_JOYBUTTONUP",
    "SDL_QUIT",
    "SDL_SYSWMEVENT",
    "SDL_EVENT_RESERVEDA",
    "SDL_EVENT_RESERVEDB",
    "SDL_VIDEORESIZE",
    "SDL_EVENT_RESERVED1",
    "SDL_EVENT_RESERVED2",
    "SDL_EVENT_RESERVED3",
    "SDL_EVENT_RESERVED4",
    "SDL_EVENT_RESERVED5",
    "SDL_EVENT_RESERVED6",
    "SDL_EVENT_RESERVED7"
};

static SDL_Event keyrepev;
static PthreadCond econd;
static PthreadMutex emutex;
static int64_t SDL_mouse_time;
static bool sdlKbdOn = true;
static int SDL_mouse_x, SDL_mouse_y;
static int SDL_mouse_off = 0;
static bool SDL_fullscreen = false;

static void MouseOff()
{
    SDL_mouse_time = 0;
    SDL_mouse_off = 3; // resistance
    SDL_ShowCursor(SDL_DISABLE);
}

static void* EventThread(void *arg)
{
    VideoRendererWithLock& vr = *(VideoRendererWithLock*) arg;
    int repeatdelay = 0;
    int64_t last_mouse = 0; // keep time of last mouse event
    for (;;)
    {
	emutex.Lock();
	// if keyboard autorepeat - shorter wait between event processing
	// otherwise wait usually for condition signal after frame Sync
        // so we will not lock drawing
	econd.Wait(emutex, (keyrepev.type == SDL_USEREVENT) ? 0.01 : 0.2);
	emutex.Unlock();

	if (vr.quit)
	    break;

        // handle keyboard autorepeat here
	if (keyrepev.type == SDL_USEREVENT)
	{
	    if (--repeatdelay < 0)
	    {
		switch (keyrepev.key.keysym.sym)
		{
		case SDLK_c:
		case SDLK_p:
		case SDLK_SPACE:
		    vr.g_pw->PW_pause_func();
		    break;
		default:
		    vr.g_pw->PW_key_func(keyrepev.key.keysym.sym, keyrepev.key.keysym.mod);
		    break;
		}
	    }
	}
	else
	    repeatdelay = 10;

	if (vr.TryLock() == 0)
	{
	    SDL_Event event;
	    if (SDL_mouse_time)
	    {
		if (to_float(longcount(), SDL_mouse_time) > 1.0)
		    MouseOff();
		else if (SDL_mouse_time != last_mouse && --SDL_mouse_off == 0)
		    SDL_ShowCursor(SDL_ENABLE); // enable if mouse is moving

		last_mouse = SDL_mouse_time;
	    }
            bool locked = true;
	    while ( SDL_PollEvent(&event) )
	    {
		vr.Unlock();
		vr.eventFilter(&event);
		if (vr.TryLock() != 0)
		{
		    locked = false;
		    break;
		}
	    }
            if (locked)
		vr.Unlock();
	}
    }
    return 0;
}

class EnvKeeper
{
    static const char* sdl_var;
    char* str;
public:
    EnvKeeper()
    {
	str = getenv(sdl_var);
	if (str)
	{
	    char* tmp = new char[strlen(str)+1];
	    strcpy(tmp, str);
	    str = tmp;
	}
	avm_setenv(sdl_var, "1", true);
    }
    ~EnvKeeper()
    {
	if (!str)
	    avm_unsetenv(sdl_var);
	delete str;
    }
};
// according to SDL we should not use this
// new XFree should have this fixed
//static EnvKeeper keeper;
const char* EnvKeeper::sdl_var = "SDL_VIDEO_X11_NODIRECTCOLOR";

class FullscreenRenderer: public VideoRendererWithLock
{
protected:
    Display* dpy;
    Visual* vis;
    Window wnd;
    GC xgc;
    GC wmxgc;
    SDL_Surface* screen;
    SDL_SysWMinfo info;
    SDL_Rect cliprect;
    int dim_w, dim_h;
    int max_w, max_h;
    int fs;
    int dga;
    int bit_depth;
    int bpp;
    int mouse_x, mouse_y;
    int zoom_divider;
    const CImage* image;
    char *convbuf;
    PthreadTask* eventchecker;
    VideoDPMS* dpmsSafe;
    char* old_text;
    const subtitle_line_t* m_pSublineRef;
    bool dirty;
    bool resizeEnabled;
    avm::vector<SDL_Surface*> sflist;
    subtitle_line_t* m_pSubline;

#ifdef I18N
    XFontSet font;
    char *i18nfileencoding;
    char *i18ndisplayencoding;
    bool i18ncodeconvert;
#else
    XFontStruct* font;
#endif
#ifdef HAVE_XFT
    XftDraw *xftdraw;
    XftFont *xftfont;
    XftColor *xftcolor;
#endif
    Uint32 sdl_systems;

public:
    static const int SUBTITLE_SPACE = 4;
    static int s_iTrickNvidia;

    FullscreenRenderer(PlayerWidget* pw, Display* _dpy,
		       int width, int height, bool subt = false)
	:VideoRendererWithLock(width, height, subt),
	dpy(0), xgc(0), wmxgc(0), screen(0), max_w(0), max_h(0), image(0),
	convbuf(0), dpmsSafe(0), old_text(0), resizeEnabled(true), font(0)
    {
	g_pw = pw;
	sdlKbdOn = true;
	m_sub = 0;
	m_pSubline = 0;
        m_pSublineRef = 0;

#ifdef HAVE_XFT
	xftdraw = 0;
	xftfont = 0;
	xftcolor = 0;
#endif
	char *s;
	int found = False;

        s = setlocale(LC_CTYPE, "");
	if (!s)
	    cerr << "Warning: Locale not supported by C library" << endl;
	else
	{
	    if (!XSupportsLocale())
	    {
		cerr << "Warning: Locale non supported by Xlib" << endl;
                setlocale(LC_CTYPE, "C");
	    }
	    //cout << "Locale OK" << endl;
#ifdef I18N
	char *p;
	i18ndisplayencoding="iso-8859-1";
	i18nfileencoding="iso-8859-1";
	if (strncmp(s,"ja",2)==0) {
		i18ndisplayencoding="EUC-JP";
		i18nfileencoding="EUC-JP";
	}
	if (strncmp(s,"ko",2)==0) {
		i18ndisplayencoding="EUC-KR";
		i18nfileencoding="EUC-KR";
	}
	if (strncmp(s,"zh",2)==0) {
		if (strncmp(s,"zh_TW",4)==0) {
			i18ndisplayencoding="BIG5";
			i18nfileencoding="BIG5";
		}
		else {
			i18ndisplayencoding="GB2312";
			i18nfileencoding="GB2312";
		}
	}
	p = getenv("SUBTITLEFILEENCODING");
	if (p)
		i18nfileencoding=p;
	p = getenv("SUBTITLEDISPLAYENCODING");
	if (p)
		i18ndisplayencoding=p;
	if (strcmp(i18ndisplayencoding,i18nfileencoding)!=0)
		i18ncodeconvert=true;
	else
		i18ncodeconvert=false;

	cout<<"Input "<<i18nfileencoding<<" output "<<i18ndisplayencoding<<endl;
#endif
	}

        sdl_systems = 0;
	Uint32 subsystem_init = SDL_WasInit(SDL_INIT_EVERYTHING);

	if (!(subsystem_init & SDL_INIT_VIDEO))
	{
	    if (subsystem_init == 0)
	    {
		SDL_Init(SDL_INIT_NOPARACHUTE);
		atexit(SDL_Quit);
	    }

	    if (SDL_InitSubSystem(SDL_INIT_VIDEO
				  //| SDL_INIT_EVENTTHREAD
				 ) < 0)
		throw FATAL("Failed to init SDL_VIDEO");
            sdl_systems |= SDL_INIT_VIDEO;
	}

	if (0 && !(subsystem_init & SDL_INIT_TIMER))
	{
	    if (SDL_Init(SDL_INIT_TIMER) < 0)
		throw FATAL("Failed to init SDL_TIMER");
            sdl_systems |= SDL_INIT_TIMER;
	}
#if 0
	SDL_EventState(SDL_SYSWMEVENT, SDL_IGNORE);
	SDL_EventState(SDL_USEREVENT, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEBUTTONUP, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEBUTTONDOWN, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);
	SDL_EventState(SDL_ACTIVEEVENT, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);
	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, 10/*SDL_DEFAULT_REPEAT_INTERVAL*/);
#endif
	//SDL_WM_SetCaption("AviPlayer: M/F max/fullscreen, [ ] adjust A-V sync", 0);
	SDL_WM_SetCaption("AviPlayer: M/F max/fullscreen", 0);

	//fs = SDL_RESIZABLE | SDL_SWSURFACE | SDL_HWACCEL | SDL_ANYFORMAT;
	fs = SDL_RESIZABLE | SDL_HWSURFACE | SDL_HWACCEL | SDL_ANYFORMAT;

	//fs |= SDL_DOUBLEBUF;
	//fs |= SDL_ASYNCBLIT;
	// Async blit makes this a bit faster - however we prefer
	// to know when the image is drawn
	// - Async uses XFlush, Sync calls XSync
	// using double buffering would be great
        // but for some reason it doesn't work for YUV
	char vname[100];
	SDL_VideoDriverName(vname, sizeof(vname));
	cout << "SDL Video driver: " << vname << endl;

	dga = 0;
        pic_w = -1;
	if (getenv("SDL_VIDEODRIVER") != NULL
	    && strcmp(getenv ("SDL_VIDEODRIVER"), "dga") == 0) {
	    dga = 1;
	    char size[10];
	    double ratio_x;
	    double ratio_y;
	    double ratio;
	    if (getenv("AVIPLAY_DGA_WIDTH") == NULL)
		ratio_x = 640.0;
	    else {
		strncpy (size, getenv("AVIPLAY_DGA_WIDTH"), 9);
		ratio_x = atof (size);
	    }
	    if (getenv("AVIPLAY_DGA_HEIGHT") == NULL)
		ratio_y = 480.0;
	    else {
		strncpy(size, getenv("AVIPLAY_DGA_HEIGHT"), 9);
		ratio_y = atof (size);
	    }
	    ratio_x = ratio_x / m_w;
	    ratio_y = ratio_y / m_h;
	    ratio = (ratio_x < ratio_y) ? ratio_x : ratio_y;
	    dim_w = (int) rint(ratio * m_w);
	    dim_h = (int) rint(ratio * m_h) * 2;
	    //cout << "dim " << dim_w << "   " << dim_h << endl;
	}
	else
	{
	    dim_w = width;
	    dim_h = height;
	}

	dirty = false;

	/*int newbpp = SDL_VideoModeOK(dim_w, dim_h, GetPhysicalDepth(_dpy), fs);
	  if (!newbpp)
	    throw FATAL("Failed to set up video mode");
	 */

	doResize(dim_w, dim_h);
	if (dga)
            resizeEnabled = false;
	// doResize already creates screen = SDL_SetVideoMode(dim_w, dim_h, 0, fs);

	if (!screen)
	    throw FATAL("Failed to set up video mode");

	try
	{
	    SDL_EventState(SDL_SYSWMEVENT, SDL_ENABLE);

	    bit_depth = screen->format->BitsPerPixel;
	    if (bit_depth == 16 && screen->format->Rmask == 0x7c00)
		bit_depth = 15;

	    bpp = (bit_depth + 7) / 8;

	    if (!dga)
		dpy = _dpy;
	    else
		MouseOff();

	    if (dpy)
	    {
		nvidiaCheck(); // hack for TNT2 and bad NVidia driver

		//::wnd=info.info.x11.wmwindow;
		dpmsSafe = new VideoDPMS(dpy);
		keyrepev.type = 0;

#ifdef HAVE_LIBXXF86VM
		XF86VidModeModeInfo** modesinfo = 0;
		int lines;
		Bool r = XF86VidModeGetAllModeLines(dpy, DefaultScreen(dpy),
						    &lines, &modesinfo);
		if (r)
		{
		    for (int i = 0; modesinfo && modesinfo[i] && i < lines; i++)
		    {
			//cout << "Modes  " << modesinfo[i]->hdisplay << " x "  << modesinfo[i]->vdisplay << endl;
			VideoMode vm;
			vm.width = modesinfo[i]->hdisplay;
			vm.height = modesinfo[i]->vdisplay;
			// some systems provides zeroes here... ???
                        // ATI Radeon VE 32MB @ 1280x1024 some user
			vm.freq = (modesinfo[i]->htotal && modesinfo[i]->vtotal)
			    ? (modesinfo[i]->dotclock * 1000
			       / modesinfo[i]->htotal
			       / (double) modesinfo[i]->vtotal) : 0;
			//calculate refresh frequency
			//cout << "dt " << modesinfo[i]->dotclock
			//    << "   ht " << modesinfo[i]->htotal
			//    << "   vt " << modesinfo[i]->vtotal << endl;
			char b[100];
			sprintf(b, "%d x %d  %dHz", vm.width, vm.height, (int)(vm.freq + 0.5));
			vm.name = b;
			modes.push_back(vm);
		    }
		}
		else
                    cout << "Can't resolve video modes..." << endl;
#endif
		SDL_VERSION(&info.version);
		SDL_GetWMInfo(&info);

		dpy = info.info.x11.display;
	    }

	    eventchecker = new PthreadTask(0, EventThread, this);
	    //cout << "SDL surface w: " << pic_w << "  h: " << pic_h << endl;
	}
	catch(...)
	{
	    SDL_QuitSubSystem(sdl_systems);
	    throw;
	}
    }

    virtual ~FullscreenRenderer()
    {
	if (screen && (screen->flags & SDL_FULLSCREEN))
	    SDL_WM_ToggleFullScreen(screen);
	quit = true;

	Lock();
        delete eventchecker;
	//SDL_SetEventFilter(0);
	SDL_ShowCursor(SDL_ENABLE);

	while (sflist.size() > 0)
	{
	    SDL_FreeSurface(sflist.back());
            sflist.pop_back();
	}
	if (image)
	{
	    image->Release();
	    image = 0;
	}
	g_pw = 0;

	if (m_pSubline)
            subtitle_line_free(m_pSubline);
	if (convbuf)
	    free(convbuf);

	freeFont();
        if (xgc)
	    XFreeGC(dpy, xgc);
        if (wmxgc)
	    XFreeGC(dpy, wmxgc);

        Unlock();
	SDL_QuitSubSystem(sdl_systems);
	delete dpmsSafe;
    }
    int Set(...)
    {
	va_list args;
	Property prop;

	va_start(args, this);

	while ((prop = (Property) va_arg(args, int)) != 0)
	{

	}
	va_end(args);
	return 0;
    }
    int Get(...) const
    {
	va_list args;
	Property prop;

	//printf("Sizeof %d\n", sizeof(Property));
	va_start(args, this);

	va_end(args);

	return 0;
    }
    virtual CImage* GetData(int idx)
    {
	// for direct rendering bitdepth must match!
        resizeEnabled = false; // not allowed for x11 resizing

	if (pic_w != m_w || pic_h != m_h)
	    return 0;

	SDL_Surface* sf = 0;
	//Lock()

	while ((unsigned)idx >= sflist.size())
	{
	    if (idx == 0)
		sf = screen;
	    else if (idx > 10000)  // disabled
	    {
		SDL_Surface* cs = SDL_GetVideoSurface();
		Uint32 rmask, gmask, bmask, amask;

		sf = SDL_CreateRGBSurface(SDL_HWSURFACE,
					  cs->w, cs->h,
					  cs->format->BitsPerPixel,
					  cs->format->Rmask,
					  cs->format->Gmask,
					  cs->format->Bmask,
					  cs->format->Amask);
	    }

	    if (!sf)
		return 0;

	    //cout << "SURFACE " << (void*) sf << endl;
	    sflist.push_back(sf);
	}

	sf = &(*sflist[idx]);

	BitmapInfo bi;
        bi.biWidth = m_w;
	bi.biHeight = -m_h;
	bi.biPlanes = 1;
	bi.SetBits(bit_depth);

	//cout << "GETDATA " << endl;
	//bi.Print();

	CImage* ci = new CImage(&bi, (uint8_t*) sf->pixels, false);
	ci->SetUserData(sf);
	//cout << "HWADDR " << (void*) screen->pixels << endl;
	return ci;
    }
    virtual int Resize(int& new_w, int& new_h)
    {
	if (dga)
            return -1;
	Lock();
	int r = doResize(new_w, new_h);
	Unlock();
	if (r == 0)
	    Refresh();
	return r;
    }
    virtual int Refresh()
    {
	if (screen && image && to_float(longcount(), m_lLastDrawStamp) > 0.1)
	{
            // wait until we could really refresh image
	    Lock();
	    Unlock();
	    // because of scaling we need to call Draw
	    if (Draw(image) == 0)
	    {
		if (m_pSubline)
		    DrawSubtitles(m_pSubline);
		return Sync();
	    }
	}
        return -1;
    }
    virtual int Draw(const CImage* data)
    {
	if (!screen || !data)
	    return -1;
	if (TryLock() != 0)
	    return -1;
	// we have to prevent resizing of screen->pixels
#if 0
	const SDL_VideoInfo* vi = SDL_GetVideoInfo();
	printf("VideoInfo: %s  %s  %s  %s  %s -  %s  %s  %s  -  %s   vmem: %d    colorkey: 0x%x  alpha: 0x%x\n",
	       (vi->hw_available) ? "hw available" : "",
	       (vi->wm_available) ? "wm_available" : "",
	       (vi->blit_hw) ? "blit_hw" : "",
	       (vi->blit_hw_CC) ? "blit_hw_CC" : "",
	       (vi->blit_hw_A) ? "blit_hw_A" : "",
	       (vi->blit_sw) ? "blit_sw" : "",
	       (vi->blit_sw_CC) ? "blit_sw_CC" : "",
	       (vi->blit_sw_A) ? "blit_sw_A" : "",
	       (vi->blit_fill) ? "blit_fill" : "",
	       vi->video_mem,

	       vi->vfmt->colorkey,
	       vi->vfmt->alpha
	      );

	printf("ScreenInfo:  colorkey: 0x%x  alpha: 0x%x\n",
	       screen->format->colorkey,
	       screen->format->alpha);
	char* tmp_array=new char[m_w*m_h];
	char* tmp_array2=new char[m_w*m_h];
	uint_t t3=localcount();
    	memcpy(tmp_array2, tmp_array, m_w*m_h);
	uint_t t4=localcount();
	SDL_LockSurface(screen);
	uint_t t1=localcount();
    	memcpy(screen->pixels, tmp_array, m_w*m_h);
	uint_t t2=localcount();
	SDL_UnlockSurface(screen);
	delete[] tmp_array;
	delete[] tmp_array2;
	printf("Pixels: %f Mb/s System: %f Mb/s\n",
	    (m_w*m_h/1048576.)/((t2-t1)/freq/1000.),
	    (m_w*m_h/1048576.)/((t4-t3)/freq/1000.)
	    );
#endif

        m_lLastDrawStamp = longcount();
	data->AddRef();
	if (image)
	{
            //cout << "Draw() release " << (void*)image << endl;
	    image->Release();
	}
	image = data;

	//cout << "DRAW TO " << pic_w << " " << pic_h << endl;
	if (dga == 1) {
	    if (!data->GetUserData())
	    {
		SDL_LockSurface(screen);
		avm_memory_lock();
		if ((pic_w == m_w) && (pic_h == m_h))
		    memcpy(screen->pixels, data->Data(), data->Bytes());
		else
		{
		    // the reason for convbuf is that writing to VideoRAM is slower
		    // compared to main memory. If you don't believe me try it
		    // yourself and see how performance drops. Strange however:
		    // doing more results in more fluent drawing than doing less
		    // :-)
		    if (!convbuf)
			convbuf = (char*) malloc(dim_w * dim_h * bpp);
		    zoom ((uint16_t*)convbuf, (const uint16_t*)data->Data(),
			  pic_w, pic_h, m_w, m_h, bit_depth);
		    memcpy(screen->pixels, convbuf, dim_w * dim_h * bpp);
		}
		avm_memory_unlock();
		SDL_UnlockSurface(screen);
	    }
	}
	else if ((pic_w == m_w) && (pic_h == m_h))
	{
	    //cout << "DATA " << (void*) data << endl;
	    SDL_LockSurface(screen);
	    avm_memory_lock();
	    if (!data->GetUserData())
	    {
                resizeEnabled = true;
		//cout << "NONDIRECT DRAW " << endl;
		//data->GetFmt()->Print();
		if (data->Direction())
		{
		    // currently only in original SIZE
		    //printf("ss %.4s\n", &data->GetFmt()->biCompression);
		    uint8_t* out = (uint8_t*) screen->pixels;
		    const uint8_t* in = data->Data();
		    const uint8_t* inbegin = in;
		    int bpl = data->Bpl();
		    if (bpl > 0)
		    {
			in += data->Bytes() - bpl;
			while (in >= inbegin)
			{
			    memcpy(out, in, bpl);
			    in -= bpl;
			    out += bpl;
			}
		    }
		}
		else
		{
		    //printf("MEM %p   %p  %d\n",
		    //       screen->pixels, data->Data(), data->Bytes());
		    if (!data->Data())
			cout << "Renderer - Data() == NULL ???" << endl;
		    else
		    {
			//cout << "MEMCPY" << endl;
			//data->GetFmt()->Print();
			//fflush(stdout);
			memcpy(screen->pixels, data->Data(), data->Bytes());
		    }
		}

	    }
            else if (data->GetUserData())
	    {
		//cout << "Call sdl flip  " << data->GetUserData() << endl;
		//SDL_Flip((SDL_Surface*)data->GetUserData());
		SDL_Surface* from = (SDL_Surface*)data->GetUserData();
		//cout << "from " << (void*) from << " to " << (void*) screen << endl;
                if (from != screen)
		    SDL_BlitSurface(from, 0, screen, 0);
	    }
	    avm_memory_unlock();
	    SDL_UnlockSurface(screen);
	    //else cout << "Direct"  << endl;
	}
	else
	{
            // FIXME write zoom with Upside-Down flipping
	    SDL_LockSurface(screen);
	    avm_memory_lock();
	    zoom((uint16_t*)screen->pixels, (const uint16_t*)data->Data(), pic_w, pic_h, m_w, m_h, bit_depth);
	    avm_memory_unlock();
	    SDL_UnlockSurface(screen);
	}

	Unlock();
	dirty = true;
	return 0;
    }
    virtual int Sync()
    {
	if (dirty && !dga)
	{
	    if (TryLock() != 0)
		return -1;
	    dirty = false;

	    //SDL_Flip(screen);
	    // equals for non double
	    SDL_UpdateRect(screen, cliprect.x, cliprect.x,
			   cliprect.w, cliprect.h);

	    //cout << "Screen flip" << endl;
	    Unlock();
	}
        econd.Broadcast();
	return 0;
    }
    virtual int ToggleFullscreen(bool maximize = false)
    {
	if (dga)
	    return -1;
	Lock();

	fs = screen->flags;
	if (!(fs & SDL_FULLSCREEN))
	{
	    storeMousePosition();
	    if (maximize && resizeEnabled)
	    {
		int w = max_w = pic_w; // set some defaults
		int h = max_h = pic_h;
		GetModeLine(w, h);
		double ratio_x = (double) w / pic_w;
		double ratio_y = (double) h / pic_h;
		double ratio = (ratio_x < ratio_y) ? ratio_x : ratio_y;
		w = (int) rint(ratio * pic_w);
		h = (int) rint(ratio * pic_h);
		if (pic_w != w || pic_h != h)
		    //XMoveWindow(dpy, info.info.x11.wmwindow, 0, 0);
		    // FIXME - needs to flush XQueue here to be able to move
		    // this window before Toggle occures
		    if (doResize(w, h) < 0)
			max_w = max_h = 0;
	    }
	}

	SDL_WM_ToggleFullScreen(screen);
	fs = screen->flags;
	SDL_fullscreen = (fs & SDL_FULLSCREEN);

        //cout << "MAX " << max_w << "   " << max_h << "  " << fs << endl;
	if (!(fs & SDL_FULLSCREEN))
	{
	    restoreMousePosition();
	    if (max_w && max_h)
		doResize(max_w, max_h);
	    max_w = max_h = 0;
	}
        MouseOff();
	Unlock();
	Refresh();
	return (fs & SDL_FULLSCREEN);
    }
    virtual int DrawSubtitles(const subtitle_line_t* sl)
    {
#ifdef I18N
	uint_t sts;
	iconv_t icvsts;
	uint_t in_size,out_size,out_count;
	char *out_buffer;
	char *out_p;
#endif
	if (!dpy || !xgc)
	    return -1;
	if (subtitle_line_equals(sl, m_pSubline))
	    return 0;

	//if(sl && sl->line && sl->line[0])
    	//    cout<<sl->line[0]<<endl;
	if (TryLock() != 0)
	    return -1;

	Window win = info.info.x11.window;
	// here is usage of two different GC better
	GC lxgc = xgc;//(fs & SDL_FULLSCREEN) ? wmxgc : xgc;
        // but here correct info is in window all the time (don't ask me why)

	XSetForeground(dpy, lxgc, 0x0);
	XFillRectangle(dpy, win, lxgc, 0, dim_h - m_sub, dim_w, m_sub);
	XSetForeground(dpy, lxgc, 0xFFFFFFFF);

	if (m_pSubline)
	    subtitle_line_free(m_pSubline);
	m_pSubline = (sl) ? subtitle_line_copy(sl) : NULL;

	for (int i = 0; m_pSubline && i < m_pSubline->lines; i++)
	{
	    char *sub = m_pSubline->line[i];

	    if (sub == NULL)
	    {
		cout << "??? Sub NULL line " << i << endl;
                break;
	    }
	    //cout << "Line " << i << "  " << sub << endl;
//	    XDrawString16(dpy, win, lxgc, dim_w/2-6*strlen(sub)/2,
//	                  dim_h-65+20*i,
//	                  (XChar2b*)(short*)QString::fromLocal8Bit(sub).unicode(),
//			  strlen(sub));
	    int slen = strlen(sub);
            int draw_h = pic_h + (i + 1) * fm_height;
#ifdef HAVE_XFT
	    if (slen && xftfont)
	    {
#ifdef I18N
		in_size = slen;
		out_size = slen * sizeof(XftChar32);
		out_buffer = (char *)malloc(out_size);
		memset(out_buffer,0,out_size);
		out_p = out_buffer;

		icvsts = iconv_open("UCS-4-INTERNAL",i18nfileencoding);
		if (icvsts == (iconv_t)(-1)) {
		    cout<<"iconv open error"<<endl;
		}
		else {
		    sts = iconv(icvsts, (char **)&sub,&in_size,
					(char **)&out_p,&out_size);
		    if (sts == (uint_t)(-1)) {
			cout <<"iconv convert error"<<endl;
			out_size=0;
		    }
		}
		iconv_close(icvsts);
		XGlyphInfo extents;
		if (out_size == 0) out_count = slen;
		else 	out_count = (out_p - out_buffer)/sizeof(XftChar32);

		XftTextExtents32(dpy, xftfont,
			(XftChar32 *)out_buffer, out_count, &extents);
		int fm_width_sub = extents.xOff;
		XftDrawString32(xftdraw, xftcolor, xftfont,
				(dim_w - fm_width_sub) / 2,
				draw_h, (XftChar32 *)out_buffer, out_count);
		free(out_buffer);

#else
		wchar_t u_sub[slen + 1];
		int c = mbstowcs(u_sub, sub, slen);
		XGlyphInfo extents;
		XftTextExtents32(dpy, xftfont, (XftChar32 *)u_sub, c, &extents);
		int fm_width_sub = extents.xOff;
		//cout << "Width " << fm_width_sub << "  " << sub << endl;
		XftDrawString32(xftdraw, xftcolor, xftfont,
				(dim_w - fm_width_sub) / 2,
				draw_h, (XftChar32 *)u_sub, c);
#endif /* I18N */
	    }
#endif /* HAVE_XFT */
	    if (slen && font)
	    {
#ifdef I18N
		if (i18ncodeconvert) {
		    in_size = slen;
		    out_size = slen*2;
		    out_buffer = (char *)malloc(out_size);
		    memset(out_buffer,0,out_size);
		    out_p = out_buffer;

		    icvsts = iconv_open(i18ndisplayencoding,i18nfileencoding);
		    if (icvsts == (iconv_t)(-1)) {
		        cout<<"iconv open error"<<endl;
		    }
		    else {
		        sts = iconv(icvsts, (char **)&sub,&in_size,
					(char **)&out_p,&out_size);
		        if (sts == (uint_t)(-1)) {
			    cout <<"iconv convert error"<<endl;
			    out_size=0;
		        }
		    }
		    iconv_close(icvsts);
		    if (out_size == 0) out_count = slen;
		    else out_count = (out_p - out_buffer);
		    int fm_width_sub = XTextWidth(font, out_buffer, out_count);
		    XDrawString(dpy, win, lxgc, (dim_w - fm_width_sub) / 2,
			    draw_h, out_buffer, out_count);
		    free(out_buffer);
		}
		else {
		    int fm_width_sub = XTextWidth(font, sub, slen);
		    XDrawString(dpy, win, lxgc, (dim_w - fm_width_sub) / 2,
			    draw_h, sub, slen);
		}
#else
		int fm_width_sub = XTextWidth(font, sub, slen);
		//printf("Width %d, height %d   %s\n", fm_width_sub, fm_height, sub);
		XDrawString(dpy, win, lxgc, (dim_w - fm_width_sub) / 2,
			    draw_h, sub, slen);
#endif
	    }
	    // cause async Xerror XFlush(d);
	}
	Unlock();
	return 0;
    }
    virtual int SetCaption(const char* title, const char* icon)
    {
	if (!dpy)
	    return -1;
        Lock();
	SDL_WM_SetCaption(title, icon);
        Unlock();
        return 0;
    }
    virtual int GetPosition(int& x, int& y)
    {
	if (!dpy)
	    return -1;

	Lock();
	//XWindowAttributes xwa;
	//Status s = XGetWindowAttributes(info.info.x11.display, info.info.x11.wmwindow, &xwa);
        Window w;
	unsigned int dw, dh, dbw, dd;
        x = 100;
	Status s = XGetGeometry(info.info.x11.display,
				info.info.x11.wmwindow, &w, &x, &y,
				&dw, &dh, &dbw, &dd);
	Unlock();
	//x = xwa.x;
	//y = xwa.y;
	//printf("Window position is %d %d  %dx%d s:%d\n", x, y, dw, dh, s);
        return s;
    }
    virtual int SetPosition(int new_x, int new_y)
    {
	if (dga)
	    return -1;
	Lock();
	Status r = XMoveWindow(info.info.x11.display, info.info.x11.wmwindow, new_x, new_y);
	Unlock();
	Refresh();
	return r;
    }
    virtual int SetFont(const char* lf)
    {
	if (!dpy)
	    return -1;
	//int xp, yp;
        //GetPosition(xp, yp);

	Lock();

	Window win = info.info.x11.window;
	if (xgc == 0)
	{
	    XGCValues gcv;
	    gcv.graphics_exposures = False;
	    xgc = XCreateGC(dpy, win, GCGraphicsExposures, &gcv);
	    wmxgc = XCreateGC(dpy, win, GCGraphicsExposures, &gcv);
	}

        freeFont();
	//XSetWindowAttributes attr;
	//XWindowAttributes src;
	//XGetWindowAttributes(dpy, info.info.x11.window, &src);
	//attr.event_mask=VisibilityChangeMask;
	//XChangeWindowAttributes(dpy, info.info.x11.wmwindow, CWEventMask, &attr);
	//XChangeWindowAttributes(dpy, info.info.x11.window, CWEventMask, &attr);
	//::wnd=info.info.x11.wmwindow;

	//const char* lf;
	//GetProperties(SUBTITLE_FONT, &lf, 0);

	cout << "Loading subfont: \"" << lf << "\"" << endl;
	fm_height = 20;

	//font = XLoadQueryFont(dpy, lf);
#ifdef HAVE_XFT
	xftfont = 0; // only truetype fonts
	if (XftDefaultHasRender(dpy) == 1)
	{
	    //cout << "Display " << (void*)dpy << "   " << (void*)dpy << endl;
	    xftcolor = new XftColor;
	    xftcolor->color.red = 0xd700;
	    xftcolor->color.green = 0xdc00;
	    xftcolor->color.blue = 0xed00;
	    xftcolor->color.alpha = 0xffff;
	    xftcolor->pixel = 0xd7dced;

	    avm::string lfn = lf;

#ifdef I18N
	    XftPattern *pat,*match;
	    XftResult  res;
	    int p=1;

	    /* Check Font Name Style
	     Type1. Single Font Style.
	     "-sony-fixed-medium-r-normal--24-230-75-75-c-120-iso8859-1"
	     Type2  Multi Font Style.
	     "-sony-fixed-medium-r-normal--24-230-75-75-c-120-jisx0208.1983-0,-sony-fixed-medium-r-normal--24-230-75-75-c-120-jisx0201.1976-0,-sony-fixed-medium-r-normal--24-230-75-75-c-120-iso8859-1"
	     Type 3  Xft Font Name Style
	     "MS UI Gothic-16"
	     */

	    pat = XftXlfdParse(lf,0,1);
	    if (pat) {
		match = XftFontMatch(dpy, DefaultScreen(dpy), pat, &res);
		if (match) {
		    Bool is_core;
		    XftPatternGetBool(match,XFT_CORE,0,&is_core);
		    if (!is_core) {
			cout<<"XftFont "<< match <<endl;
			xftfont = XftFontOpenXlfd
			    (dpy, DefaultScreen(dpy), lfn.c_str());
		    }
		    XftPatternDestroy(match);
		}
		cout<<"XftFont Not Match"<<endl;
	    }
	    else {
		if (strchr(lf,',')==NULL) {/* Is FontList? */
		    cout<<"Try Xft Font"<<endl;
		    xftfont = XftFontOpenName(dpy, DefaultScreen(dpy), lf);
		}
		if (xftfont) cout<<"Open Success"<<lf<<endl;
		else cout<<"Open Fail Xft Font"<<endl;
	    }
#else // I18N

	    // stupid check for truetype font
	    // FIXME:
	    char* p = strstr(lfn.c_str(), "type-");
	    if (p || strstr(lfn.c_str(), "ttf-"))
	    {
		// the purpose of the folloing code is replace last digit
		// in the fontname:  *-iso8859-2  -> *-iso8859-*
		// it's necessary so the Xft renderer will work correctly
		p = strstr(lfn.c_str(), "iso8859-");
		if (p)
		{
		    p += 9;
		    *p = '*';
		    cout << "Modified font name for iso8859 support " << endl;
		}
#if 0
		xftfont = XftFontOpen (dpy, DefaultScreen(dpy),
				       XFT_FAMILY, XftTypeString, "Arial",
				       //XFT_ENCODING, XftTypeString, "iso10646",
				       XFT_ENCODING, XftTypeString, "iso8859-2",
				       XFT_SIZE, XftTypeDouble, 20.0,
				       0);
#endif // just testing family open
		xftfont = XftFontOpenXlfd(dpy, info.info.x11.window,
					  lfn.c_str());

		if (!xftfont)
		    cerr << "Failed to open Xft font" << endl;
		else
		    cout << "Using antialiased Xft renderer." << endl;

	    }
	    //xftfont = XftFontOpenName(d, info.info.x11.window, "verdana:pixelsize=30");
#endif // I18N
	}

	if (!xftfont)
	{
	    font = XLoadQueryFont(dpy, lf);
	    if (!font)
		cerr << "Failed to open X11 font" << endl;
	}
	else
	{
	    fm_height = xftfont->height;
	    //cout << "HEIGHT  " << fm_height << endl;
	    xftdraw = XftDrawCreate(dpy, win, DefaultVisual(dpy, DefaultScreen(dpy)),
				    DefaultColormap(dpy, DefaultScreen(dpy)));
	}
#else  // HAVE_XFT

	font = XLoadQueryFont(dpy, lf);
	if (!font)
	{
	    cerr << "Failed to open font" << endl;
	}
#endif
	if (font)
	{
#ifdef I18N
	    XFontSetExtents *extent;

	    extent = XExtentsOfFontSet(font);
	    /*
	     fm_height = extent->max_logical_extent.height * 4 / 5;
	     fm_height += extent->max_logical_extent.height / 5;
	     */
	    fm_height = extent->max_logical_extent.height;
#else
	    XSetFont(dpy, xgc, font->fid);
	    XSetFont(dpy, wmxgc, font->fid);
	    fm_height = font->max_bounds.ascent + font->max_bounds.descent;
	    //printf("Max bounds: %d, %d\n",
	    //       _font->max_bounds.ascent, _font->max_bounds.descent);
#endif // I18N
	}

	if (subtitles)
	{
	    m_sub = fm_height * SUBTITLE_SPACE;
	    dim_h = pic_h + m_sub;

	    screen = SDL_SetVideoMode(dim_w, dim_h, 0, fs);
	    //		printf("Max bounds: %d, %d\n",
	    //		    _font->max_bounds.ascent, _font->max_bounds.descent);
	    /*	    else
	     {
	     XGetGCValues(d, xgc, GCFont, &gcv);
	     _font=XQueryFont
	     */
	}
        Unlock();
        return 0;
    }

    virtual void GetModeLine(int& w, int& h)
    {
	// method to detect actuall screen resolution in pixels
	// e.g. when user switches resolution with Ctrl Alt '+'
	if (!dpy)
            return;
	int unused;
#ifdef HAVE_LIBXXF86VM
	XF86VidModeModeLine vidmode;
	XF86VidModeGetModeLine(dpy, DefaultScreen(dpy),
			       &unused, &vidmode);
	w = vidmode.hdisplay;
	h = vidmode.vdisplay;
#else
	// this will return just size of the whole screen
	// and I don't know other way then XF86VidMode
	// to read current screen size
	w = DisplayWidth(dpy, DefaultScreen(dpy));
	h = DisplayHeight(dpy, DefaultScreen(dpy));
#endif // HAVE_LIBXXF86VM
	//cout << "w " << w << "  h " << h << endl;
    }
    virtual void storeMousePosition()
    {
	// save original mouse position
#define RESTOREMOUSEPOS
#ifdef RESTOREMOUSEPOS

#if 0
        // dangerous - causes deadlocks - using MouseMotion event instead
	int root_x, root_y;
	int x, y;
	uint_t buttons;
	Window root, child;
	// this should lock SDL X queue
	// as suggested by Sam Lantinga <slouken@devolution.com>
	if (SDL_GetWMInfo(&info) > 0)
	{
	    XQueryPointer(dpy, info.info.x11.window,
			  &root, &child,
			  &root_x, &root_y, &mouse_x, &mouse_y, &buttons);
	    //cout << "Saving mouse position: rx: " << root_x << " ry: "
	    //    <<  root_y << " x: " << mouse_x << " y: " << mouse_y << endl;
	}
#endif // 0
        // using mouse motion event
        mouse_x = SDL_mouse_x;
        mouse_y = SDL_mouse_y;
#endif // RESTOREMOUSEPOS
    }
    virtual void restoreMousePosition()
    {
	//cout << "RESTORE" << endl;
	// recover original mouse position
        // called with LOCK
        SDL_WarpMouse(mouse_x, mouse_y); // this seems to be doing same thing
/*
#ifdef RESTOREMOUSEPOS
	SDL_SysWMinfo info;
	SDL_VERSION(&info.version);
	if (SDL_GetWMInfo(&info) > 0)
	{
#ifdef unix
	    info.info.x11.lock_func();
	    XWarpPointer(dpy, None, info.info.x11.window,
			 0, 0, 0, 0, mouse_x, mouse_y);
	    info.info.x11.unlock_func();
	    //cout << "Recovering  x: " << mouse_x << "  y: " << mouse_y << endl;
	}
#else
#error Need to implement these functions for other systems
#endif // unix
#endif // RESTOREMOUSEPOS
*/
    }
    virtual int Lock()
    {
	int  r = m_Mutex.Lock();
	//cerr << "lock " << getpid() <<  endl;
        if (!dga)
	    info.info.x11.lock_func();
        return r;
    }
    virtual int TryLock()
    {
	int r = m_Mutex.TryLock();
	if (r == 0)
	{
	    if (!dga)
		info.info.x11.lock_func();
	    //cerr << "trylock "  << getpid() << endl;
	}
	return r;
    }
    virtual int Unlock()
    {
	//cerr << "unlock " << getpid() << endl;
	if (!dga)
	    info.info.x11.unlock_func();
	return m_Mutex.Unlock();
    }

protected:
    virtual int doResize(int& new_w, int& new_h)
    {
        // allow original size
        if (new_w != m_w)
	    new_w = new_w & ~7;
	if (new_h != m_h)
	    // some resizing algorithms requires this for now
	    new_h = new_h & ~7;

	if (new_w == pic_w && new_h == pic_h)
	    return -1;
	if (!resizeEnabled)
	{
	    cout << "Resize is unavailable in RGB Direct mode!" << endl;
	    //printf("Disabled %dx%d  %dx%d\n", new_w, new_h, pic_w, pic_h);
	    new_w = m_w;
	    new_h = m_h;
	}

	if (new_h < m_h / 4 || new_w < m_w / 4)
	    return -1;

	pic_w = new_w;
	pic_h = new_h;
        // preferences
	cliprect.x = 0;
	cliprect.y = 0;
        cliprect.w = pic_w;
        cliprect.h = pic_h;
	dim_w = pic_w;
	dim_h = pic_h + (m_sub?fm_height*SUBTITLE_SPACE:0);
	//cout << "New image size " << pic_w << "   " << dim_h << "   " << m_sub << "  " << endl;

	//if (screen)
	//    SDL_FreeSurface(screen);

        // 0 is intentional -> use current screen depth and color model
	screen = SDL_SetVideoMode(dim_w, dim_h, 0, fs);
	return 0;
    }
private:
    // private implementation - called with lock
    void nvidiaCheck()
    {
	if (s_iTrickNvidia == -1)
	{
	    int lines;
	    //XExtensionVersion* ext = XGetExtensionVersion(dpy, "NVIDIA-GLX");
	    int n = 0;
	    char **extlist = XListExtensions(dpy, &n);

	    //printf ("number of extensions:    %d\n", n);
	    s_iTrickNvidia = 0;
	    if (extlist) {
		int i;
		int opcode, event, error;

		for (i = 0; i < n; i++)
		    if (strcmp("NVIDIA-GLX", extlist[i]) == 0)
		    {
                        s_iTrickNvidia = 2;
			cout << "Detected nVidia GLX driver - clearing "
			    << s_iTrickNvidia << " lowest lines..." << endl;
		    }
		/* do not free, Xlib can depend on contents being unaltered */
		/* XFreeExtensionList (extlist); */
	    }

	    if (getenv("AVIPLAY_NVIDIA_ENABLE"))
	    {
		s_iTrickNvidia = 2;
		cout << "nVidia - line clearing hack - forced On" << endl;
	    }
	    if (getenv("AVIPLAY_NVIDIA_DISABLE"))
	    {
		s_iTrickNvidia = 0;
                cout << "nVidia - line clearing hack - forced Off" << endl;
	    }
	}
    }
    int eventFilter(const SDL_Event* ev)
    {
	if (!g_pw)
	    return 0; // consumed anyway

	if (ev->type == SDL_KEYUP)
	{
	    //cout << "Key " << (int) ev->key.keysym.sym << " mod: "
	    //    << ev->key.keysym.mod << endl;
	    if (keyrepev.type); // disable autorepeat
	    keyrepev.type = 0; // disable autorepeat
	}
	else if (sdlKbdOn && ev->type == SDL_KEYDOWN && (keyrepev.type == 0))
	{
	    bool autorep = false;
	    switch (ev->key.keysym.sym)
	    {
	    case SDLK_RETURN:
		autorep = true;
		if (!(ev->key.keysym.mod & (KMOD_ALT | KMOD_META)))
		    break;
		autorep = false;
		// fall through - alt + enter
	    case SDLK_ESCAPE:
	    case SDLK_f:
		g_pw->PW_fullscreen();
		break;
	    case SDLK_m:
		g_pw->PW_maximize_func();
		break;
	    case SDLK_x:
		g_pw->PW_stop_func();
		break;
	    case SDLK_v:
		g_pw->PW_play_func();
		break;
	    case SDLK_q:
		g_pw->PW_quit_func();
		break;
	    case SDLK_p:
	    case SDLK_c:
	    case SDLK_SPACE:
		autorep = true;
		g_pw->PW_pause_func();
		break;
	    default:
		autorep = true;
		break;
	    }

	    // passing all keys
	    g_pw->PW_key_func(ev->key.keysym.sym, ev->key.keysym.mod);
	    if (autorep && !keyrepev.type)
	    {
		memcpy(&keyrepev, ev, sizeof(keyrepev));
		keyrepev.type = SDL_USEREVENT;
	    }
	}
	else if (ev->type == SDL_VIDEORESIZE)
	{
	    g_pw->PW_resize(ev->resize.w,
			    ev->resize.h - (m_sub?fm_height*SUBTITLE_SPACE:0));
	}
	else if (ev->type == SDL_MOUSEBUTTONDOWN)
	{
	    switch (ev->button.button)
	    {
	    case SDL_BUTTON_RIGHT:
		if (SDL_fullscreen)
		    g_pw->PW_fullscreen();
                // fix race for Pointer grab
		Lock();
		XUngrabPointer(info.info.x11.display, CurrentTime);
		XUngrabKeyboard(info.info.x11.display, CurrentTime);
		Unlock();
		g_pw->PW_menu_slot();
		break;
	    case SDL_BUTTON_MIDDLE:
		g_pw->PW_middle_button();
		break;
	    case SDL_BUTTON_LEFT:
		g_pw->PW_pause_func();
		break;
	    }
	}
#ifdef SDL_VIDEOEXPOSE
	else if (ev->type == SDL_VIDEOEXPOSE)
	{
	    // we are not receiving this event so far !!
	    cout << "SDL_EXPOSE arrived" << endl;
	}
#endif
	else if (ev->type == SDL_ACTIVEEVENT)
	{
	    //cout << "ACTIVE " << (int) ev->active.gain <<"  "<<(int) ev->active.state << endl;
	    if (ev->active.state == SDL_APPINPUTFOCUS)
	    {
		sdlKbdOn = (ev->active.gain) ? true : false;
		g_pw->PW_refresh();
	    }
	}
	else if (ev->type == SDL_MOUSEMOTION)
	{
	    // safer for reading mouse position
	    SDL_mouse_x = ev->motion.x;
	    SDL_mouse_y = ev->motion.y;
	    SDL_mouse_time = longcount();
	    //cout << "MOUSE MOVE " << ev->motion.x << "  " << ev->motion.y << endl;
	}
	else if (ev->type == SDL_QUIT)
	{
	    g_pw->PW_quit_func();
	}

	return 0; // event has been filtered
    }

    virtual void freeFont()
    {
	if (!dpy)
            return;
#ifdef HAVE_XFT
	if (xftfont)
	    XftFontClose(dpy, xftfont);
        xftfont = 0;
	if (xftdraw)
	    XftDrawDestroy(xftdraw);
	xftdraw = 0;
	if (xftcolor)
	    delete xftcolor;
        xftcolor = 0;
#endif
	if (font)
	    XFreeFont(dpy, font);
        font = 0;
    }
};

int FullscreenRenderer::s_iTrickNvidia = -1;



class YUVRenderer: public FullscreenRenderer
{
protected:
    SDL_Overlay* m_ov;
    fourcc_t m_fmt;
    avm::vector<SDL_Overlay*> ovlist;

    virtual int doResize(int& new_w, int& new_h)
    {
	//cout << "Resize " << new_w << "  " << new_h << "  " << pic_w << "  " << pic_h << endl;
	if (new_w < m_w/4 || new_h < m_h/4)
	    return -1;
#if 1
	pic_w = new_w;
	pic_h = new_h;
	int ndim_w = pic_w;
        int ndim_h = pic_h + m_sub;
#else
	// better way - user would probably expect this behavior
	// as we do not need to rescale image again for subtitles
	// however it would be harder to preserver Ratio in this case
        // FIMXE later....
	int ndim_w = new_w;
	int ndim_h = new_h;
	pic_w = dim_w;
        pic_h = dim_h - m_sub;
#endif
	if (ndim_w == dim_w && ndim_h == dim_h)
	    return -1;

	dim_w = ndim_w;
        dim_h = ndim_h;
	screen = SDL_SetVideoMode(dim_w, dim_h, bit_depth, fs);

        return 0;
    }
public:
    YUVRenderer(PlayerWidget* pw, Display* _dpy,
		int _width, int _height, fourcc_t yuvm_fmt, bool _subtitles = false)
    : FullscreenRenderer(pw, _dpy, _width, _height, _subtitles)
    {
	if (dga)
	    throw FATAL("Requested DGA driver - YUV not available!");

	Lock();

	printf("Creating YUV overlay %dx%d, fourcc 0x%x (%.4s)\n",
	       _width, _height, yuvm_fmt, (char*)&yuvm_fmt);
	fs &= ~SDL_DOUBLEBUF; // doesn't work with HW overlay
//	fs |= SDL_HWSURFACE;
        screen->flags = fs;
        screen = SDL_SetVideoMode(dim_w, dim_h, 0*bit_depth, fs);
#if 0
	const SDL_VideoInfo* vi = SDL_GetVideoInfo();
	printf("VideoInfo: %s  %s  %s  %s  %s -  %s  %s  %s  -  %s   vmem: %d    colorkey: 0x%x  alpha: 0x%x\n",
	       (vi->hw_available) ? "hw available" : "",
	       (vi->wm_available) ? "wm_available" : "",
	       (vi->blit_hw) ? "blit_hw" : "",
	       (vi->blit_hw_CC) ? "blit_hw_CC" : "",
	       (vi->blit_hw_A) ? "blit_hw_A" : "",
	       (vi->blit_sw) ? "blit_sw" : "",
	       (vi->blit_sw_CC) ? "blit_sw_CC" : "",
	       (vi->blit_sw_A) ? "blit_sw_A" : "",
	       (vi->blit_fill) ? "blit_fill" : "",
	       vi->video_mem,

	       vi->vfmt->colorkey,
	       vi->vfmt->alpha
	      );

	printf("ScreenInfo:  colorkey: 0x%x  alpha: 0x%x\n",
	       screen->format->colorkey,
	       screen->format->alpha);
#endif
#if 0
	char* tmp_array=new char[_width*_height*2];
	char* tmp_array2=new char[_width*_height*2];
	uint_t t1=localcount();
	memcpy(tmp_array2, tmp_array, _width*_height*2);
	uint_t t2=localcount();
	printf("Memory->memory copy: %f Mb/s\n",
		(_width*_height*3/2/1048576.)/((t2-t1)/freq/1000.));
	delete[] tmp_array2;
	SDL_Overlay* ovs[200];
	for(int i=0; i<200; i++)
	{
	    ovs[i]=SDL_CreateYUVOverlay(_width, _height, fccYV12, screen);
	    SDL_LockYUVOverlay(ovs[i]);
	    uint_t t1=localcount();
	    memcpy(ovs[i]->pixels[0], tmp_array, _width*_height*3/2);
	    uint_t t2=localcount();
	    SDL_UnlockYUVOverlay(ovs[i]);
	    printf("overlay %d ( %f Mb used ): %f Mb/s\n",
		i, i*_width*_height*3/2/1048576.,
		(_width*_height*3/2/1048576.)/((t2-t1)/freq/1000.));
	}
	for(int i=0; i<200; i++)
	    SDL_FreeYUVOverlay(ovs[i]);
	delete[] tmp_array;
#endif
	m_ov = SDL_CreateYUVOverlay(_width, _height, yuvm_fmt, screen);

#define _SDL_VER SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL)

	if ((!m_ov)
#if _SDL_VER>1104
    // SDL <=1.1.5 does not have hwm_overlay flag
	    || !m_ov->hw_overlay
#endif
	   )
	{
	    const char* errmsg = (!m_ov) ? "Failed to create overlay" :
		"No hardware YUV acceleration detected!";

	    cout << errmsg << endl;
#if _SDL_VER>1106
	    if  (m_ov && allow_sw_yuv)
	    {
		cout << "*** Using SDL software YUV emulation ***" << endl
                    << "  Usually most codecs supports RGB modes - so you may" << endl
		    << "  achieve better performance with disabled YUV flag"
		    << endl;
                // do not bother user with Xlib warning messages
		avm_setenv("SDL_VIDEO_YUV_HWACCEL", "0", 1);
	    }
            else
#endif
	    {
		if (m_ov)
		{
		    cout << "Your SDL library is too old and doesn't support software YUV emulation - upgrade SDL package!" << endl;
		    SDL_FreeYUVOverlay(m_ov);
		}
		Unlock();
		throw FATAL(errmsg);
	    }
	}
	m_fmt = yuvm_fmt;
	ovlist.push_back(m_ov);
	//cout << "--- MEMORY ADDR " << (void*) m_ov->pixels[0] << "  " << m_ov->hwm_overlay << endl;
	Unlock();
    }
    ~YUVRenderer()
    {
        Lock();
	m_ov = 0;
	while (ovlist.size() > 0)
	{
	    SDL_FreeYUVOverlay(ovlist.back());
            ovlist.pop_back();
	}
	Unlock();
    }
    virtual CImage* GetData(int idx)
    {
	while ((unsigned)idx >= ovlist.size())
	{
	    SDL_Overlay* o = SDL_CreateYUVOverlay(m_w, m_h, m_fmt, screen);
	    if (!o)
		return 0;

	    ovlist.push_back(o);
	}
	BitmapInfo bi;
        bi.biWidth = m_w;
	bi.biHeight = m_h;
        bi.biPlanes = 1;
	bi.SetSpace(m_fmt);
	//bi.Print();
	CImage* ci = new CImage(&bi, (const uint8_t**) ovlist[idx]->pixels, false);
	ci->SetUserData(&(*ovlist[idx]));
	return ci;
    }
    virtual int Draw(const CImage* data)
    {
	if (!data)
	    return -1;

	if (TryLock() != 0)
	    return -1;

	//cout << "DRAWYUV " << (void*)data->GetUserData() << endl;
	m_lLastDrawStamp = longcount();

	data->AddRef();
	if (image)
	    image->Release();
	image = data;

	switch(m_fmt)
	{
#if _SDL_VER > 1104
	case fccYV12:
	case fccIYUV:
	case fccI420:
	    //cout << "YV12" << endl;
	    if (!image->GetUserData())
	    {
		if (data->Data())
		{
		    SDL_LockYUVOverlay(m_ov);
		    avm_memory_lock();
		    memcpy(m_ov->pixels[0], data->Data(), data->Pixels());
		    memcpy(m_ov->pixels[2], data->Data() + data->Pixels(),
			   data->Pixels() / 4);
		    memcpy(m_ov->pixels[1], data->Data() + data->Pixels() * 5 / 4,
			   data->Pixels() / 4);
		    avm_memory_unlock();
		    SDL_UnlockYUVOverlay(m_ov);
		}
		else
                    cout << "Image data" << endl;
	    }
	    break;
#else
	case fccYV12:
	case fccIYUV:
	case fccI420:
#endif
	case fccYUY2:
	case fccUYVY:
	case fccYVYU:
	    //cout << "YUY2" << endl;
	    if (!image->GetUserData())
	    {
		//printf("FMT %.4s\n", (char*) &m_fmt);
		SDL_LockYUVOverlay(m_ov);
		avm_memory_lock();
                if (data->Data())
		    memcpy(m_ov->pixels[0], data->Data(), 2 * data->Pixels());
		else
		    printf("NULL memory: %p   %p  %d\n",
			   m_ov->pixels[0], data->Data(), 2 * data->Pixels());
		avm_memory_unlock();
		SDL_UnlockYUVOverlay(m_ov);
	    }
	    //else cout << "no memcpy" << endl;
	    break;
	}

	Unlock();

	return 0;
    }
    virtual int Sync()
    {
        //cout << "SYNC" << endl;
	if (TryLock() != 0)
	    return -1;

	SDL_Rect rect;
	rect.x = rect.y = 0;
	rect.w = pic_w;
	rect.h = pic_h;

	// as the following operation might lock itself in XServer
	// (when someone is resizing windows
	// we can't hold any memory lock here!
	SDL_Overlay* o = (SDL_Overlay*) image->GetUserData();
	if (!o)
	    o = m_ov;

        // no surface lock as SDL documentation recomends!
	if (s_iTrickNvidia > 0)
	{
	    GC lxgc = (fs & SDL_FULLSCREEN) ? wmxgc : xgc;
	    XSetForeground(dpy, lxgc, 0x0);
	    XFillRectangle(dpy, info.info.x11.window,
			   lxgc, 0, pic_h - s_iTrickNvidia,
			   dim_w, s_iTrickNvidia);
	}
	SDL_DisplayYUVOverlay(o, &rect);
	Unlock();

        econd.Broadcast();
	return 0;
    }
};

VideoRenderer* CreateFullscreenRenderer(PlayerWidget* pw, void* dpy,
					int width, int height, bool sub)
{
    return new FullscreenRenderer(pw, (Display*)dpy, width, height, sub);
}

VideoRenderer* CreateYUVRenderer(PlayerWidget* pw, void* dpy,
				 int width, int height,
				 fourcc_t yuv_fmt, bool sub)
{
    //return new XvYUVRenderer(pw, dpy, width, height, yuv_fmt, sub);
    return new YUVRenderer(pw, (Display*)dpy, width, height, yuv_fmt, sub);
}

#else

VideoRenderer* CreateFullscreenRenderer(PlayerWidget* pw, void* dpy,
					int width, int height, bool sub)
{
    cout << "library compiled without SDL support - sorry on rendering" << endl;
    return 0;
}

VideoRenderer* CreateYUVRenderer(PlayerWidget* pw, void* dpy,
				 int width, int height,
				 fourcc_t yuv_fmt, bool sub)
{
    cout << "library compiled without SDL support - sorry on rendering" << endl;
    return 0;
}

#endif


#else /*WITHOUT_X*/
int GetPhysicalDepth(void* dpy)
{
    return 0;
}

#endif  /*WITHOUT_X*/
/*
vim: tabstop=8
*/
