#include "utils.h"

#include <sys/time.h>
#include <time.h>
#include <stdlib.h> // getenv, malloc,...
#include <string.h>  // strcat,...
#include <pthread.h>

// mutex for locking parallelism in large memory transfers
static pthread_mutex_t memorymut;

uint_t (*localcount)(void);
int64_t (*longcount)(void);

/*
 * A thread-safe usec sleep
 * (Note: on solaris, usleep is not thread-safe)
 */
#ifndef WIN32
int avm_usleep(unsigned long delay)
{
#if   HAVE_NANOSLEEP
    struct timespec tsp;
    tsp.tv_sec  =  delay / 1000000;
    tsp.tv_nsec = (delay % 1000000) * 1000;
    return nanosleep(&tsp, NULL);
#else
    return usleep(delay);
#endif
}
#endif

void avm_memory_lock(void)
{
    pthread_mutex_lock(&memorymut);
}

void avm_memory_unlock(void)
{
    pthread_mutex_unlock(&memorymut);
}

void avm_memory_lock_init(void)
{
    pthread_mutex_init(&memorymut, NULL);
}

/*
 * Solaris (maybe other operating systems, too) does not have avm_setenv(),
 * and avm_unsetenv() in libc, provide our own implementation.
 */
int avm_setenv(const char *name, const char *value, int overwrite)
{
#if HAVE_SETENV
    return setenv(name, value, overwrite);
#else
    char *env;

    if (getenv(name) != NULL && !overwrite)
	return 0;

    env = malloc(strlen(name) + strlen(value) + 2);
    strcpy(env, name);
    strcat(env, "=");
    strcat(env, value);
    return putenv(env);
#endif
}

void avm_unsetenv(const char *name)
{
#if HAVE_UNSETENV
    unsetenv(name);
#else
    extern char **environ;
    char **ep;
    int len = strlen(name);
    for (ep = environ; *ep; ep++)
    {
	if (strncmp(*ep, name, len) == 0 && (*ep)[len] == '=')
	{
	    while ((ep[0] = ep[1]) != NULL)
		ep++;
	    break;
	}
    }
#endif
}
