#include "infotypes.h"
#include "utils.h"
#include "cpuinfo.h"
#include "creators.h"

#include <string.h>
#include <stdlib.h> //atof
#ifndef WIN32
#include <unistd.h>
#include <sys/time.h>
#else
#include <windows.h>
#endif
#include <time.h>
#ifdef	HAVE_KSTAT
#include <kstat.h>
#endif
#include <ctype.h>
#include <stdio.h>


BaseInfo::BaseInfo()
{
}

BaseInfo::BaseInfo(const char* info, const char* _about)
    : name(info), about(_about)
{
}

BaseInfo::~BaseInfo()
{
}

const char* BaseInfo::GetName() const
{
    return name.c_str();
}

const char* BaseInfo::GetAbout() const
{
    return about.c_str();
}

AttributeInfo::AttributeInfo()
{
}

AttributeInfo::AttributeInfo(const char* n, const char* a, const char** o)
    : BaseInfo(n, a), kind(Select)
{
//    cout<<"AttributeInfo("<<n<<") at "<<(int)this<<endl;
    while(*o)
    {
	options.push_back(*o);
	o++;
    }
}

AttributeInfo::AttributeInfo(const char* n, const char* a, Kind k, int minval, int maxval)
    : BaseInfo(n, a), kind(k), i_min(minval), i_max(maxval)
{
//    cout<<"AttributeInfo("<<n<<") at "<<(int)this<<endl;
}

AttributeInfo::AttributeInfo(const char* n)
    : BaseInfo(n, ""), kind(Integer), i_min(0), i_max(-1)
{
//    cout<<"AttributeInfo("<<n<<") at "<<(int)this<<endl;
}

AttributeInfo::~AttributeInfo()
{
//    cout<<"~AttributeInfo("<<name<<") at "<<(int)this<<endl;
}

CodecInfo::CodecInfo() :handle(0)
{
}

CodecInfo::CodecInfo(const fourcc_t* array, const char* info, const char* path, const char* a,
		     Kind _kind, const char* pn, Media _media, Direction _direction, const GUID* id,
		     const avm::vector<AttributeInfo>& ei,
		     const avm::vector<AttributeInfo>& di)
    : BaseInfo(info, a), fourcc(array[0]), kind(_kind), media(_media),
    direction(_direction), decoder_info(di), encoder_info(ei),
    privatename(pn), handle(0), dll(path)
{
//    cout<<"CodecInfo("<<info<<") at "<<(int)this<<endl;
    if (id)
	guid = *id;

    if (!*array) // uncompressed codec
    {
	fourcc_array.push_back(0);
	array++;
    }

    while (*array)
    {
	fourcc_array.push_back(*array);
	array++;
    }
}

CodecInfo::CodecInfo(const CodecInfo& ci) :handle(0)
{
    operator=(ci);
}

CodecInfo::~CodecInfo()
{
//    cout<<"~CodecInfo("<<text<<") at "<<(int)this<<endl;
}

CodecInfo& CodecInfo::operator=(const CodecInfo& ci)
{
    name = ci.name;
    about = ci.about;

    handle = ci.handle;
    fourcc = ci.fourcc;
    fourcc_array = ci.fourcc_array;
    kind = ci.kind;
    media = ci.media;
    direction = ci.direction;
    decoder_info = ci.decoder_info;
    encoder_info = ci.encoder_info;
    modulename = ci.modulename;
    guid = ci.guid;
    dll = ci.dll;
    privatename = ci.privatename;
    return *this;
}

const CodecInfo* CodecInfo::match(fourcc_t codec, CodecInfo::Media cimedia, const CodecInfo* start, CodecInfo::Direction cdir)
{
    if (video_codecs.size() == 0 && audio_codecs.size() == 0)
    {
	BITMAPINFOHEADER bi;
        bi.biCompression = 0xffffffff;
	// just to fill video_codecs list
	Creators::CreateVideoDecoder(bi, 0, 0);
    }

    avm::vector<CodecInfo>& c =
	(cimedia == Video) ? video_codecs : audio_codecs;

    for (unsigned i = 0; i < c.size(); i++)
    {
	if (start)
	{
	    if (&c[i] != start)
		continue;
	    start = 0;
	}

	CodecInfo& ci = c[i];	
	
	if(((int)ci.direction & (int)cdir) != (int)cdir)
	    continue;
	    
	for (unsigned j = 0; j < ci.fourcc_array.size(); j++)
	    if (codec == ci.fourcc_array[j])
		return &ci;
    }
    return 0;
}

const CodecInfo* CodecInfo::match(CodecInfo::Media cimedia, const char* pname)
{
    if (video_codecs.size() == 0 && audio_codecs.size() == 0)
    {
	BITMAPINFOHEADER bi;
        bi.biCompression = 0xffffffff;
	// just to fill video_codecs list
	Creators::CreateVideoDecoder(bi, 0, 0);
    }

    avm::vector<CodecInfo>& c = (cimedia == Video ) ? video_codecs : audio_codecs;
    unsigned i;
    for (i = 0; i < c.size(); i++)
    {
	CodecInfo& ci = c[i];	
	
	if(ci.privatename == pname)
	    return &ci;
    }

    return 0;
}

#ifdef WIN32
#define rdtsc __asm _emit 0fh __asm _emit 031h
#define cpuid __asm _emit 0fh __asm _emit 0a2h

#define strncasecmp strnicmp
#endif

static void do_cpuid(uint_t *regs, int op)
{

#ifndef WIN32
#ifdef i386
	unsigned int ax;
	ax=op;
	__asm__ __volatile__(
	"pushl %%ebx; pushl %%ecx; pushl %%edx;"
	".byte  0x0f, 0xa2;"
	"movl   %%eax, (%2);"
	"movl   %%ebx, 4(%2);"
	"movl   %%ecx, 8(%2);"
	"movl   %%edx, 12(%2);"
	"popl %%edx; popl %%ecx; popl %%ebx;"
	: "=a" (ax)
	:  "0" (ax), "S" (regs)
	);
#else
	*regs = 0;
#endif
#else
	__asm
	{
		push eax
		push ebx
		push ecx
		push edx
		push esi
		mov esi, regs
		mov eax, op
		cpuid
		mov dword ptr [esi], eax
		add esi, 4
		mov dword ptr [esi], ebx
		add esi, 4
		mov dword ptr [esi], ecx
		add esi, 4
		mov dword ptr [esi], edx
		pop esi
		pop edx
		pop ecx
		pop ebx
		pop eax
	}
#endif
	return;
}

static uint_t localcount_tsc()
{
    int a;
#ifndef WIN32
#ifdef i386
    __asm__ __volatile__("rdtsc\n\t" : "=a"(a) : : "edx");
#else
    a = 0;
#endif
#else
    __asm
    {
	rdtsc
    	mov a, eax
    }
#endif
    return a;
}

static int64_t longcount_tsc()
{
    unsigned long i;
    unsigned long j;
#ifndef WIN32
#ifdef i386
    __asm__ __volatile__("rdtsc\n\t":"=a"(i), "=d"(j):);
#else
    i = j = 0;
#endif
#else
    __asm
    {
	rdtsc
	mov i, eax
	mov j, edx
    }
#endif
    return ((int64_t)j<<32)+(int64_t)i;
}

static uint_t localcount_notsc()
{
#ifndef WIN32
    struct timeval tv;
    unsigned long limit=~0U;
    limit/=1000000;
    gettimeofday(&tv, 0);
    return limit*tv.tv_usec;
#else
    return time(NULL);
#endif
}

static int64_t longcount_notsc()
{
#ifndef WIN32
    struct timeval tv;
    unsigned limit=~0U;
    limit/=1000000;
    gettimeofday(&tv, 0);
    uint64_t result;
    result=tv.tv_sec;
    result<<=32;
    result+=limit*tv.tv_usec;
    return result;
#else
	return time(0);
#endif
}

static double old_freq()
{
#if HAVE_KSTAT
    /*
     * try to extact the CPU speed from the Solaris kernel's kstat data
     */
    kstat_ctl_t   *kc;
    kstat_t       *ksp;
    kstat_named_t *kdata;
    int            mhz = 0;

    kc = kstat_open();
    if (kc != NULL)
    {
	ksp = kstat_lookup(kc, "cpu_info", 0, "cpu_info0");

	/* kstat found and name/value pairs? */
	if (ksp != NULL && ksp->ks_type == KSTAT_TYPE_NAMED)
	{
	    /* read the kstat data from the kernel */
	    if (kstat_read(kc, ksp, NULL) != -1)
	    {
		/*
		 * lookup desired "clock_MHz" entry, check the expected
		 * data type
		 */
		kdata = (kstat_named_t *)kstat_data_lookup(ksp, "clock_MHz");
		if (kdata != NULL && kdata->data_type == KSTAT_DATA_INT32)
		    mhz = kdata->value.i32;
	    }
	}
	kstat_close(kc);
    }

    if (mhz > 0)
	return mhz * 1000.;
#endif	/* HAVE_KSTAT */

    int i=time(NULL);
    int64_t x,y;
    while(i==time(NULL));
    x=longcount();
    i++;
    while(i==time(NULL));
    y=longcount();
    return (double)(y-x)/1000.;
}

CPU_Info::CPU_Info() : freq(-1), have_tsc(0), have_mmx(0), have_mmxext(0), have_sse(0)
{
    char line[200];
    char model[200]="unknown";
    char flags[500]="";
    char* s, *value;
    bool is_AMD=false;
    avm_memory_lock_init();
    FILE *f = fopen("/proc/cpuinfo", "r");
    if (!f)
    {
	uint_t regs[4];
	do_cpuid(regs, 0);
	is_AMD = ((regs[1] == 0x68747541) && 
	         (regs[2] == 0x444d4163) && 
    	         (regs[3] == 0x69746e65));

	do_cpuid(regs, 1);
	have_tsc=regs[3] & 0x00000010;
	have_mmx=regs[3] & 0x00800000;
	have_mmxext=regs[3] & 0x02000000;
	have_sse=regs[3] & 0x02000000;
	do_cpuid(regs, 0x80000000);
	if (regs[0] >= 0x80000001) 
	{
	    do_cpuid(regs, 0x80000001);

	    if(is_AMD && (regs[3] & 0x00400000))
		have_mmxext=1;
	}

	if(have_tsc)
	{
	    longcount=longcount_tsc;
	    localcount=localcount_tsc;
	}
	else
	{
	    longcount=longcount_notsc;
	    localcount=localcount_notsc;
	}

	freq=old_freq();
	return;
    }

    while (fgets(line,200,f) != NULL)
    {

	/* NOTE: the ':' is the only character we can rely on */
        value = strchr(line,':');
	if (!value)
	    continue;
	/* terminate the valuename */
	*value++ = '\0';
	/* skip any leading spaces */
	while (*value == ' ')
	    value++;
        s = strchr(value, '\n');
	if (s)
	    *s = '\0';

	if (!strncasecmp(line, "cpu MHz",strlen("cpu MHz")))
	{
	    freq=atof(value);
	    freq*=1000;
	}
	if (!strncasecmp(line, "model name", strlen("model name")))
	    strncpy(model, value, sizeof(model)-1);
	if (!strncasecmp(line, "flags", strlen("flags")) ||
	    !strncasecmp(line, "features", strlen("features")))
	    strncpy(flags, value, sizeof(flags)-1);
	continue;
    }
    fclose(f);

    printf("Available CPU flags: %s\n", flags);
    have_tsc = (strstr(flags, "tsc") != 0);
    have_mmx = (strstr(flags, "mmx") != 0);
    have_sse = (strstr(flags, "sse") != 0);
    have_mmxext = (strstr(flags, "mmxext") != 0) | have_sse;

    if (have_tsc)
    {
	longcount=longcount_tsc;
	localcount=localcount_tsc;
    }
    else
    {
	longcount=longcount_notsc;
	localcount=localcount_notsc;
    }

    if (freq < 0)
	freq = old_freq();
    if (have_tsc)
	printf("%f MHz %s processor detected\n", freq/1000., model);
}


AVM_BEGIN_NAMESPACE

string::string()
{
    str = new char[1];
    str[0] = 0;
}

string::string(char s)
{
    str = new char[2];
    str[0] = s;
    str[1] = 0;
}

string::string(const char* s, uint_t len)
{
    uint_t slen = (s) ? strlen(s) : 0;
    if (len == 0 || len > slen)
	len = slen;
    str = new char[len + 1];
    if (s)
	strncpy(str, s, len);
    str[len] = 0;
}

string::string(const string& s, uint_t len)
{
    if (len == 0)
	len = s.size();

    str = new char[len + 1];
    strncpy(str, s.str, len);
    str[len] = 0;
}

string::~string()
{
    delete[] str;
}

uint_t string::size() const
{
    char* p = str;
    while (*p)
	p++;
    return p - str;
}

bool string::operator==(const char* s) const
{
    if(!s)
	s="";
    return !strcmp(str, s);
}

bool string::operator==(const string& s) const
{
    return !strcmp(str, s.str);
}

string& string::operator=(const string& s)
{
    if (this == &s)
	return *this;
    uint_t sz = s.size();
    delete[] str;
    str = new char[sz + 1];
    strcpy(str, s.str);
    str[sz] = 0;
    return *this;
}

bool string::operator<(const string& s) const
{
    return (strcmp(str, s.str)<0);
}

string& string::operator=(const char* s)
{
    if (str == s)
	return *this;
    if(!s)
	s="";
    uint_t sz = strlen(s);
    delete[] str;
    str = new char[sz + 1];
    strcpy(str, s);
    str[sz] = 0;
    return *this;
}

string& string::operator+=(const string& s)
{
    uint_t s1 = size();
    uint_t s2 = s.size();
    if (s2 > 0)
    {
        s2 += s1;
	char* p = new char[s2 + 1];
	strcpy(p, str);
	strcpy(p + s1, s.str);
	p[s2] = 0;
	delete[] str;
        str = p;
    }
    return *this;
}

string& string::operator+=(const char* s)
{
    if(!s)
	s="";
    uint_t s1 = size();
    uint_t s2 = strlen(s);
    if (s2 > 0)
    {
        s2 += s1;
	char* p = new char[s2 + 1];
	strcpy(p, str);
	strcpy(p + s1, s);
	p[s2] = 0;
	delete[] str;
        str = p;
    }
    return *this;
}

string& string::erase(uint_t from, uint_t to)
{
    char* p = str + from;
    if (to != npos && to > 0)
    {
        // add check for size() ???
	char* i = p + to;
	while (*i)
	    *p++ = *i++;
    }
    *p = 0;
    return *this;
}

void string::insert(uint_t pos, const string& s)
{
    uint_t l = s.size();
    uint_t k = size();
    char* p = new char[k + l + 1];
    strcpy(p, str);
    strcpy(p + pos, s.str);
    strcpy(p + pos + l, str + pos);
    delete[] str;
    str = p;
    str[k + l] = 0;
}

string::size_type string::find(const string& s, size_type startpos) const
{
    char* p = strstr(str + startpos, s.str);
    return (p) ? p - str : npos;
}

string::size_type string::find(char c) const
{
    const char* p=strchr(str, c); return p ? (p-str) : npos;
}

string::size_type string::rfind(char c) const
{
    const char* p=strrchr(str, c); return p ? (p-str) : npos;
}

AVM_END_NAMESPACE
