/********************************************************

	AviWriteStream/AviWriteFile class implementation
	Copyright 2000 Eugene Kuznetsov  (divx@euro.ru)

*********************************************************/

#define _LARGEFILE64_SOURCE
#include "AviWrite.h"

#include <fcntl.h>
#include <cstdio>

#if defined(__FreeBSD__) || defined(__NetBSD__)
#define lseek64 lseek
#define O_LARGEFILE 0
#endif
Registrator Registrator::Reg;

AviWriteFile::AviWriteFile(const char* name, int flags, int mask)
    :m_fd(0), m_filename(name)
{
    try
    {
        m_status=0;
	m_fd = new FileBuffer(name, O_WRONLY | O_CREAT | O_TRUNC | O_LARGEFILE, mask);

	const int junk_size=0x800;
        char* junk = new char[junk_size];
	memset(junk, 0, junk_size);
        m_fd->write(junk, junk_size);
	delete[] junk;
        m_status = 1;

	memset(&m_header, 0, sizeof(m_header));
        m_header.dwFlags = flags;
	Registrator::Reg.push_back(this);
    }
    catch (FatalError&)
    {
	delete m_fd;
	throw;
    }
}

// temp=(X);
// write(m_fd, &temp, 4);
AviWriteFile::~AviWriteFile()
{
    if (!m_status)
	return;

    try
    {
	uint_t videoendpos = m_fd->lseek(0, SEEK_CUR);
	if (videoendpos & 1)
	{
	    int junk = -1;
	    m_fd->write(&junk, 1);
	    videoendpos++;
	}
	WriteHeaders();

	write_le32(ckidAVINEWINDEX);
	write_le32(m_index.size() * sizeof(AVIINDEXENTRY));
	m_fd->write(&m_index[0], m_index.size()*sizeof(AVIINDEXENTRY));

	videoendpos=m_fd->lseek(0, SEEK_END);
	m_fd->lseek(4, SEEK_SET);
	write_le32(videoendpos-8);
	_destruct();
    }
    catch(...)
    {
	_destruct();
	throw;
    }
}

void AviWriteFile::_destruct()
{
    if (m_fd)
	delete m_fd;
    m_fd = 0;

    for (unsigned i=0; i<m_streams.size(); i++)
        delete m_streams[i];
    m_streams.clear();
    m_index.clear();
    m_status=0;

    if (!Registrator::Reg._destroying)
	Registrator::Reg.remove(this);
}

IAviWriteStream* AviWriteFile::AddStream(AviStream::StreamType type,
					 const void* format,
					 uint_t format_size,
					 fourcc_t handler, int frame_rate,
					 uint_t samplesize, int quality,
					 int flags)
{
    int ckid = MAKEAVICKID((type==AviWriteStream::Video)?cktypeDIBcompressed:cktypeWAVEbytes, m_streams.size());

    AviWriteStream* result=new AviWriteStream(this, ckid, type,
					      format, format_size,
					      handler, frame_rate,
					      samplesize,
					      quality, flags);
    m_streams.push_back(result);
    return result;
}

IAviVideoWriteStream* AviWriteFile::AddVideoStream(const CodecInfo& ci,
						   BITMAPINFOHEADER* srchdr,
						   int frame_rate, int flags)
{
    int ckid = MAKEAVICKID(cktypeDIBcompressed, m_streams.size());

    AviVideoWriteStream* r = new AviVideoWriteStream(this, ckid, ci, srchdr,
						     frame_rate, flags);
    if (!r)
    	throw FATAL("Failed to create new video stream");

    // result->Create(fourcc, srchdr);
    m_streams.push_back(r);
    return r;
}

IAviVideoWriteStream* AviWriteFile::AddVideoStream(fourcc_t fourcc,
						   BITMAPINFOHEADER* srchdr,
						   int frame_rate, int flags)
{
    const CodecInfo* pci = CodecInfo::match(fourcc, CodecInfo::Video, 0, CodecInfo::Encode);
    if (!pci)
	throw FATAL("No known video codecs for this fourcc");
    return AddVideoStream(*pci, srchdr, frame_rate, flags);
}

IAviAudioWriteStream* AviWriteFile::AddAudioStream(const CodecInfo& ci,
						   WAVEFORMATEX* fmt,
						   int bitrate, int flags)
{
    int ckid = MAKEAVICKID(cktypeWAVEbytes, m_streams.size());

    AviAudioWriteStream* r;
    r = new AviAudioWriteStream(this, ckid, ci, fmt, bitrate, flags);

    if (!r)
    	throw FATAL("Failed to create new audio stream");

    m_streams.push_back(r);
    return r;
}

IAviAudioWriteStream* AviWriteFile::AddAudioStream(fourcc_t fourcc,
						   WAVEFORMATEX* fmt,
						   int bitrate, int flags)
{
    const CodecInfo* pci = CodecInfo::match(fourcc, CodecInfo::Audio,
					    0, CodecInfo::Encode);
    if (!pci)
	throw FATAL("No known audio codecs for this fourcc");
    return AddAudioStream(*pci, fmt, bitrate, flags);
}



void AviWriteFile::AddChunk(uint_t offset, uint_t size, uint_t id, int flags)
{
//    m_index=(AVIINDEXENTRY*)realloc(m_index, (m_indsize+1)*sizeof(AVIINDEXENTRY));
    AVIINDEXENTRY entry;
    entry.ckid=id;
    entry.dwFlags=flags;
    entry.dwChunkOffset=offset;
    entry.dwChunkLength=size;
    try
    {
	m_index.push_back(entry);
    }
    catch(...)
    {
	throw FATAL("Chunk table reallocation failure");
    }
    if (m_index.size()%1000==0)
	WriteHeaders();
}

int64_t AviWriteFile::GetFileSize() const
{
    return m_fd->lseek(0, SEEK_CUR);
}

AviWriteStream::AviWriteStream(AviWriteFile* file, int ckid,
			       enum AviStream::StreamType type,
			       const void* format, uint_t format_size,
			       fourcc_t handler, int frame_rate,
			       uint_t samplesize,
			       int quality, int flags)
    :m_file(file), m_ckid(ckid)
{
    if (format == 0)
        throw FATAL("Bad format");
    if (type == Other)
        throw FATAL("Unsupported stream type");

    m_type = type;

    m_fd = file->m_fd;
    m_header.dwLength = 0;

    m_format = new char[format_size];
    m_forsize = format_size;
    memcpy(m_format, format, format_size);

    memset(&m_header, 0, sizeof(m_header));
    m_header.fccType = ((m_type == Video)?streamtypeVIDEO:streamtypeAUDIO);
    m_header.fccHandler = handler;
    m_header.dwFlags = flags;
    if (m_type == Video)
    {
	const BITMAPINFOHEADER* bh = (const BITMAPINFOHEADER*) m_format;
	m_header.rcFrame.right = bh->biWidth;
	m_header.rcFrame.bottom = bh->biHeight;
	m_header.dwRate = 1000000;
        m_header.dwScale = frame_rate;
    }
    else
    {
        m_header.dwRate = frame_rate;
        m_header.dwScale = samplesize;
    }
    m_header.dwSampleSize = samplesize;
    m_header.dwQuality = quality;
    m_header.dwFlags = flags;
}

AviWriteStream::AviWriteStream(AviWriteFile* file, int ckid,
			       enum AviStream::StreamType type,
			       fourcc_t handler, int frame_rate, int flags)
    :m_file(file), m_format(0), m_ckid(ckid)
{
    m_fd = file->m_fd;

    memset(&m_header, 0, sizeof(m_header));

    m_header.fccHandler = handler;
    m_header.dwFlags = flags;

    switch(type)
    {
    case Other:
	throw FATAL("Bad type");
    case Video:
	m_type=Video;
        m_header.fccType = streamtypeVIDEO;
	m_header.dwRate = 1000000;
	m_header.dwScale = frame_rate;
	break;
    case Audio:
	m_type=Audio;
        m_header.fccType = streamtypeAUDIO;
	m_header.dwRate = frame_rate;
	m_header.dwScale = 1;
	break;
    }
}

HRESULT AviWriteStream::AddChunk(const void* chunk, uint_t size, int flags)
{
    if (chunk == 0 && size)
    {
	printf("Invalid argument to AviWriteStream::AddChunk()\n");
	return -1;
    }
    uint_t offset = m_fd->lseek(0, SEEK_CUR);
    if (offset > 0x7f000000)
	return -1;
    m_fd->write(&m_ckid, 4);
    m_fd->write(&size, 4);
    if (chunk)
    {
	m_fd->write(chunk, size);
	if (size & 1)
	    m_fd->write(chunk, 1);
    }
//    fsync(m_fd);
    m_file->AddChunk(offset-0x7FC, size, m_ckid, flags);
    if (m_header.dwSampleSize==0)
    	m_header.dwLength++;
    else
	m_header.dwLength += size/m_header.dwSampleSize;
    return 0;
}

HRESULT AviWriteFile::Reserve(uint_t size)
{
    return -1;
}

HRESULT AviWriteFile::WriteChunk(fourcc_t fourcc, void* data, uint_t size)
{
    return -1;
}

AviWriteStream::~AviWriteStream()
{
    if (m_format)
	delete m_format;
}

IAviWriteStream::~IAviWriteStream(){}
IAviVideoWriteStream::~IAviVideoWriteStream(){}
IAviWriteFile::~IAviWriteFile(){}

IAviWriteFile* CreateIAviWriteFile(const char* name, int flags, int mask)
{
    return new AviWriteFile(name, flags, mask);
}

void AviWriteFile::WriteHeaders()
{
    //cerr<<"**** Writing headers ****"<<endl;
    if (m_status == 0)
	return;
    m_header.dwFlags |= AVIF_HASINDEX | AVIF_TRUSTCKTYPE;
    m_header.dwPaddingGranularity = 0;
    m_header.dwTotalFrames = 0;
    for (unsigned i = 0; i < m_streams.size(); i++)
    {
	if (m_streams[i]->GetType() == AviWriteStream::Video)
	{
	    m_header.dwTotalFrames = m_streams[i]->GetLength();
	    m_header.dwMicroSecPerFrame = m_streams[i]->m_header.dwScale;
	    m_header.dwWidth = m_streams[i]->m_header.rcFrame.right;
	    m_header.dwHeight = labs(m_streams[i]->m_header.rcFrame.bottom);
	    break;
	}
    }
    if (m_header.dwTotalFrames == 0)
	if (m_streams.size())
	{
	    m_header.dwTotalFrames = m_streams[0]->GetLength();
	    m_header.dwWidth = m_header.dwHeight = 0;
	}
    m_header.dwStreams = m_streams.size();
    uint_t endpos = m_fd->lseek(0, SEEK_END);
    m_fd->lseek(0, SEEK_SET);
    write_le32(FOURCC_RIFF);
    write_le32(endpos-8);
    write_le32(formtypeAVI);// Here goes chunk with all headers
    write_le32(FOURCC_LIST);
    int hdr_pos = 0x10;
    write_le32(0);//here header chunk size hdr_size will be
    int hdr_size = 12 + sizeof(MainAVIHeader);

    // Write MainAVIHeader
    write_le32(listtypeAVIHEADER);
    write_le32(ckidAVIMAINHDR);
    write_le32(sizeof(MainAVIHeader));

    write_le32(m_header.dwMicroSecPerFrame);
    write_le32(m_header.dwMaxBytesPerSec);
    write_le32(m_header.dwPaddingGranularity);
    write_le32(m_header.dwFlags);
    write_le32(m_header.dwTotalFrames);
    write_le32(m_header.dwInitialFrames);
    write_le32(m_header.dwStreams);
    write_le32(m_header.dwSuggestedBufferSize);
    write_le32(m_header.dwWidth);
    write_le32(m_header.dwHeight);
    for (unsigned i = 0; i < 4; i++)
	write_le32(m_header.dwReserved[i]);


    for (unsigned j = 0; j < m_streams.size(); j++)
    {
	int s = sizeof(AVIStreamHeader) + m_streams[j]->m_forsize
	    + (m_streams[j]->m_forsize & 1);
	hdr_size += 28 + s;
	write_le32(FOURCC_LIST);
	write_le32(20 + s);
        write_le32(listtypeSTREAMHEADER);
	write_le32(ckidSTREAMHEADER);
        write_le32(sizeof(AVIStreamHeader));
	m_fd->write(&m_streams[j]->m_header, sizeof(AVIStreamHeader));

	write_le32(ckidSTREAMFORMAT);
	write_le32(m_streams[j]->m_forsize);
	//printf("WRITE HEADER %d\n", m_streams[j]->m_forsize);
	m_fd->write(m_streams[j]->m_format, m_streams[j]->m_forsize);

	if (m_streams[j]->m_forsize & 1)
	    m_fd->write(&m_fd, 1);
    }

    if (hdr_size > 0x700)
	throw FATAL("Too large header. Aborting");

    int curpos = m_fd->lseek(0, SEEK_CUR);
    write_le32(ckidAVIPADDING);
    write_le32(0x7F4 - (curpos + 8));

    m_fd->lseek(0x7F4, SEEK_SET);
    write_le32(FOURCC_LIST);
    write_le32(endpos - 0x7FC);
    write_le32(listtypeAVIMOVIE);

    m_fd->lseek(hdr_pos, SEEK_SET);
    write_le32(hdr_size);
    m_fd->lseek(0, SEEK_END);
}
