#ifndef AVIFILE_READHANDLERS_H
#define AVIFILE_READHANDLERS_H

/**
 *
 * Interfaces that provide uniform access to data in Audio Video streams,
 * local and networked files  (Avi, Asf, maybe more later)
 *
 */

#include "except.h"
#include "StreamInfoPriv.h"

template<class T> inline T max(const T x, const T y)
{
    return (x > y) ? x : y;
}

template<class T> inline T min(const T x, const T y)
{
    return (x < y) ? x : y;
}

class IMediaReadStream {
  public:
    static const framepos_t ERR = ~0U;

    virtual ~IMediaReadStream() {}
    /**
     * Return structure describing main stream header
     *
     * NOTE: format is in the stream based one!
     * Use GetStreamInfo to obtain standard set of informations
     * about stream
     * this call will be eliminated later!!
     */
    virtual uint_t GetHeader(void *pheader, uint_t size) const = 0;

    /**
     * Return true if frame no. pos ( ERR means current frame ) is keyframe
     */
    virtual bool IsKeyFrame(framepos_t pos = ERR) const		= 0;

    /**
     * Seek to frame position pos in stream
     */
    virtual HRESULT Seek(framepos_t pos)			= 0;

    /** Seek to time */
    virtual HRESULT SeekTime(double time)			= 0;

    /**
     * Read up to lSamples into user-specified buffer, starting from
     * current position, and increase counter.
     * For video stream lpBuffer may be 0, in which case lSamples
     * should be 1 and function returns size of next video frame
     * in plBytes. */
    virtual HRESULT Read(uint_t lSamples, void *pBuffer, uint_t lBuffer,
    			 uint_t* plBytes, uint_t* plSamples)	= 0;
    /**
     * Return pointer to precached data if they exitst in local cache
     *
     * If stream doesn't support this it will return 0
     * Stream reader is than expected to fallback to use
     * standard 'Read' method above
     *
     * Main purpose is to speedup processing of uncompressed video streams
     * where we can safe copying over 1MB of memory per frame
     */
    virtual const void* ReadDirect(uint_t* plBytes, uint_t* plSamples) { return 0; }
    virtual HRESULT SkipFrame()					= 0;
    virtual HRESULT SkipTo(double time)				= 0;

    /** stream frame time - might be just average for whole stream */
    virtual double GetFrameTime() const				= 0;

    /** current stream position */
    virtual framepos_t GetPos() const				= 0;

    /**
     * position of previous, next and nearest key frame
     * related to frame pos ( ERR means current frame )
     */
    virtual framepos_t GetPrevKeyFrame(framepos_t pos = ERR) const = 0;
    virtual framepos_t GetNextKeyFrame(framepos_t pos = ERR) const = 0;
    virtual framepos_t GetNearestKeyFrame(framepos_t pos = ERR) const = 0;

    /* length of the stream in samples - equals with last frame position */
    virtual framepos_t GetLength() const			= 0;

    /* length of the stream in seconds */
    virtual double GetLengthTime() const			= 0;

    /* timestamp of sample no. lSample ( ERR = current sample ) */
    virtual double GetSampleTime(framepos_t lSample = ERR) const = 0;

    /**
     * various useful stream information
     *
     * this is preffered way to obtain information about stream!
     */
    virtual StreamInfo* GetStreamInfo() const			= 0;

    /* retrieval of stream-specific format data */
    virtual uint_t GetFormat(void *pFormat = 0, uint_t lSize = 0) const	= 0;

    /* 1 cache full, 0 cache empty */
    virtual double CacheSize() const				= 0;

    /* clear the internal stream cache */
    virtual void ClearCache()					= 0;

    /* usually fcc 'vids' or 'auds' */
    virtual fourcc_t GetFourCC() const				= 0;

    /* returns 0 for video ( frames of variable size ), >0 for audio */
    virtual uint_t GetSampleSize() const			= 0;
};

class IMediaReadHandler {
public:
    virtual ~IMediaReadHandler() {}
    /* vids/auds, 0-... */
    virtual IMediaReadStream *GetStream(fourcc_t fccType, uint_t lParam) = 0;
    virtual HRESULT GetHeader(void* pheader, uint_t n) const 	= 0;

    /* All calls for network-oriented readers should be interruptible  */
    virtual void Interrupt() 					= 0;

    /* in case of asynchronous opening */
    virtual bool IsOpened() const 				= 0;
    virtual bool IsValid() const				= 0;
    virtual bool IsRedirector() const 				= 0;
    virtual bool GetURLs(avm::vector<avm::string> &urls) const 	= 0;
};

/*  Read handler for AVI file  */
IMediaReadHandler *CreateAviMediaReadHandler(const char *pszFile) throw(FatalError);

/*  Read handler for ASF/WMV/ASX file  */
IMediaReadHandler *CreateASFReadHandler(const char *pszFile) throw(FatalError);

#endif // AVIFILE_READHANDLERS_H
