#ifndef AVIFILE_ITERATOR_H
#define AVIFILE_ITERATOR_H

#include "default.h"
#include "asffmt.h"

struct chunk_info
{
    static const uint_t KEYFRAME = 0x80000000U;

    uint_t packet_id; // 31bit marks keyframe - anyone has file with >2^31 packets?
    uint_t fragment_id;
    uint_t object_start_time;
    uint_t object_length;

    uint_t GetChunkLength() const { return object_length & ~KEYFRAME; }
    bool IsKeyFrame() const { return object_length & KEYFRAME; }
    void SetKeyFrame(bool s) { if (s) object_length |= KEYFRAME; }

};

class ASFStreamSeekInfo: public avm::vector<chunk_info>
{
public:
    static const framepos_t ERR = ~0U;

    framepos_t prevKeyFrame(framepos_t kf = ERR) const;
    framepos_t nextKeyFrame(framepos_t kf = ERR) const;
    framepos_t nearestKeyFrame(framepos_t kf = ERR) const;
    framepos_t find(uint_t time) const;
};

/**
 This logic is needed to properly handle reading from networked streams.
 Each stream owns an 'iterator' that can be used to access packets.
 If the stream is local, it allows iterators for different streams
 to point at completely independent packets ( and seek independently ).
 If it isn't, seek() on one iterator invalidates the other, and when
 seek() is called on invalidated iterator, it's automatically
 reposition at the place of last seek.
**/

// Interface
class Iterator
{
protected:
    int m_iRefcount;
public:
    Iterator() : m_iRefcount(1) {}
    virtual ~Iterator() {}
    virtual const ASFStreamSeekInfo* getSeekInfo()		=0;
    virtual void addRef() { m_iRefcount++; }
    virtual void release() { m_iRefcount--; if (!m_iRefcount) delete this; }
    virtual HRESULT seek(double timestamp, chunk_info* pch)	=0;
    virtual HRESULT seek(int64_t timestamp, chunk_info* pch)	=0;
    virtual packet getPacket(bool& success)			=0;
    virtual void next()						=0;
    virtual void undoPosition()					=0;
    virtual void undo()						=0;
    virtual void forgetUndo()					=0;
};

#endif // AVIFILE_ITERATOR_H
