#include "Cache.h"
#include "utils.h"
#include <cpuinfo.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>

#include <iomanip>
#include <ostream.h>
#include <string.h>
#include <stdio.h>

using namespace std;

// for smooth playing from CDROM with lower speed (setcd -x 4)
#define BUFFERING_SIZE 200

// how many bytes we will reuse when its possible
#define CACHE_BUFFER_REUSABLE 8196

// limit the maximum size of each prefetched stream in bytes
#define STREAM_SIZE_LIMIT 200000

#define __MODULE__ "Cache AVI"

#define Debug if(0)

struct req
{
    streamid_t id;//number of stream
    framepos_t position;//position in stream
    int64_t offset;
    uint_t size;
    uint_t alloc;
    char* memory;

    req() : alloc(0), memory(0) {}
    ~req() { delete memory; }

    char* getMemory()
    {
	if (alloc > CACHE_BUFFER_REUSABLE && (alloc - size) > 5000)
	{
	    // free large chunks
	    //cout << "freebuf " << alloc << "  m:" << (void *) memory << endl;
	    delete memory;
	    memory = 0;
	    alloc = 0;
	}

	if (size > alloc)
	{
	    // new data will not fit into current buffer -> realloc
	    // rounding to the nearest 1KB boundary
	    delete memory;
	    alloc = (size + 0xfff) & 0x7FFFF000;
	    if (alloc > 10000000)
		printf("WARNING  Too large cache chunk: %d ??\n", alloc);
            else
		memory = new char[alloc];
	}

	return memory;
    }
};

//
// think twice before you try to change anything in this file
// all the mutex locks "should" be right in place
//

Cache::Cache(uint_t size)
    :m_uiSize(size), m_bQuit(false), m_iFd(-1),
    cache_access(0), cache_right(0), cache_miss(0)
{
}

Cache::~Cache()
{
    m_bQuit = true;

    if (m_iFd > 0)
    {
        clear();
	cond_out.Broadcast();

	delete thread;

	for (unsigned i = 0; i < m_streams.size(); i++)
	{
	    stream_entry& stream = m_streams[i];

	    while (stream.freebuffers.size() > 0)
	    {
		req* r = stream.freebuffers.back();
		stream.freebuffers.pop_back();
		delete r;
	    }
	    while (stream.buffers.size() > 0)
	    {
		req* r = stream.buffers.back();
		stream.buffers.pop_back();
		delete r;
	    }
	}

        delete loaded;
    }

    if (cache_access != 0)
	printf("Destroying cache. Total accesses %d, hits %.2f%%, misses %.2f%%, errors %.2f%%\n",
	       cache_access, 100. * double(cache_right - cache_miss) / cache_access,
	       100. * double(cache_miss) / cache_access,
	       100. * double(cache_access - cache_right) / cache_access);
}

int Cache::addStream(streamid_t id, const avm::vector<AVIINDEXENTRY2>& table)
{
    cout << "Cache: Adding stream " << id << ", "
	<< table.size() << " chunks" << endl;

    stream_entry se(table, 0);
    while (se.freebuffers.size() < m_uiSize)
        se.freebuffers.push_back(new req());

    mutex_out.Lock();
    m_streams.push_back(se);
    mutex_out.Unlock();

    mutex_in.Lock();
    cond_in.Broadcast();
    mutex_in.Unlock();

    return 0;
}

// starts caching thread once we know file descriptor
int Cache::create(int fd)
{
    m_iFd = fd;
    cout << "Cache:   Creating cache for file descriptor: " << m_iFd << endl;
    if (m_streams.size() > 0)
    {
#if defined(__FreeBSD__) || defined(__NetBSD__)
	thread_lock.Lock();
#endif
        thread = new PthreadTask(0, &startReadfuncThread, this);
#if defined(__FreeBSD__) || defined(__NetBSD__)
	thread_cond.Wait(thread_lock);
	thread_lock.Unlock();
#endif
    }
    else
	cout << "Cache: WARNING no stream for caching! " << endl;

    return 0;
}

// return true if this stream should be waken and read new data
inline bool Cache::isCachable(stream_entry& stream, streamid_t id)
{
    // trick to allow precaching even very large image sizes
    return ((stream.sum < STREAM_SIZE_LIMIT
	     // assuming id=0 is video stream
	     // uncompressed video could be really huge!
	     //|| ((id == 0) && stream.buffers.size() < 3)
	    )
	    && stream.last < stream.table->size()
	    && stream.freebuffers.size() > 0
	    && (stream.filling
                || (stream.sum < STREAM_SIZE_LIMIT/2
		    && stream.buffers.size() < m_uiSize/2)));
}

//
// currently preffered picking alghorithm
//
// seems to be having good seek strategy
uint_t Cache::pickChunk()
{
    streamid_t id = m_uiId;

    do
    {
	stream_entry& stream = m_streams[id];

	// determine next needed chunk in this stream
	stream.last = (stream.buffers.size() > 0) ?
	    stream.buffers.back()->position + 1 : stream.position;

	if (isCachable(stream, id))
	    return id;

	// try next channel from streams
	id++;
	if (id >= m_streams.size())
	    id = 0; // wrap around

    }
    while (id != m_uiId);

    return WAIT; // nothing for caching found
}

// caching thread
void* Cache::readfunc()
{
    bool changed = false;

    m_uiId = 0;
    mutex_out.Lock();
    uint_t rb = 0;
    while (!m_bQuit)
    {
        m_uiId = pickChunk();

	if (m_uiId == WAIT)
	{
	    m_uiId = 0;
	    mutex_in.Lock();
	    mutex_out.Unlock();

	    // one could be trying to send signal to this thread
	    Debug cout << "full buffers - waiting for read"
		<< " s#1: " << m_streams[0].buffers.size()
		<< " s#2: " << m_streams[1].buffers.size()
		<< endl;
	    cond_in.Wait(mutex_in);
	    Debug cout << "full buffers - waiting done" << endl;
	    //cout << "full buffers - waiting done" << endl;
	    mutex_in.Unlock();
	    mutex_out.Lock();
            continue;
	}

	stream_entry& stream = m_streams[m_uiId];

        // get free buffer
	loaded = stream.freebuffers.back();
	stream.freebuffers.pop_back();

	loaded->id = m_uiId;
	loaded->position = stream.last;
	loaded->offset = (*stream.table)[stream.last].qwChunkOffset;
	loaded->size = (*stream.table)[stream.last].GetChunkLength() + 8;

	Debug cout << "id: " << m_uiId
	    << " buffered: " << (double)(stream.last - stream.position)
	    << " sum: " << m_streams[0].sum << " - " << m_streams[1].sum << endl;

	mutex_out.Unlock();

	//cout << "BUFFER Waiting Id:" << loaded->id
	//    << " Pos:" << loaded->position
	//    << " seek:" << loaded->offset << endl;
	off_t scur = lseek64(m_iFd, 0, SEEK_CUR);
	Debug cout << setw(1) << loaded->id
	    << " pos:" << setw(6) << loaded->position
	    << " seek:" << setw(7) << loaded->offset
	    << " df:" << setw(8) << loaded->offset - scur
	    << " lsz" << setw(6) << loaded->size
	    << " sum:" << setw(7) << stream.sum
	    << " buf:" << setw(3) << stream.buffers.size()
	    << endl;

	lseek64(m_iFd, loaded->offset & 0xFFFFFFFFFFFFLL, SEEK_SET);

        int r = 0;
	if (loaded->getMemory())
	    r = ::read(m_iFd, loaded->getMemory(), loaded->size & 0x7FFFFFFF);

	//cout << "BUFFER Read    Id:" << loaded->id
	//    << " Pos:" << loaded->position
	//    << " size:" << loaded->size << "  readed: " << r << endl;
	mutex_out.Lock();
        rb += r;
        // check if we still want same buffer
	stream.last = (stream.buffers.size() > 0) ?
	    stream.buffers.back()->position + 1 : stream.position;

	if (stream.last == loaded->position)
	{
	    stream.buffers.push_back(loaded);
	    stream.sum += loaded->size;
	    stream.filling = !(stream.sum > STREAM_SIZE_LIMIT);
	    Debug cout << "----------------      "
                << " id: " << loaded->id
		<< " pos: " << loaded->position
		<< " sum: " << stream.sum << "   size: " << loaded->size
		<< " filling: " << stream.filling << endl;
	}
	else
	{
	    Debug cout << " CHANGED LAST!!!" << endl;
	    stream.freebuffers.push_back(loaded);
	}
        loaded = 0;

	// anounce update
	cond_out.Broadcast();
	if (r > 10000)
	{
            // don't stress hardrive too much
	    mutex_in.Lock();
	    mutex_out.Unlock();
	    cond_in.Wait(mutex_in, 0.001);
	    mutex_in.Unlock();
	    mutex_out.Lock();
            r = 0;
	}
#if defined(__FreeBSD__)
	// I think we should not need usleep anymore avm_usleep(50);
#endif
    }

    mutex_out.Unlock();

    return 0;
}

/*
const void* Cache::readDirect(streamid_t id, framepos_t position)
{
    return 0;
}
*/
// called by stream reader - most of the time this read should
// be satisified from already precached chunks
int Cache::read(void *buffer, streamid_t id, framepos_t position, uint_t size, bool& success,
		uint_t offset)
{
    //Debug printf("Cache: read(buffer %p, id %d, pos %d, size %d, in_pos %d)\n",
    //    	 buffer, id, position, size, offset);
    cache_access++;
    success = false;
    if (id >= m_streams.size())
	return E_ERROR;

    stream_entry& stream = m_streams[id];
    if (position >= stream.table->size())
	return E_ERROR;

    //cout << "Before " << endl;
    mutex_out.Lock();

    stream.position = position;

    //cout << "Wait: " << id << " for " << position << endl;
    //while (stream.actual != position || stream.buffers.size() == 0)
    req* r = 0;

    while (!m_bQuit)
    {
	if (stream.buffers.size() > 0)
	{
	    r = stream.buffers[0];
	    if (r->position == position)
	    {
		cache_right++;
                break;
	    }

	    Debug cout << "position " << r->position << "   want: " << position << endl;
            // remove this chunk
            stream.buffers.pop_front();
	    stream.freebuffers.push_back(r);
            stream.sum -= r->size;
            continue;
	}
	cache_miss++;

	mutex_in.Lock();
	cond_in.Broadcast();
	mutex_in.Unlock();

	m_uiId = id;

	Debug cout << "--- actual: " << id << " size: " << stream.buffers.size() << endl;
	cond_out.Wait(mutex_out);
	Debug cout << "--- actual: " << id << "  done - size:"
	    << stream.buffers.size() << endl;
    }

    mutex_in.Lock();
    cond_in.Broadcast();
    mutex_in.Unlock();

    Debug cout << "id: " << id << " bsize: " << stream.buffers.size()
	<< "  memory: " << (void*) r->memory << "  pp:" << r->position << endl;

    fourcc_t chunk_id = *(fourcc_t*) r->memory;

    if (StreamFromFOURCC(chunk_id) != id)
    {
	cout << "WARNING: Read(): FOURCC mismatch ( received " << hex
            << chunk_id << dec << " )" << endl;
	mutex_out.Unlock();
	return size;
    }
    //if (size > max_size - in_pos - 8)
    //	size = max_size - in_pos - 8;

    memcpy(buffer, r->memory + offset + 8, size);
    mutex_out.Unlock();
    success = true;
    return size;
}

/*
int Cache::read(void *buffer, streamid_t id, framepos_t position, uint_t size,
		uint_t offset)
{


}
*/

int Cache::clear()
{
    Debug cout << "*** CACHE CLEAR ***" << endl;

    mutex_out.Lock();
    m_uiId = 0;
    mutex_out.Unlock();

    mutex_in.Lock();
    cond_in.Broadcast();
    mutex_in.Unlock();

#ifndef QUIET
    // DumpBuffers("Cache::Clear", false);
#endif
    return 0;
}

double Cache::getSize()
{
    /*
       int status=0;
       for(int i=0; i<m_uiSize; i++)
       if(req_buf[i].st==req::BUFFER_READY)status++;
       return (double)status/m_uiSize;
     */
    return 1.;
}

void* Cache::startReadfuncThread(void* arg)
{
    Cache* c = (Cache*)arg;
#if defined(__FreeBSD__) || defined(__NetBSD__)
    c->thread_lock.Lock();
    c->thread_cond.Broadcast();
    c->thread_lock.Unlock();
#endif
    return c->readfunc();
}


void Cache::dumpBuffers(const char *txt, bool show_table)
{

}

// pick next chunk for precaching - good alghorithm here could improve
// load speed - however good strategy is not the obvisious
// this is test of different alghorithm
uint_t Cache::pickChunk1()
{
    unsigned didx = ~0U;
    int64_t dmin = 0;
    int64_t coffset = lseek64(m_iFd, 0, SEEK_CUR);

    unsigned i;

    // pick some initial stream which could be filled
    for (i = 0; i < m_streams.size(); i++)
    {
	stream_entry& stream = m_streams[i];

	// determine next needed chunk in this stream
	stream.last = (stream.buffers.size() > 0) ?
	    stream.buffers.back()->position + 1 : stream.position;

	//printf("id: %d  sum: %d     last: %d   tsize: %d  fsize: %d\n",
	//       i, stream.sum, stream.last, stream.table->size(),
	//       stream.freebuffers.size());
	if (stream.sum < STREAM_SIZE_LIMIT
	    && stream.last < stream.table->size()
	    && stream.freebuffers.size() > 0)
	{
	    didx = i;
	    dmin = (*stream.table)[stream.last].qwChunkOffset;

	    //cout << i <<  "  LAST " << stream.last << " coffset " << coffset << "  dmin: " << dmin << endl;
	    break;
	}
    }

    // pick channel with smallest positive offset
    // -> nearest chunk in the stream flow
    for (i++; i < m_streams.size(); i++)
    {
	stream_entry& stream = m_streams[i];

	// determine next needed chunk in this stream
	stream.last = (stream.buffers.size() > 0) ?
	    stream.buffers.back()->position + 1 : stream.position;

	//printf("id: %d  sum: %d     last: %d   tsize: %d  fsize: %d\n",
	//       i, stream.sum, stream.last, stream.table->size(),
	//       stream.freebuffers.size());
	if (stream.sum > STREAM_SIZE_LIMIT
	    || stream.last >= stream.table->size()
	    || stream.freebuffers.size() == 0)
	    continue;

	int64_t d = (*stream.table)[stream.last].qwChunkOffset;
	if (d < dmin)
	{
	    //cout << i <<  "  XLAST " << stream.last << " newdmin " << dmin << endl;
	    dmin = d;
	    didx = i;
	}
    }

    return didx;
}

/*************************************************************/

InputStream::InputStream(const char *pszFile) throw(FatalError)
    :cache(new Cache(BUFFERING_SIZE))
{
    m_iFd = open(pszFile, O_RDONLY);
    if (m_iFd < 0)
	throw FATAL("Could not open file");
    off_t spos = lseek64(m_iFd, 0, SEEK_CUR);

    ulLength = lseek64(m_iFd, 0, SEEK_END);
    seek(spos);
    m_iPos = ~0;
    m_iBuffered = 0;
}

InputStream::~InputStream()
{
    delete cache;

    if (m_iFd >= 0)
	close(m_iFd);
}

int InputStream::async()
{
    return (cache) ? cache->create(m_iFd) : -1;
}

int InputStream::prefetch(streamid_t id, framepos_t position)
{
    if (cache)
    {
	cache->prefetch(id, position);
	cache->update();
    }
    return 0;
}

uint_t InputStream::seek(int64_t offset)
{
    m_iPos = 0;
    m_iBuffered = 0;
    return lseek64(m_iFd, offset, SEEK_SET);
}

uint_t InputStream::seekCur(int64_t offset)
{
    //cout << "seekcur " << offset << "   " << m_iPos << endl;
    if (m_iPos >= m_iBuffered)
	return lseek64(m_iFd, offset, SEEK_CUR);

    if (offset >= 0)
    {
	m_iPos += offset;
	if (m_iPos >= m_iBuffered)
	    return lseek64(m_iFd, m_iPos - m_iBuffered, SEEK_CUR);
    }
    else
    {
	if (m_iPos < -offset)
	{
	    offset += m_iBuffered - m_iPos;
	    m_iBuffered = 0;
	    return lseek64(m_iFd, offset, SEEK_CUR);
	}
	m_iPos += offset;
    }
    return pos();
}

uint_t InputStream::pos()
{
    uint_t o = lseek64(m_iFd, 0, SEEK_CUR);
    if (m_iPos < m_iBuffered)
	o -= (m_iBuffered - m_iPos);
    //if (o > 733810000)
    //    cout << "pos " << o  << "   " << m_iPos << endl;
    return o;
}

int InputStream::read(void* buffer, uint_t size)
{
    int r = 0;
    if (m_iBuffered > 0)
    {
	//cout << "pos " << m_iPos << endl;
	uint_t copy = m_iBuffered - m_iPos;
	if (size < copy)
	    copy = size;
	memcpy(buffer, bfr + m_iPos, copy);
	m_iPos += copy;
	r = copy;
	size -= copy;
        buffer = (char*) buffer + copy;
	//cout << "READfrompos " << copy << "  " << size << endl;
    }
    if (size > 0)
	r += ::read(m_iFd, buffer, size);

    //cout << "rd " << r << "   " << m_iPos << endl;
    return r;
}

uint8_t InputStream::readByte()
{
    if (m_iPos >= m_iBuffered)
    {
	int r = ::read(m_iFd, bfr, sizeof(bfr));
	m_iBuffered = (r > 0) ? r : 0;
	m_iPos = 0;
	//std::cout << "READ " << sizeof(bfr) << std::endl;
    }
    //std::cout << "READBYTE " << bfr[m_iPos] << std::endl;
    return bfr[m_iPos++];
#if 0
    uint8_t c;
    ::read(m_iFd, &c, 1);

    return c;
#endif
}

uint32_t InputStream::readDword()
{
    uint32_t i = readByte();
    i |= (readByte() << 8);
    i |= (readByte() << 16);
    i |= (readByte() << 24);
    //cout << "readdword " << i << endl;

    return i;
}

uint16_t InputStream::readWord()
{
    uint16_t i = readByte();
    i |= (readByte() << 8);

    return i;
}
