/*************************************************************

	AviReadStreamA ( audio AVI stream ) Functions

**************************************************************/

#include "AviRead.h"
#include "AviReadStreamA.h"
#include "avifmt.h"
#include "creators.h"
#include "audiodecoder.h"
#include "cpuinfo.h"
#include "utils.h"

#include <string.h> // memcpy
#include <stdlib.h>
#include <stdio.h>

#define Debug if(0)
#define __MODULE__ "AviReadStreamV"

// code here inspired & cut&pasted from gmerlin sources

static int mpeg_bitrates[16][6] =
{
  {   0,         0,      0,      0,      0,      0 },
  {  32000,  32000,  32000,  32000,  32000,   8000 },
  {  64000,  48000,  40000,  64000,  48000,  16000 },
  {  96000,  56000,  48000,  96000,  56000,  24000 },
  { 128000,  64000,  56000, 128000,  64000,  32000 },
  { 160000,  80000,  64000, 160000,  80000,  64000 },
  { 192000,  96000,  80000, 192000,  96000,  80000 },
  { 224000, 112000,  96000, 224000, 112000,  56000 },
  { 256000, 128000, 112000, 256000, 128000,  64000 },
  { 288000, 160000, 128000, 288000, 160000, 128000 },
  { 320000, 192000, 160000, 320000, 192000, 160000 },
  { 352000, 224000, 192000, 352000, 224000, 112000 },
  { 384000, 256000, 224000, 384000, 256000, 128000 },
  { 416000, 320000, 256000, 416000, 320000, 256000 },
  { 448000, 384000, 320000, 448000, 384000, 320000 },
  {      0,      0,      0,      0,      0,      0 }
};

static int mpeg_samplerates[3][3] =
{
  { 44100, 48000, 32000 }, // MPEG1
  { 22050, 24000, 16000 }, // MPEG2
  { 11025, 12000, 8000 }   // MPEG2.5
};

#define MPEG_ID_MASK        0x00180000
#define MPEG_MPEG1          0x00180000
#define MPEG_MPEG2          0x00100000
#define MPEG_MPEG2_5        0x00000000

#define MPEG_LAYER_MASK     0x00060000
#define MPEG_LAYER_III      0x00020000
#define MPEG_LAYER_II       0x00040000
#define MPEG_LAYER_I        0x00060000
#define MPEG_PROTECTION     0x00010000
#define MPEG_BITRATE_MASK   0x0000F000
#define MPEG_FREQUENCY_MASK 0x00000C00
#define MPEG_PAD_MASK       0x00000200
#define MPEG_PRIVATE_MASK   0x00000100
#define MPEG_MODE_MASK      0x000000C0
#define MPEG_MODE_EXT_MASK  0x00000030
#define MPEG_COPYRIGHT_MASK 0x00000008
#define MPEG_HOME_MASK      0x00000004
#define MPEG_EMPHASIS_MASK  0x00000003

#define LAYER_I_SAMPLES       384
#define LAYER_II_III_SAMPLES 1152

struct AudioInfoMp3
{
    enum MPEG_MODE { MPEG1, MPEG2, MPEG2_5 } mode;
    enum STEREO_MODE { STEREO = 0, JSTEREO, DUAL_CHANNEL, MODE_MONO } stereo_mode;
    int vbr;
    int layer;
    int bitrate;
    int start_offset;
    int samplerate;
    int samples_per_frame;
    int num_channels;
    int frame_size;
    AudioInfoMp3() : vbr(0), layer(0), bitrate(0), num_channels(0), frame_size(0) {}

    // MP3 header needs 40 bytes
    int Init(const char* buf)
    {
	frame_size = 0;
	uint32_t header = avm_get_be32(buf);

	//printf("HDR %x    %.4s\n", header, (char*)&header);
	// CHECK if it's mpeg & Get the sampling frequency
	if ((header & 0xffe00000) != 0xffe00000
	    || (header & 0x0000fc00) == 0x0000fc00)
	    return 0;


	start_offset = 40;
	int frequency_index = (header & MPEG_FREQUENCY_MASK) >> 10;
	mode = (MPEG_MODE) (3 - ((header & MPEG_ID_MASK) >> 19));
	if (mode > 2)
            mode = MPEG2_5;
	samplerate = mpeg_samplerates[mode][frequency_index];

	//printf("HDR %x    m:%d  fi:%d   sr:%d\n", header, mode, frequency_index, samplerate);
        // Get stereo mode index
	stereo_mode = (STEREO_MODE)((header & MPEG_MODE_MASK) >> 6);
	num_channels = (stereo_mode == MODE_MONO) ? 1 : 2;

	// Get layer
	switch (header & MPEG_LAYER_MASK)
	{
	case MPEG_LAYER_I:
	    layer = 1;
	    samples_per_frame = LAYER_I_SAMPLES;
	    break;
	case MPEG_LAYER_II:
	    layer = 2;
	    samples_per_frame = LAYER_II_III_SAMPLES;
	    break;
	case MPEG_LAYER_III:
	    layer = 3;
	    samples_per_frame = LAYER_II_III_SAMPLES;
	    break;
	default:
            return 0;
	}

	if (mode == MPEG2)
	    samples_per_frame /= 2;
	else if (mode == MPEG2_5)
	    samples_per_frame /= 4;

        // Check for VBR stream (Xing header)
        int b = 4;
	if (header & MPEG_ID_MASK)
	    b += (stereo_mode != MODE_MONO) ? 32 : 17; // mpeg1
        else
	    b += (stereo_mode != MODE_MONO) ? 17 : 9; // mpeg2
	uint32_t xing = avm_get_le32(buf + b);
	vbr = (mmioFOURCC('X', 'i', 'n', 'g') == xing);

	bitrate = (vbr) ? 0 : GetBitrate(header);

	//printf ("%d %d    l:%d  %d   ch:%d\n", samplerate, bitrate, layer, samples_per_frame, num_channels);
	frame_size = GetFrameSize(header);
	if (vbr)
	    start_offset += frame_size + 4;

        if (frame_size)
	    printf("Stream MPEG header  VBR:%d  Layer:%d  SampleRate:%d Bitrate:%d\n"
		   "SamplesPerFrame:%d  NumChannels:%d  StereoMode:%d  Offset:%d\n",
		   vbr, layer, samplerate, bitrate, samples_per_frame,
		   num_channels, stereo_mode, start_offset);

        return start_offset;
    }

    int GetBitrate(uint32_t header)
    {
	int32_t bitrate_index = (header & MPEG_BITRATE_MASK) >> 12;
	return mpeg_bitrates[bitrate_index][layer + ((layer > 0 && mode == MPEG1) ? -1 : 2)];
    }

    int GetFrameSize(uint32_t header)
    {
	int tmp_bitrate = GetBitrate(header);

	if (!tmp_bitrate)
	    return 0;

	int ret;

	if (layer == 1)
	{
	    ret = (12 * tmp_bitrate / samplerate);
	    if (header & MPEG_PAD_MASK)
		ret++;
	    ret *= 4;
	}
	else
	{
	    ret = 144 * tmp_bitrate / samplerate;
	    if (header & MPEG_PAD_MASK)
		ret++;
	}
	if (mode == MPEG2)
	    ret /= 2;
	else if (mode == MPEG2_5)
	    ret /= 4;

	return ret;
    }
};


using namespace Creators;

static const uint_t wfmtx_size = sizeof(WAVEFORMATEX);
// sizeof(WAVEFORMATEX) depends on packing
// in this case we use some attributes


AviReadStreamA::AviReadStreamA(streamid_t id, IMediaReadStream* stream)
    :AviReadStream(id, stream), audiodecoder(0), m_pAudioInfo(0)
{
}

AviReadStreamA::~AviReadStreamA()
{
    StopStreaming();
    delete m_pAudioInfo;
}

// FIXME - when audiodecoder is running we may supply real data
uint_t AviReadStreamA::GetAudioFormat(void* wf, uint_t size) const
{
    if (wf)
	memcpy(wf, m_format, (size < wfmtx_size) ? size : wfmtx_size);
    return wfmtx_size;
}

uint_t AviReadStreamA::GetFrameSize() const
{
    return (audiodecoder) ? audiodecoder->GetMinSize() : 0;
}

framepos_t AviReadStreamA::GetPos() const
{
    uint_t dwSampleSize = m_pIStream->GetSampleSize();
    framepos_t fpos = m_pIStream->GetPos();
    Debug printf("AviReadStreamA::GetPos() Sample: %d, rem_size: %d  sample_size: %dn",
		 m_pIStream->GetPos(), rem_size, dwSampleSize);

    if (dwSampleSize != 0)
    {
	uint_t tmp = rem_size / dwSampleSize;

	if (tmp <= fpos)
	    fpos -= tmp;
	//else printf("********* fpos: %d   tmp: %d   rem_size: %d  ssize: %d\n", fpos, tmp, rem_size, dwSampleSize);
    }
    return fpos;
}

double AviReadStreamA::GetTime(framepos_t pos) const
{
    int dwBps = ((WAVEFORMATEX*)m_format)->nAvgBytesPerSec;

    double smtime = m_pIStream->GetSampleTime(pos);

    if (pos == ERR && dwBps > 0)
    {
	//printf("AudioGetTime remsize %f   time: %f  pos: %d\n", rem_size/(double)dwBps, smtime, pos);
	smtime -= rem_size/(double)dwBps;
	if (smtime < 0.0)
	    smtime = 0.0;
    }

    return smtime;
}

uint_t AviReadStreamA::GetOutputFormat(void* format, uint_t size) const
{
    if (!audiodecoder)
	return 0;
    if (!format)
	return wfmtx_size;
    if (size < wfmtx_size)
	return 0;

     return audiodecoder->GetOutputFormat((WAVEFORMATEX*)format);
}

bool AviReadStreamA::IsStreaming() const
{
    return (audiodecoder != NULL);
}

// few CPU cycles are wasted here
// FIXME later
HRESULT AviReadStreamA::ReadFrames(void* buffer, uint_t bufsize, uint_t samples,
				   uint_t& samples_read, uint_t& bytes_read)
{
    if (!audiodecoder || !samples)
	return -1;

    uint_t rem_diff = 0;
    uint_t lBytes, lSamples;
    samples_read = 0;
    bytes_read = 0;
    HRESULT hr = -1;

    char* temp_buffer = new char[bufsize * 2];

    if (rem_buffer)
    {
#if 0
	rem_diff = min(rem_size, bufsize);
	memcpy(temp_buffer, rem_buffer, rem_diff);
	if (rem_diff < rem_size)
	    memcpy(rem_buffer, rem_buffer + rem_diff, rem_size - rem_diff);
#endif
	rem_diff = rem_size;
	memcpy(temp_buffer, rem_buffer, rem_diff);
    }

    lBytes = lSamples = 0;

    uint_t srcsize = audiodecoder->GetSrcSize(bufsize + 3000);
    if (bufsize < srcsize)
        srcsize = bufsize;

    if (rem_diff < srcsize)
    {
	srcsize -= rem_diff;
	//printf("bufsize: %d   remdiff: %d   srcsize: %d  samples: %d\n", bufsize, rem_diff, srcsize, samples);
	//m_pIStream->Read(1, 0, 0, &lBytes, &lSamples);
	hr = m_pIStream->Read(samples, temp_buffer + rem_diff, srcsize,
			      &lBytes, &lSamples);
	if (hr == 0)
	{
	    Debug printf("AviReadStreamA: Read %d samples ( %d bytes ) from pos %d\n",
			 (int)lSamples, (int)lBytes, m_pIStream->GetPos());
	    //printf("AviReadStreamA: Read %d samples ( %d bytes ) from pos %d\n",
	    //       lSamples, lBytes, m_pIStream->GetPos());
	    samples_read += lSamples;
	}
    }

    lBytes += rem_diff;
    uint_t local_read = 0, ocnt = 0;
    if (lBytes)
    {
	uint_t convsz = 0;
	uint_t lr = 0;
        //memset(buffer, 0, bufsize);
	while ((convsz + ocnt) < bufsize && (local_read + 2 * lr) < lBytes)
	{
	    hr = audiodecoder->Convert((char*)temp_buffer + local_read, lBytes - local_read,
				       (char*)buffer + convsz, bufsize - convsz,
				       &lr, &ocnt);
	    //printf("lBytes: %d   locread: %d    outputsz:%d   convsz: %d\n", lBytes, local_read, ocnt, convsz);
	    if (lr > lBytes)
		lr = lBytes;
            local_read += lr;
            convsz += ocnt;
	    if (!ocnt)
                break;
	}
#if 0
	unsigned char* p = (unsigned char*)buffer + convsz;
	for (int i = -32; i < 64; i++)
	{
	    if (!(i % 8))
		printf("\n%4d ", i);
	    //printf("  0x%4.4x", abs(p[i] - p[i-2]) & 0xffff);
	    printf("  0x%02X", p[i]);
	}
#endif
        ocnt = convsz;
    }
    else
	m_iEof++;

    if (!rem_buffer && (lBytes > local_read))
    {
	rem_limit = max(2 * audiodecoder->GetMinSize(), 50000U);
	rem_buffer = new char[rem_limit];
    }
    rem_size = min(lBytes-local_read, (uint_t)rem_limit);
    Debug printf("AviReadStreamA::ReadFrames: converted %d bytes of %d to %d, bufsize %d\n",
		 (int)local_read, (int)lBytes, (int)ocnt, (int)bufsize);
    if (!local_read && !ocnt)
    {
	//shouldn't happen. Just skip some data, hoping that things will be OK later.
	local_read = lBytes;
	ocnt = 0;
	rem_size = 0;
    }
    else
    {
	//cout << "REMSIZE " << rem_size << endl;
	memcpy(rem_buffer, temp_buffer + local_read, rem_size);
	bytes_read = ocnt;
    }

    delete[] temp_buffer;
    Debug printf("AviReadStreamA: new m_sample is %d\n", m_pIStream->GetPos());
    return hr;
}

HRESULT AviReadStreamA::SkipTo(double pos)
{
    if (pos > m_pIStream->GetLengthTime())
	return -1;

    char* b = new char[10000];
    uint_t samples_read, bytes_read;
    while (pos - GetTime() > 0.2)
    {
	if (m_pIStream->Read(1, b, 5000, 0, 0) != 0)
	    break;
	//if (ReadFrames(b, 10000, 1, samples_read, bytes_read) != 0) break;
    }
    delete[] b;
    return 0;
}

HRESULT AviReadStreamA::StartStreaming(const char* privname)
{
    if (audiodecoder)
	return -1;

    WAVEFORMATEX* wf = (WAVEFORMATEX*)m_format;

    if (wf->wFormatTag == 0x55 || wf->wFormatTag == 0x50)
    {
	if (!m_pAudioInfo)
	{
            m_pAudioInfo = new AudioInfoMp3;
	    framepos_t o = m_pIStream->GetPos();
	    char* b = new char[10000];
	    uint_t lBytes = 0, lSamples = 0;

	    m_pAudioInfo->frame_size = 0;
	    while (m_pAudioInfo->frame_size == 0 && m_pAudioInfo->layer != 3)
	    {
		int hr = m_pIStream->Read(5000, b, 5000, &lBytes, &lSamples);
		if (hr != 0 || lBytes == 0)
		    break;
		//int ss = 0;
		//for (uint_t x = 0; x < lBytes; x++)
		//    ss += b[x];
		//printf("HR  %d    lb %d   samp: %d    sum  %d\n", hr, lBytes, lSamples, ss);
		if (lBytes >= 4)
		{
		    uint_t j = 0;
		    while (j < lBytes - 4)
		    {
			m_pAudioInfo->Init(b + j++);
			if (m_pAudioInfo->frame_size)
			    break;
		    }
		}
	    }
	    delete[] b;
	    m_pIStream->Seek(o);
	}

	if (m_pAudioInfo->frame_size)
	{
	    wf->nChannels = m_pAudioInfo->num_channels;
	    wf->nSamplesPerSec = m_pAudioInfo->samplerate;
	    wf->nBlockAlign = m_pAudioInfo->frame_size;
	    if (0 && m_pAudioInfo->bitrate) // we leave original value here
		wf->nAvgBytesPerSec = m_pAudioInfo->bitrate / 8;
	}
    }

    audiodecoder = CreateAudioDecoder(wf, privname);
    if (!audiodecoder)
    {
	printf("Failed to initialize audio decoder for format 0x%x\n",
	       ((WAVEFORMATEX*)m_format)->wFormatTag);
        return -1;
    }

    rem_size = 0;
    return 0;
}

HRESULT AviReadStreamA::StopStreaming()
{
    if (audiodecoder)
    {
	FreeAudioDecoder(audiodecoder);
	audiodecoder = 0;
    }
    return 0;
}
