/********************************************************************

 AviReadStream functions that are common for video and audio streams

*********************************************************************/

#include "AviReadStream.h"
#include "fourcc.h"

#include <stdio.h>
#define Debug if(0)

#define __MODULE__ "AviReadStream"

AviReadStream::AviReadStream(streamid_t id, IMediaReadStream* stream)
    :m_pIStream(stream), m_id(id), m_ch_off(0),
    m_iEof(0), rem_buffer(0), rem_size(0), rem_limit(0)
{
    if (!stream)
	throw FATAL("NULL stream passed!");

    //m_pIStream->GetHeader(&m_header, sizeof(m_header));
    m_format_size = m_pIStream->GetFormat();
    m_format = new char[m_format_size];
    m_pIStream->GetFormat(m_format, m_format_size);

    printf("Initialized stream %d (Chunk table size %d, format size %d)\n",
	   id, m_pIStream->GetLength(), m_format_size);
}

AviReadStream::~AviReadStream()
{
    delete[] rem_buffer;
    delete[] m_format;
}

bool AviReadStream::Eof() const
{
    //cout << "EOF  " << m_iEof << "   P:" << GetPos() << "  " << m_pIStream->GetLength() << "  eof " << m_iEof << endl;
    framepos_t pos = GetPos();
    uint_t len = m_pIStream->GetLength();
    if (pos >= len || m_iEof > 15)
    {
	//Debug cout<<"End of stream ( "<<pos<<" >= "<< len <<" )"<<endl;
	return 1;
    }
    //cout<<"Not end of stream ( "<<pos<<"<"<<len<<" ) " << m_iEof <<  endl;
    return 0;
}

AviReadStream::StreamType AviReadStream::GetType() const
{
    switch (m_pIStream->GetFourCC())
    {
    case mmioFOURCC('v', 'i', 'd', 's'):
	return Video;
    case mmioFOURCC('a', 'u', 'd', 's'):
	return Audio;
    }
    return Other;
}

uint_t AviReadStream::GetHeader(void* pheader, uint_t n) const
{
    Debug printf("AviReadStream::GetHeader()\n");
    return m_pIStream->GetHeader(pheader, n);
}

HRESULT AviReadStream::Seek(framepos_t pos)
{
    Debug printf("AviReadStream::Seek() %d   id=%d\n", pos, m_id);
    if (pos >= m_pIStream->GetLength())
	return -1;

    HRESULT hr = m_pIStream->Seek(pos);
    if (hr == 0)
	m_iEof = 0;

    rem_size = 0;
    return hr;
}

HRESULT AviReadStream::SeekTime(double timepos)
{
    Debug printf("AviReadStream::SeekTime() %f\n", timepos);
    if (timepos >= m_pIStream->GetLengthTime())
	return -1;

    HRESULT hr = m_pIStream->SeekTime(timepos);
    if (hr == 0)
	m_iEof = 0;

    rem_size = 0;
    return hr;
}

framepos_t AviReadStream::SeekToKeyFrame(framepos_t pos)
{
    if (Seek(pos) != -1)
	return pos;
    return ERR;
}

double AviReadStream::SeekTimeToKeyFrame(double timepos)
{
    if (SeekTime(timepos) != -1)
	return timepos;//GetTime();
    return -1.0;
}

HRESULT AviReadStream::SkipFrame()
{
    Debug printf("AviReadStream::SkipFrame()\n");
    return m_pIStream->SkipFrame();
}

HRESULT AviReadStream::SkipTo(double pos)
{
    Debug printf("AviReadStream::SkipTo()\n");
    return m_pIStream->SkipTo(pos);
}

framepos_t AviReadStream::GetNextKeyFrame(framepos_t pos) const
{
    Debug printf("AviReadStream::GetNextKeyFrame() %d\n", pos);
    if (pos == ERR)
        pos = GetPos() + 1;
    return m_pIStream->GetNextKeyFrame(pos);
}

framepos_t AviReadStream::GetPrevKeyFrame(framepos_t pos) const
{
    Debug printf("AviReadStream::GetPrevKeyFrame() %d\n", pos);
    if (pos == ERR)
        pos = GetPos();

    return m_pIStream->GetPrevKeyFrame(pos);
}

framepos_t AviReadStream::SeekToNextKeyFrame()
{
    Debug printf("AviReadStream::SeekToNextKeyFrame()\n");
    framepos_t newpos = GetNextKeyFrame();
    if (newpos != ERR)
	Seek(newpos);

    return GetPos();
}

framepos_t AviReadStream::SeekToPrevKeyFrame()
{
    Debug printf("AviReadStream::SeekToPrevKeyFrame()\n");
    framepos_t newpos = GetPrevKeyFrame();
    if (newpos == ERR)
        newpos = 0;
    Seek(newpos);
    return m_pIStream->GetPos();
}

HRESULT AviReadStream::ReadDirect(void* buffer, uint_t bufsize, uint_t samples,
				  uint_t& samples_read, uint_t& bytes_read,
				  int* flags)
{
    if (flags)
	*flags = (m_pIStream->IsKeyFrame()) ? KEYFRAME : 0;
    HRESULT hr = m_pIStream->Read(samples, buffer, bufsize,
				  &bytes_read, &samples_read);
    if ((bufsize > 0) && !bytes_read && !samples_read)
	m_iEof++;
    else
        m_iEof = 0;

    return hr;
}

HRESULT AviReadStream::GetFrameFlags(int* flags) const
{
    if (!flags)
	return -1;

    // FIXME - add more flags later
    *flags = (m_pIStream->IsKeyFrame()) ? KEYFRAME : 0;

    return 0;
}

double AviReadStream::GetFrameTime() const
{
    return m_pIStream->GetFrameTime();
}

double AviReadStream::GetTime(framepos_t pos) const
{
    return m_pIStream->GetSampleTime(pos);
}

framepos_t AviReadStream::GetPos() const
{
    return m_pIStream->GetPos();
}

framepos_t AviReadStream::GetLength() const
{
    return m_pIStream->GetLength();
}

double AviReadStream::GetLengthTime() const
{
    return m_pIStream->GetLengthTime();
}

StreamInfo* AviReadStream::GetStreamInfo() const
{
    return m_pIStream->GetStreamInfo();
}

uint_t AviReadStream::GetVideoFormat(void* bi, uint_t size) const
{
    printf("WARNING: GetVideoFormat() for non-video stream\n");
    return 0;
}

uint_t AviReadStream::GetAudioFormat(void* bi, uint_t size) const
{
    printf("WARNING: GetAudioFormat() for non-audio stream\n");
    return 0;
}

IVideoDecoder* AviReadStream::GetVideoDecoder() const
{
    printf("WARNING: Getting video decoder for non-video stream\n");
    return 0;
}

HRESULT AviReadStream::ReadFrame(bool render)
{
    printf("WARNING: AviReadStream::ReadFrame() called for non-video stream\n");
    return -1;
}

CImage* AviReadStream::GetFrame(bool readframe)
{
    printf("WARNING: AviReadStream::GetFrame() called for non-video stream\n");
    return 0;
}

uint_t AviReadStream::GetFrameSize() const
{
    printf("WARNING: AviReadStream::GetFrameSize() called for non-video stream\n");
    return 0;
}

HRESULT AviReadStream::SetOutputFormat(void* bi, uint_t size)
{
    printf("WARNING: AviReadStream::SetOutputFormat() called\n");
    return -1;
}

HRESULT AviReadStream::ReadFrames(void* buffer, uint_t bufsize, uint_t samples,
				  uint_t& samples_read, uint_t& bytes_read)
{
    printf("WARNING: AviReadStream::ReadFrames() called for non-audio stream\n");
    return -1;
}
