/********************************************************************

 AviMediaReadStream functions that are common for video and audio streams

*********************************************************************/

#include "AviMediaReadStream.h"
#include "AviMediaReadHandler.h"
#include "formats.h"

#include <string.h>
#include <stdio.h>

#define Debug if(0)

AviMediaReadStream::AviMediaReadStream(AviMediaReadHandler* parent,
                                       const AVIStreamHeader& hdr,
				       streamid_t id,
				       char* format, uint_t fsize)
    :m_pParent(parent), m_Header(hdr), m_iId(id), m_pcFormat(format),
    m_uiFormatSize(fsize), m_uiPosition(0), m_uiChunk(0)
{
    m_dAvgBytesPerSec = (m_Header.dwScale) ? m_Header.dwRate / (double) m_Header.dwScale : 1.0;
    if (m_Header.fccType == streamtypeAUDIO && m_Header.dwSampleSize)
    {
	m_dAvgBytesPerSec = ((WAVEFORMATEX*)m_pcFormat)->nAvgBytesPerSec;
	if (m_Header.dwSampleSize > 1)
	    m_dAvgBytesPerSec /= m_Header.dwSampleSize;
    }
    //cout << "AvgBytesPerSec: " << m_dAvgBytesPerSec << endl;
}

AviMediaReadStream::AviMediaReadStream(const AviMediaReadStream& c)
    :m_pcFormat(0)
{
    operator=(c);
}

AviMediaReadStream::~AviMediaReadStream()
{
    delete[] m_pcFormat;
}

AviMediaReadStream& AviMediaReadStream::operator=(const AviMediaReadStream& c)
{
    m_iId = c.m_iId;
    m_Header = c.m_Header;
    m_Index = c.m_Index;
    m_pParent = c.m_pParent;
    m_dAvgBytesPerSec = c.m_dAvgBytesPerSec;
    m_uiPosition = c.m_uiPosition;
    m_uiChunk = c.m_uiChunk;
    delete[] m_pcFormat;
    if (c.m_pcFormat)
    {
	m_pcFormat = new char[c.m_uiFormatSize];
	m_uiFormatSize = c.m_uiFormatSize;
	memcpy(m_pcFormat, c.m_pcFormat, m_uiFormatSize);
    }
    else
    {
	m_pcFormat = 0;
	m_uiFormatSize = 0;
    }

    return *this;
}

framepos_t AviMediaReadStream::find(framepos_t lSample) const
{
    if (lSample >= GetLength() || lSample == ERR)
	return ERR;

    //printf("FIND %d   %d\n", lSample, m_Index.size());
    framepos_t low_limit = 0;
    framepos_t high_limit = m_Index.size() - 1;
    while (low_limit != high_limit)
    {
	framepos_t middle = (low_limit + high_limit) / 2;
	if (lSample >= m_Index[middle].qwTimestamp)
	{
            // this fix is necessary to prevent deadlock loop here!
	    if (low_limit == middle)
                break;

	    low_limit = middle;
	    if (lSample < m_Index[middle + 1].qwTimestamp)
		//+ (m_Index[middle].dwChunkLength / m_Header.dwSampleSize)))
		break;
	}
	else
	    high_limit = middle;
    }

    return low_limit;
}


uint_t AviMediaReadStream::GetHeader(void* pheader, uint_t size) const
{

    if (pheader)
    {
	memset(pheader, 0, size);
	memcpy(pheader, &m_Header, sizeof(m_Header));

#if 0
	for (int i = 0 ; i < 150; i++)
	    cout << i
		<< "   " << m_Index[i].qwChunkOffset
		<< "   " << m_Index[i].GetLength()
		<< "   " << m_Index[i].IsKeyFrame()
		<< "   " << m_Index[i].qwTimestamp << endl;
#endif
    }
    return sizeof(m_Header);
}

bool AviMediaReadStream::IsKeyFrame(framepos_t pos) const
{
    if (m_Header.dwSampleSize)
	return true; //audio

    framepos_t chunk_id = (pos == ERR) ? GetPos() : pos;
    //cout << "ISKF " << chunk_id << "   " << pos << endl;
    if (chunk_id >= m_Index.size() || chunk_id == ERR)
	return true; //or whatever

    return m_Index[chunk_id].IsKeyFrame();
}

HRESULT AviMediaReadStream::Seek(framepos_t pos)
{
    Debug printf("AviMediaReadStream::Seek() %d\n", pos);
    framepos_t chunk_id = (m_Header.dwSampleSize) ? find(pos * m_Header.dwSampleSize) : pos;
    if (chunk_id == ERR)
	return -1;

    m_uiChunk = chunk_id;
    m_uiPosition = (m_Header.dwSampleSize) ? m_Index[chunk_id].qwTimestamp : m_uiChunk;
    //ClearCache(); // ?? not needed - cache will do this when necessary

    //Debug cout << "AviMediaReadStream::Seek() p: " << pos << " c: " << chunk_id << "  time: " << m_uiPosition  / m_dAvgBytesPerSec << endl;

    return 0;
}

HRESULT AviMediaReadStream::SeekTime(double timepos)
{
    framepos_t p = (framepos_t)(timepos * m_dAvgBytesPerSec);
    return Seek(p);
}

framepos_t AviMediaReadStream::GetPos() const
{
    if (m_Header.dwSampleSize)
	return m_uiPosition / m_Header.dwSampleSize;
    return m_uiPosition;
}

HRESULT AviMediaReadStream::Read(uint_t lSamples, void *lpBuffer,
				 uint_t cbBuffer, uint_t *plBytes,
				 uint_t *plSamples)
{
    Debug printf("AviMediaReadStream::Read(%d, %p) chunk:%d  pos:%d\n",
		 lSamples, lpBuffer, m_uiChunk, m_uiPosition);

    int retval = 0;
    if (m_uiChunk >= m_Index.size() && m_uiChunk != ERR) // end of file
    {
	if (plBytes)
	    *plBytes = 0;
	if (plSamples)
	    *plSamples = 0;
	return -1;
    }

    if (!lpBuffer) // only query
    {
	if (plBytes)
	{
	    *plBytes = (m_Header.dwSampleSize) ?
		lSamples * m_Header.dwSampleSize : m_Index[m_uiChunk].GetChunkLength();
	}
	return 0;
    }

    uint_t read_samples = 0;
    uint_t read_bytes = 0;

    if (m_Header.dwSampleSize)
    {
	cbBuffer -= (cbBuffer % m_Header.dwSampleSize);
	if (cbBuffer < (lSamples * m_Header.dwSampleSize))
	    lSamples = cbBuffer / m_Header.dwSampleSize;
    }
    /*while((m_uiChunk<m_Index.size()) && (read_samples<lSamples) && (m_Index[m_uiChunk].dwChunkLength==0))
    {
	m_uiChunk++;
	read_samples++;
	}*/

    unsigned int chunklen;

    while (m_uiChunk < m_Index.size() && read_samples < lSamples
	   && ((!(chunklen = m_Index[m_uiChunk].GetChunkLength()))
               || (read_bytes < cbBuffer)))
    {
	//cout << "INREADLOOP  " << read_samples << "   "
	//    << lSamples << "  " << m_uiChunk << "  " <<  m_Index.size() << endl;
	int bytes;
	bool success;
	if (!m_Header.dwSampleSize)
	{ // usually video - but also all VBR audio tracks (MP3, Vorbis)
	    if (cbBuffer - read_bytes < chunklen)
		break;
	    if (chunklen)
	    {
		bytes = m_pParent->m_Input.read((char*)lpBuffer + read_bytes,
						m_iId, m_uiChunk, chunklen, success);
		if (!bytes)
		    break;
		if(success)
		    read_bytes += bytes;
		else
		    retval = -1;
	    }
	    else
		bytes = 0;

	    read_samples++;
	    m_uiPosition++;
	    m_uiChunk++;
	    continue;
	}
	else
	{ // audio
	    // stream init has to call seek(0)!!!
	    // if (m_uiPosition < m_Index[0].qwTimestamp)
	    //    m_uiPosition = m_Index[0].qwTimestamp;

	    int read_offset = m_uiPosition - m_Index[m_uiChunk].qwTimestamp;
	    uint_t bytes_to_read = chunklen - read_offset;
	    if (bytes_to_read > (cbBuffer - read_bytes))
		bytes_to_read = cbBuffer - read_bytes;
	    if (chunklen)
	    {
		//printf("READ %Ld   %d    %d  %d    %d\n", m_Index[m_uiChunk].qwTimestamp,
		//       m_Index[m_uiChunk].dwChunkLength, read_bytes, read_offset, bytes_to_read);
                if (read_offset >= 0)
		    bytes = m_pParent->m_Input.read((char*)lpBuffer + read_bytes,
						    m_iId, m_uiChunk,
						    bytes_to_read, success, read_offset);
		else bytes = 0;
		/*else
		{
		    printf("******************************\n");
		    bytes = -read_offset;
		    if (bytes > bytes_to_read)
                        bytes = bytes_to_read;
                    memset((char*)lpBuffer + read_bytes, 0, bytes);
                    read_bytes = bytes;
		}*/
		if (!bytes)
		    break;
		if (bytes < 0)
		    bytes = 0;
		read_bytes += bytes;
		read_samples = read_bytes / m_Header.dwSampleSize;
                m_uiPosition += bytes;
	    }
	    else
	    {
		bytes = 0;
		if (read_offset > 0)
		    break;
	    }

	    if (bytes == int(chunklen - read_offset))
		m_uiChunk++;
	}
    }

    //if (lpBuffer && read_bytes) printf("read_bytes  %d  sampe %d\n", read_bytes, read_samples);
    if (plBytes)
	*plBytes = read_bytes;
    if (plSamples)
	*plSamples = read_samples;

    return retval;
}

const void* AviMediaReadStream::ReadDirect(uint_t* plBytes, uint_t* plSamples)
{
    return 0;
}

framepos_t AviMediaReadStream::GetPrevKeyFrame(framepos_t pos) const
{
    if (pos == ERR)
	pos = GetPos();

    if (m_Header.dwSampleSize)
	return pos;

    if (pos >= m_Index.size() || pos == ERR)
	return 0;

    framepos_t inpos = pos;
    while (pos > 0)
	if (m_Index[--pos].IsKeyFrame())
	    break;

    return pos;
}

framepos_t AviMediaReadStream::GetNextKeyFrame(framepos_t pos) const
{
    if (pos == ERR)
	pos = GetPos();

    if (m_Header.dwSampleSize)
	return pos;

    if (pos == ERR)
	return GetPrevKeyFrame(pos);

    bool kf = false;
    while (pos < (m_Index.size() - 1))
    {
	if (m_Index[pos].IsKeyFrame())
	{
	    kf = true;
	    break;
	}
	pos++;
    }

    return (kf) ? pos : GetPrevKeyFrame(m_Index.size() - 1);
}

framepos_t AviMediaReadStream::GetNearestKeyFrame(framepos_t pos) const
{
    if (pos == ERR)
	pos = GetPos();

    if (m_Header.dwSampleSize)
	return pos;

    if (pos >= m_Index.size() || pos == ERR)
	return 0;

    if (m_Index[pos].IsKeyFrame())
	return pos;

    framepos_t prev = GetPrevKeyFrame(pos);
    framepos_t next = GetNextKeyFrame(pos);

    return (pos - prev < next - pos) ? prev : next;
}

double AviMediaReadStream::GetFrameTime() const
{
    return (double)m_Header.dwScale / m_Header.dwRate;
}

uint_t AviMediaReadStream::GetLength() const
{
    if ((m_Header.fccType != streamtypeAUDIO) || m_Header.dwSampleSize == 0)
	return m_Index.size();

    if (m_Index.size() == 0)
	return 0;

    const AVIINDEXENTRY2& end = m_Index.back();
    return end.qwTimestamp + end.GetChunkLength() / m_Header.dwSampleSize;
}

double AviMediaReadStream::GetLengthTime() const
{
    return GetSampleTime(GetLength());
}

StreamInfo* AviMediaReadStream::GetStreamInfo() const
{
    if (m_StreamInfo.m_p->m_dLengthTime == 0.0)
    {
	uint_t kfmax = 0;
	uint_t kfmin = ~0U;
	uint_t kfchunks = 0;
	int64_t kfsize = 0;
	uint_t fmax = 0;
	uint_t fmin = ~0U;
	uint_t chunks = 0;
	int64_t size = 0;

	for (unsigned i = 0; i < m_Index.size(); i++)
	{
	    uint_t l = m_Index[i].GetChunkLength();
	    if (m_Index[i].IsKeyFrame())
	    {
		kfmax = (kfmax > l) ? kfmax : l;
		kfmin = (kfmin < l) ? kfmin : l;
		kfsize += l;
                kfchunks++;
	    }
	    else
	    {
		fmax = (fmax > l) ? fmax : l;
		fmin = (fmin < l) ? fmin : l;
		size += l;
		chunks++;
	    }
	}

	m_StreamInfo.m_p->setKfFrames(kfmax, kfmin, kfchunks, kfsize);

	if (fmin > fmax)
            fmin = fmax; // usually no delta frames
	m_StreamInfo.m_p->setFrames(fmax, fmin, chunks, size);

	m_StreamInfo.m_p->m_dLengthTime = GetLengthTime();
	m_StreamInfo.m_p->m_iQuality = m_Header.dwQuality;

	if (m_Header.fccType == streamtypeVIDEO)
	{
	    BITMAPINFOHEADER* h = (BITMAPINFOHEADER*) m_pcFormat;
	    m_StreamInfo.m_p->setVideo(h->biWidth, h->biHeight);
	    m_StreamInfo.m_p->m_Type = StreamInfo::Video;
	    m_StreamInfo.m_p->m_uiFormat = h->biCompression;
	}
	else if (m_Header.fccType == streamtypeAUDIO)
	{
	    WAVEFORMATEX* w = (WAVEFORMATEX*) m_pcFormat;

	    m_StreamInfo.m_p->setAudio(w->nChannels, w->nSamplesPerSec,
				       w->wBitsPerSample, m_Header.dwSampleSize);
	    m_StreamInfo.m_p->m_Type = StreamInfo::Audio;
	    m_StreamInfo.m_p->m_uiFormat = w->wFormatTag;
	}
    }

    return new StreamInfo(m_StreamInfo);
}

double AviMediaReadStream::GetSampleTime(framepos_t pos) const
{
    if (pos == ERR)
	pos = GetPos();

    return pos / m_dAvgBytesPerSec;
}

uint_t AviMediaReadStream::GetFormat(void *pFormat, uint_t lSize) const
{
    if (pFormat)
	memcpy(pFormat, m_pcFormat, (lSize < m_uiFormatSize) ? lSize : m_uiFormatSize);

    return m_uiFormatSize;
}

double AviMediaReadStream::CacheSize() const
{
    return m_pParent->m_Input.cacheSize();
}

void AviMediaReadStream::ClearCache()
{
    m_pParent->m_Input.clear();
}
