#include "AviMediaReadHandler.h"
#include "AviMediaReadStream.h"
#include "utils.h"
#include "fourcc.h"
#include "utils.h"
#include <stdio.h>
#include <string.h>

#define __MODULE__ "MediaReadHandler"
#define Debug if(0)

// reset profiler status - just for debuging
//extern "C" void kprof_reset_stats(void);
//template vector<AviMediaReadStream>;
struct FileInfo {
    const char* fcc;
    const char* title;
} infos[] = {
    // here will follow various info about the movie file
    // comments here are copied from OpenDML specs by Matrox

    // Indicates where the subject of the file is archived
    { "IARL", "Archival Location" },
    // Lists the artist of the original subject of the file;
    // for example, "Michaelangelo."
    { "IART",  "Artist" },
    // Lists the name of the person or organization that commissioned
    // the subject of the file; for example "Pope Julian II."
    { "ICMS", "Commissioned" },
    // Provides general comments about the file or the subject
    // of the file. If the comment is several sentences long, end each
    // sentence with a period. Do not include new-line characters.
    { "ICMT", "Comments" },
    // Records the copyright information for the file; for example,
    // "Copyright Encyclopedia International 1991." If there are multiple
    // copyrights, separate them by semicolon followed by a space.
    { "ICOP", "Copyright" },
    // Specifies the date the subject of the file was created. List
    // dates in year-month-day format, padding one-digit months and days with
    // a zero on the left; for example, "1553-05-03" for May 3, 1553.
    { "ICRD", "Creation Date" },
    // Describes whether an image has been cropped and, if so, how it
    // was cropped; for example, "lower-right corner."
    { "ICRP", "Cropped" },
    // Specifies the size of the original subject of the file; for
    // example, "8.5 in h, 11 in w."
    { "IDIM", "Dimensions" },
    // Stores dots per inch setting of the digitizer used to
    // produce the file, such as "300."
    { "IDPI", "Dots Per Inch" },
    // Stores the of the engineer who worked on the file. If there are
    // multiple engineers, separate the names by a semicolon and a blank;
    // for example, "Smith, John; Adams, Joe."
    { "IENG", "Engineer" },
    // Describes the original work, such as "landscape,", "portrait,"
    // "still liefe," etc.
    { "IGNR", "Genre" },
    // Provides a list of keywords that refer to the file or subject of the
    // file. Separate multiple keywords with a semicolon and a blank;
    // for example, "Seattle, aerial view; scenery."
    { "IKEY", "Keywords" },
    // ILGT - Describes the changes in the lightness settings on the digitizer
    // required to produce the file. Note that the format of this information
    // depends on the hardware used.
    { "ILGT", "Lightness" },
    // IMED - Decribes the original subject of the file, such as
    // "computer image," "drawing," "lithograph," and so on.
    { "IMED", "Medium" },
    // INAM - Stores the title of the subject of the file, such as
    // "Seattle from Above."
    { "INAM", "Name" },
    // IPLT - Specifies the number of colors requested when digitizing
    // an image, such as "256."
    { "IPLT", "Palette Setting" },
    // IPRD - Specifies the name of title the file was originally intended
    // for, such as "Encyclopedia of Pacific Northwest Geography."
    { "IPRD", "Product" },
    // ISBJ - Decsribes the contents of the file, such as
    // "Aerial view of Seattle."
    { "ISBJ", "Subject" },
    // ISFT - Identifies the name of the software packages used to create the
    // file, such as "Microsoft WaveEdit"
    { "ISFT", "Software" },
    // ISHP - Identifies the change in sharpness for the digitizer
    // required to produce the file (the format depends on the hardware used).
    { "ISHP", "Sharpness" },
    // ISRC - Identifies the name of the person or organization who
    // suplied the original subject of the file; for example, "Try Research."
    { "ISRC", "Source" },
    // ISRF - Identifies the original form of the material that was digitized,
    // such as "slide," "paper," "map," and so on. This is not necessarily
    // the same as IMED
    { "ISRF", "Source Form" },
    // ITCH - Identifies the technician who digitized the subject file;
    // for example, "Smith, John."
    { "ITCH", "Technician" },

    // ISMP
    { "ISMP", "Time Code" },
    // IDIT
    { "IDIT", "Digitization Time" },

    { 0, 0 }
};

AviMediaReadHandler::AviMediaReadHandler(const char* pszFile) throw(FatalError)
    :m_Input(pszFile)
{
    m_Input.seek(0);
    checkDword(FOURCC_RIFF);
    /*int riff_chunk=*/
    m_Input.readDword();
    checkDword(formtypeAVI);

    uint_t movie_chunk = 0;
    bool index_chunk_found = false;
    uint32_t chunk_type;
    uint_t type, nextpos;
    uint_t chunk_size;
    while (!m_Input.eof())
    {
	chunk_type = m_Input.readDword();
	chunk_size = m_Input.readDword();
	nextpos = m_Input.pos() + chunk_size + (chunk_size & 1);
	switch(chunk_type)
	{
	case FOURCC_LIST:
	    if (chunk_size < 4)
	    {
		printf("Damaged Avi with chunk size %d detected!\n", chunk_size);
		break;//throw FATAL("Damaged AVI file");
	    }
	    type = m_Input.readDword();
	    switch(type)
	    {
	    case listtypeAVIMOVIE:
		movie_chunk = m_Input.pos();
	        m_Input.seek(nextpos);
		// readMovieChunk();
		break;
	    case listtypeAVIHEADER:
		continue;
	    case listtypeSTREAMHEADER:
		readStreamHeader();
		break;
	    default:
		break;
	    };
	    break;
	case ckidAVIMAINHDR:
	    readMainHeader(chunk_size);
	    break;
	case ckidAVINEWINDEX:
	    if (readIndexChunk(chunk_size, movie_chunk) == 0)
		index_chunk_found = true;
	    break;
	case RIFFINFO_IARL:
	case RIFFINFO_IART:
	case RIFFINFO_ICMS:
	case RIFFINFO_ICMT:
	case RIFFINFO_ICOP:
	case RIFFINFO_ICRD:
	case RIFFINFO_ICRP:
	case RIFFINFO_IDIM:
	case RIFFINFO_IDPI:
	case RIFFINFO_IENG:
	case RIFFINFO_IGNR:
	case RIFFINFO_IKEY:
	case RIFFINFO_ILGT:
	case RIFFINFO_IMED:
	case RIFFINFO_INAM:
	case RIFFINFO_IPLT:
	case RIFFINFO_IPRD:
	case RIFFINFO_ISBJ:
	case RIFFINFO_ISFT:
	case RIFFINFO_ISHP:
	case RIFFINFO_ISRC:
	case RIFFINFO_ISRF:
	case RIFFINFO_ITCH:
	case RIFFINFO_ISMP:
	case RIFFINFO_IDIT:
    	default:
	    break;
	};
	m_Input.seek(nextpos);
    }

    if (!index_chunk_found)
	reconstructIndexChunk(movie_chunk);

    for (unsigned i = 0; i < m_Streams.size(); i++)
    {
	char b[100];

	if (m_Streams[i].m_Header.fccType == streamtypeAUDIO)
	{
	    strncpy(b, avm_wave_format_name(m_Streams[i].m_Header.fccHandler), sizeof(b));
            b[sizeof(b) - 1] = 0;
	}
	else
	{
	    strncpy(b, (const char*)&m_Streams[i].m_Header.fccHandler, 4);
            b[4] = 0;
	}

	m_Streams[i].m_Index.resize(m_Streams[i].m_Index.size());

	char fcctype[4];
	avm_set_le32(fcctype, m_Streams[i].m_Header.fccType);

	printf("Stream %d: 0x%x (%.4s) : 0x%x (%s)   %d chunks (%.2fKB)\n",
	       i, m_Streams[i].m_Header.fccType, fcctype,
	       m_Streams[i].m_Header.fccHandler, b,
	       (int) m_Streams[i].m_Index.size(),
	       m_Streams[i].m_Index.size() * sizeof(AVIINDEXENTRY2) / 1024.0);

#if 0
	printf("Idx   Offset   Length  Timestamp Flags\n");
	AviMediaReadStream& stream = m_Streams[i];
	for (unsigned j = 0 ; j < stream.m_Index.size(); j++)
	    printf("%3d  %Ld  %d  %d %Ld\n", j, stream.m_Index[j].qwChunkOffset,
		   stream.m_Index[j].GetLength(), stream.m_Index[j].dwFlags,
		   stream.m_Index[j].qwTimestamp);
#endif

	//hack   interleaved format  -  iavs
	//if (m_Streams[i].m_Header.fccType == mmioFOURCC('i', 'a', 'v', 's'))
	//    m_Streams[i].m_Header.fccType = streamtypeVIDEO;

	m_Input.addStream(m_Streams[i].m_iId, m_Streams[i].m_Index);
    }
    m_Input.async();
}

AviMediaReadHandler::~AviMediaReadHandler()
{
}

HRESULT AviMediaReadHandler::GetHeader(void* pheader, uint_t size) const
{
    if (!pheader || (size < sizeof(MainAVIHeader)))
        return -1;
    memset(pheader, 0, size);
    memcpy(pheader, &m_MainHeader, sizeof(MainAVIHeader));
    return 0;
}

void AviMediaReadHandler::checkDword(uint32_t value) throw (FatalError)
{
    uint_t v = m_Input.readDword();
    if (v != value)
    {
	//printf("%.4s  %.4s\n", (char*)&v, (char*)&value);
	throw FATAL("Not an AVI file!");
    }
}

int AviMediaReadHandler::readMainHeader(uint_t data_size)
{
//    checkInt(ckidAVIMAINHDR);//avih
//    uint_t data_size=m_Input.readInt();
    data_size += (data_size & 1);
    uint_t read_size = sizeof(MainAVIHeader);

    memset(&m_MainHeader, 0, read_size);
    if (data_size > read_size)
	printf("WARNING: unexpected main AVI header size\n");
    else
	read_size = data_size;

    m_Input.read(&m_MainHeader, read_size);
    if (read_size < data_size)
        m_Input.seekCur(data_size - read_size);

#ifdef BIG_ENDIAN
    uint8_t* b = (uint8_t*) &m_MainHeader;
    m_MainHeader.dwMicroSecPerFrame	= avm_get_le32(b);
    m_MainHeader.dwMaxBytesPerSec	= avm_get_le32(b + 4);
    m_MainHeader.dwPaddingGranularity	= avm_get_le32(b + 8);
    m_MainHeader.dwFlags		= avm_get_le32(b + 12);
    m_MainHeader.dwTotalFrames		= avm_get_le32(b + 16);
    m_MainHeader.dwInitialFrames	= avm_get_le32(b + 20);
    m_MainHeader.dwStreams		= avm_get_le32(b + 24);
    m_MainHeader.dwSuggestedBufferSize	= avm_get_le32(b + 28);
    m_MainHeader.dwWidth		= avm_get_le32(b + 32);
    m_MainHeader.dwHeight		= avm_get_le32(b + 36);
    for (int i = 0; i < 4; i++)
	m_MainHeader.dwReserved[i]	= avm_get_le32(b + 40 + i * 4);
#endif

    PrintMainAVIHeader(m_MainHeader);

    return data_size + 8;
}

int AviMediaReadHandler::readStreamHeader() throw(FatalError)
{
    checkDword(ckidSTREAMHEADER);//strh

    AVIStreamHeader ash;
    uint_t read_size = sizeof(ash);
    uint_t data_size = m_Input.readDword();
    data_size += (data_size & 1);
    memset(&ash, 0, read_size);
    if (data_size > read_size)
	printf("WARNING: unexpected AVI stream header size\n");
    else
        read_size = data_size;

    m_Input.read(&ash, read_size);
    if (read_size < data_size)
        m_Input.seekCur(data_size - read_size);

#ifdef BIG_ENDIAN
    ash.fccType		= avm_get_le32(&ash.fccType);
    ash.fccHandler	= avm_get_le32(&ash.fccHandler);
    ash.dwFlags		= avm_get_le32(&ash.dwFlags);
    ash.wPriority 	= avm_get_le16(&ash.wPriority);
    ash.wLanguage 	= avm_get_le16(&ash.wLanguage);
    ash.dwInitialFrames	= avm_get_le32(&ash.dwInitialFrames);
    ash.dwScale		= avm_get_le32(&ash.dwScale);
    ash.dwRate		= avm_get_le32(&ash.dwRate);
    ash.dwStart		= avm_get_le32(&ash.dwStart);
    ash.dwLength	= avm_get_le32(&ash.dwLength);
    ash.dwSuggestedBufferSize = avm_get_le32(&ash.dwSuggestedBufferSize);
    ash.dwQuality 	= avm_get_le32(&ash.dwQuality);
    ash.dwSampleSize	= avm_get_le32(&ash.dwSampleSize);
    ash.rcFrame.left	= avm_get_le16(&ash.rcFrame.left);
    ash.rcFrame.right	= avm_get_le32(&ash.rcFrame.right);
    ash.rcFrame.top	= avm_get_le32(&ash.rcFrame.top);
    ash.rcFrame.bottom	= avm_get_le32(&ash.rcFrame.bottom);
#endif
    PrintAVIStreamHeader(ash);
    //if (m_Header.dwStart && m_Header.dwStart)
    //    cout << "Calc starttime " << m_Header.dwStart * (m_Header.dwRate / (double)m_Header.dwStart) << endl;

    checkDword(ckidSTREAMFORMAT);//strf

    data_size = m_Input.readDword();
    uint_t fsize = data_size;
    data_size += (data_size & 1);

    char* format = new char[data_size];
    m_Input.read(format, data_size);

    if (ash.fccType == streamtypeVIDEO)
    {
#ifdef BIG_ENDIAN
	BITMAPINFOHEADER* h = (BITMAPINFOHEADER*) format;
	h->biSize		= avm_get_le32(&h->biSize);
	h->biWidth		= avm_get_le32(&h->biWidth);
	h->biHeight		= avm_get_le32(&h->biHeight);
	h->biPlanes		= avm_get_le16(&h->biPlanes);
	h->biBitCount		= avm_get_le16(&h->biBitCount);
	h->biCompression	= avm_get_le32(&h->biCompression);
	h->biSizeImage		= avm_get_le32(&h->biCompression);
	h->biXPelsPerMeter	= avm_get_le32(&h->biXPelsPerMeter);
	h->biYPelsPerMeter	= avm_get_le32(&h->biYPelsPerMeter);
	h->biClrUsed		= avm_get_le32(&h->biClrUsed);
	h->biClrImportant	= avm_get_le32(&h->biClrImportant);
#endif
	//printf("format %x  %.4s  %d\n",
	//       s.m_Header.fccHandler, (char*)&s.m_Header.fccHandler, data_size);
	ash.dwSampleSize = 0;
    }
    else if (ash.fccType == streamtypeAUDIO)
    {
#ifdef BIG_ENDIAN
	WAVEFORMATEX* h = (WAVEFORMATEX*) format;
	h->wFormatTag		= avm_get_le16(&h->wFormatTag);
	h->nChannels		= avm_get_le16(&h->nChannels);
	h->nSamplesPerSec	= avm_get_le32(&h->nSamplesPerSec);
	h->nAvgBytesPerSec	= avm_get_le32(&h->nAvgBytesPerSec);
	h->nBlockAlign		= avm_get_le16(&h->nBlockAlign);
	h->wBitsPerSample      	= avm_get_le16(&h->wBitsPerSample);
	h->cbSize	      	= avm_get_le16(&h->cbSize);
#endif
	if (!ash.dwSampleSize && ash.dwLength > 400000)
	{
	    // ((*(ashort*)s.m_pcFormat == WAVE_FORMAT_MPEGLAYER3)
	    // only if we have a lot of chunks
	    // if the stream would be VBR then ther would be far
	    // less chunks
	    printf("Warning: setting SampleSize=1\n");
	    ash.dwSampleSize = 1;
	}
    }

#if 0
    if (ash.dwScale > 100)
        ash.dwScale = 9600;
    printf("out rate: %d    scale: %d\n", ash.dwRate, ash.dwScale);
#endif

    // format ptr is given to another object
    streamid_t id = m_Streams.size();
    m_Streams.push_back(AviMediaReadStream(this, ash, id, format, fsize));
    return 0;
}

/*
int AviMediaReadHandler::readMovieChunk()
{
    int moviem_uiChunk;
    while(!m_Input.eof())
    {
	if(m_Input.readInt()!=FOURCC_LIST)
	{
	    int chunk_size=m_Input.readInt();
	    if(chunk_size<0)throw FATAL("Damaged AVI file");
	    if(chunk_size%2) chunk_size++;
	    m_Input.seekCur(chunk_size);
	    continue;
	}
	int chunk_size=m_Input.readInt();
	if(chunk_size<4)throw FATAL("Damaged AVI file");
	if(chunk_size%2)chunk_size++;
	checkInt(listtypeAVIMOVIE);
	moviem_uiChunk=m_Input.pos();
	m_Input.seekCur(chunk_size-4);
	return moviem_uiChunk;
    }
    throw FATAL("Movie chunk not found");
}
*/
int AviMediaReadHandler::readIndexChunk(uint_t index_size, uint_t movie_chunk_offset)
{
//    if(m_Input.readInt()!=ckidAVINEWINDEX)
//	return -1;

    bool zero_based_offsets = false;
//    movie_chunk_offset-=4;

//    int index_size=m_Input.readInt();
//    if(index_size<0)return -1;
    int64_t positions[m_Streams.size()];

    for (unsigned i = 0; i < m_Streams.size(); i++)
	positions[i] = m_Streams[i].m_Header.dwStart;

    AVIINDEXENTRY entry;

    for(unsigned index_pos = 0; index_pos <= (index_size/sizeof(AVIINDEXENTRY))
	&& !m_Input.eof(); index_pos++)
    {
	AVIINDEXENTRY2 entry2;

	entry.ckid = m_Input.readDword();
	entry.dwFlags = m_Input.readDword();
	entry.dwChunkOffset = m_Input.readDword();
	entry.dwChunkLength = m_Input.readDword();

	if (entry.dwChunkLength > 10000000)
	{
	    printf("WARNING: Invalid index entry %d -- id: %d  offset: %d size:  %d\n",
		   index_pos, entry.ckid, entry.dwChunkOffset,
		   entry.dwChunkLength);
	    return reconstructIndexChunk(entry.dwChunkOffset);
	}

	uint_t id = StreamFromFOURCC(entry.ckid);
	if (id >= m_Streams.size())
	    continue;

	AviMediaReadStream& stream = m_Streams[id];

	if (entry.dwChunkOffset < (int) movie_chunk_offset)
	    zero_based_offsets = true;

	entry2.qwChunkOffset = entry.dwChunkOffset;
	if (zero_based_offsets)
	    entry2.qwChunkOffset += movie_chunk_offset - 4;

	entry2.dwChunkLength = entry.dwChunkLength;

	entry2.qwTimestamp = positions[id];
	//printf ("Id  %d    ps: %Ld   len:%d\n", id, entry2.qwTimestamp, entry2.dwChunkLength);
	if (stream.m_Header.dwSampleSize)
	    positions[id] += entry2.dwChunkLength;
	else
	    positions[id]++;  // video stream, one frame / chunk

	entry2.SetKeyFrame(entry.dwFlags & AVIIF_KEYFRAME);
	stream.m_Index.push_back(entry2);
    }

    //kprof_reset_stats();
    return 0;
}

int AviMediaReadHandler::reconstructIndexChunk(uint_t offset)
{
    m_Input.seek(offset);
    int64_t positions[m_Streams.size()];
    int isdivx[m_Streams.size()];

    printf("Reconstructing index...\n");
    // fixing KeyFrames for DivX movies by checking flag bit
    for (unsigned i = 0; i < m_Streams.size(); i++)
    {
	positions[i] = m_Streams[i].m_Header.dwStart;
	isdivx[i] = 0;

	if (m_Streams[i].m_Header.fccType == streamtypeVIDEO)
	    switch (m_Streams[i].m_Header.fccHandler)
	    {
	    case fccDIV3:
	    case fccdiv3:
	    case fccDIV4:
	    case fccdiv4:
	    case fccDIV5:
	    case fccdiv5:
	    case fccDIV6:
	    case fccdiv6:
	    case fccMP43:
	    case fccmp43:
	    case fccMP42:
	    case fccmp42:
	    case fccAP41:
		isdivx[i] = 1;
                break;
	    case fccDIVX:
	    case fccdivx:
	    case fccDIV1:
	    case fccdiv1:
	    case fccMP4S:
	    case fccmp4s:
	    case 0x4:
		isdivx[i] = 2;
	    }
    }

    // set when trying to reconstruct even very broken .avi files
    bool seriouslyBroken = false;
    while (!m_Input.eof())
    {
	//printf("pos: %d\n", m_Input.pos());
	uint_t ckid = m_Input.readDword();
	streamid_t id = StreamFromFOURCC(ckid);

	uint_t ckidt = ckid >> 16;
	if (id >= m_Streams.size())
	{
	    if (m_Input.eof())
		break;
	    m_Input.seekCur(-2);
            if (!seriouslyBroken)
		printf("Trying to reconstruct broken avi stream (could take a lot of time)\n");
	    seriouslyBroken = true;
	    continue;
	}
	AviMediaReadStream& stream = m_Streams[id];
	AVIINDEXENTRY2 entry2;
	bool keyframe = true;
	entry2.qwChunkOffset = m_Input.pos() - 4;
	uint32_t size = m_Input.readDword();
        // max size for frame will be 250MB
	if (size > ((seriouslyBroken) ? 200000U : 0xfffffffU))
	    continue;
	entry2.dwChunkLength = size;
	entry2.qwTimestamp = positions[id];
	//printf ("Id  %d    ps: %Ld   len:%d\n", id, entry2.qwTimestamp, entry2.dwChunkLength);
	if (size & 1)
	    size++;

	if (isdivx[id] && size)
	{
	    uint_t a = m_Input.readDword();

	    if (isdivx[id] == 1 && (a & 0x40))
	    {
		keyframe = false;
		//printf("key 0x%x  %d  %Ld  %Ld \t%d\n", a, keyframe, positions[id], entry2.qwChunkOffset, id);
	    }
	    else if (isdivx[id] == 2)
	    {
		// DivX4 keyframe detection
		//printf("Dword 0x%x\n", a);
		if (a != 0x00010000) // old OpenDivX
		{
		    if (a == 0xb6010000)
		    {
			// looks like new DivX frame
			//printf("Byte 0x%x\n", b);
			keyframe = ((m_Input.readByte() & 0xc0) == 0);
			// 0x00 Key frame,  0x40 P-frame
			size--;
		    }
		    else
			keyframe = false;
		}
	    }
	    size -= 4;
	}

	if (stream.m_Header.dwSampleSize) //audio stream, chunks contain variable number of frames
	    positions[id] += entry2.dwChunkLength;
	else //video stream, one frame / chunk
	    positions[id]++;

	entry2.SetKeyFrame(keyframe);
	//cout << "Size " << entry2.dwChunkLength << " offset " << entry2.qwChunkOffset << endl;
	stream.m_Index.push_back(entry2);
	m_Input.seekCur(size);
    }

    return 0;
}

void AviMediaReadHandler::PrintMainAVIHeader(const MainAVIHeader& hdr)
{
    char buffer[200];
    printf("MainAVIHeader: MicroSecPerFrame=%d MaxBytesPerSec=%d\n"
	   " PaddingGranularity=%d  Flags=[%s] TotalFrames=%d\n"
           " InitialFrames=%d  Streams=%d  SuggestedBufferSize=%d WxH=%dx%d\n",
	   hdr.dwMicroSecPerFrame, hdr.dwMaxBytesPerSec,

	   hdr.dwPaddingGranularity, GetAviFlags(buffer, hdr.dwFlags),
	   hdr.dwTotalFrames,

	   hdr.dwInitialFrames, hdr.dwStreams,
	   hdr.dwSuggestedBufferSize,
	   hdr.dwWidth, hdr.dwHeight);
}

void AviMediaReadHandler::PrintAVIStreamHeader(const AVIStreamHeader& hdr)
{
    char buffer[200];

    char sb[10];
    if (hdr.fccType == streamtypeVIDEO)
    {
	avm_set_le32(sb, hdr.fccHandler);
	sb[4] = 0;
    }
    else
	sprintf(sb, "0x%x", hdr.fccHandler);
    char ft[4];
    printf("AVIStreamHeader: FccType=%.4s FccHandler=%s Flags=[%s]\n"
	   " InitialFrames=%d Scale=%d Rate=%d Start=%d Length=%d\n"
	   " SuggestedBufferSize=%d Quality=%d SampleSize=%d"
	   " Rect l,r,t,b=%d,%d,%d,%d\n",
	   avm_set_le32(ft, hdr.fccType), sb,

	   GetAviFlags(buffer, hdr.dwFlags),
	   hdr.dwInitialFrames, hdr.dwScale,
	   hdr.dwRate, hdr.dwStart, hdr.dwLength,

	   hdr.dwSuggestedBufferSize, hdr.dwQuality, hdr.dwSampleSize,
	   hdr.rcFrame.left, hdr.rcFrame.right, hdr.rcFrame.top,
	   hdr.rcFrame.bottom);
}

char* AviMediaReadHandler::GetAviFlags(char* buffer, uint_t flags)
{
    sprintf(buffer, "%s%s%s%s%s%s ",
	    (flags & AVIF_HASINDEX) ? " HAS_INDEX" : "",
	    (flags & AVIF_MUSTUSEINDEX) ? " MUST_USE_INDEX" : "",
	    (flags & AVIF_ISINTERLEAVED) ? " IS_INTERLEAVED" : "",
            (flags & AVIF_TRUSTCKTYPE) ? " TRUST_CKTYPE" : "",
	    (flags & AVIF_COPYRIGHTED) ? " COPYRIGHTED" : "",
	    (flags & AVIF_WASCAPTUREFILE) ? " WAS_CAPTURED_FILE" : ""
	   );
    return buffer;
}

IMediaReadStream* AviMediaReadHandler::GetStream(fourcc_t fccType, streamid_t id)
{
    avm::vector<AviMediaReadStream>::iterator it;
    streamid_t match = 0;
    for (it = m_Streams.begin(); it != m_Streams.end(); it++)
	if (it->m_Header.fccType == fccType)
	{
	    if (match == id)
	    {
		//printf("GETHDR  %.4s   %.4s\n", (char*) &fccType, (char*)&it->m_Header.fccHandler);
		return &(*it); //safe for gcc3.0
	    }
	    match++;
	}
    return 0;
}

IMediaReadHandler *CreateAviMediaReadHandler(const char *pszFile) throw(FatalError)
{
    return new AviMediaReadHandler(pszFile);
}
