#include "AsfInputStream.h"
#include "AsfReadStream.h"
#include "AsfReadHandler.h"
#include "asf_guids.h"

#include <string.h>
#include <stdio.h>

using namespace std;

ASFReadStream::ASFReadStream(ASFReadHandler* parent)
    :m_pParent(parent), m_pSeekInfo(0), m_uiPosition(0),
    m_pIterator(0), m_iPointer(0), m_uiSubpointer(0), m_uiLastTimestamp(0),
    m_bValidBuf(false)
{
}

ASFReadStream::ASFReadStream(const ASFReadStream& c)
{
    operator=(c);
}

ASFReadStream& ASFReadStream::operator=(const ASFReadStream& c)
{
    m_Header = c.m_Header;
    m_pParent = c.m_pParent;
    m_iId = c.m_iId;
    m_uiPosition = c.m_uiPosition;
    m_pIterator = c.m_pIterator;
    if (m_pIterator)
	m_pIterator->addRef();
    m_iPointer = c.m_iPointer;
    m_uiSubpointer = c.m_uiSubpointer;
    m_uiLastTimestamp = c.m_uiLastTimestamp;
    m_pSeekInfo = c.m_pSeekInfo;
    m_bIsAudio = c.m_bIsAudio;
    m_bIsScrambled = c.m_bIsScrambled;
    m_pcScrambleBuf = c.m_pcScrambleBuf;
    m_bValidBuf = c.m_bValidBuf;
    if (m_bIsScrambled)
    {
    	char* tmp = (char*)(&m_Header.hdr.aud.wfex);
    	tmp += 18+m_Header.hdr.aud.wfex.cbSize;
	m_pScrambleDef = (const ASFAudioScrambleDef*)tmp;
	m_iScrOffset = c.m_iScrOffset;
    }
    return *this;
}

ASFReadStream::~ASFReadStream()
{
    if (m_pIterator)
	m_pIterator->release();
}

uint_t ASFReadStream::GetHeader(void* pheader, uint_t size) const
{
    if (!pheader)
	return sizeof(AVIStreamHeader);

    memset(pheader, 0, size);

    AVIStreamHeader* p = (AVIStreamHeader*) pheader;

    if (!m_pSeekInfo)
    {
	p->dwRate = 15;
	p->dwScale = 1;
	p->dwLength = 0x7FFFFFFF;
    }
    else
    {
	p->dwRate = 1000000;
	if (!m_pSeekInfo->size())
	    p->dwScale = 1000000/15;
	else
	{
	    double stream_length = m_pParent->m_Header.play_time/10000000.;
	    p->dwScale = int(1000000*stream_length/m_pSeekInfo->size());
	}
	p->dwLength = m_pSeekInfo->size();
    }

    if (m_bIsAudio)
    {
	p->fccType = streamtypeAUDIO;
	p->fccHandler = m_Header.hdr.aud.wfex.wFormatTag;
	p->dwSampleSize = m_Header.hdr.aud.wfex.nBlockAlign;
    }
    else if (GuidIsType(m_Header.hdr.uid_type, GUID_VIDEO_STREAM))
    {
	p->fccType = streamtypeVIDEO;
	p->fccHandler = m_Header.hdr.vid.biCompression;
	p->rcFrame.right = m_Header.hdr.vid.biWidth;
	p->rcFrame.top = m_Header.hdr.vid.biHeight;
    }
    return sizeof(AVIStreamHeader);
}

bool ASFReadStream::IsKeyFrame(framepos_t lFrame) const
{
    if (!m_pSeekInfo || m_bIsAudio)
	return true;

    framepos_t info_pos;

    if (lFrame != ERR)
    {
	if (lFrame >= m_pSeekInfo->size())
	    return true;
	info_pos = lFrame;
    }
    else
	info_pos = GetPos();

    return (info_pos != ERR) ? (*m_pSeekInfo)[info_pos].IsKeyFrame() : true;
}

framepos_t ASFReadStream::GetPos() const
{
    return (m_pSeekInfo) ? m_pSeekInfo->find(m_uiLastTimestamp) : 0;
}

framepos_t ASFReadStream::GetPrevKeyFrame(framepos_t lFrame) const
{
    if (!m_pSeekInfo)
	return 0;

    framepos_t info_pos;
    if (lFrame != ERR)
    {
	if (lFrame >= m_pSeekInfo->size())
	    return 0;
	info_pos = lFrame;
    }
    else
	info_pos = GetPos();

    return m_pSeekInfo->prevKeyFrame(info_pos);
}

framepos_t ASFReadStream::GetNextKeyFrame(framepos_t lFrame) const
{
    if (!m_pSeekInfo)
	return 0;

    framepos_t info_pos;

    if (lFrame != ERR)
    {
	if (lFrame >= m_pSeekInfo->size())
	    return 0;
	info_pos = lFrame;
    }
    else
	info_pos = GetPos();

    return m_pSeekInfo->nextKeyFrame(info_pos + 1);
}

framepos_t ASFReadStream::GetNearestKeyFrame(framepos_t lFrame) const
{
    if (!m_pSeekInfo)
	return 0;

    framepos_t info_pos;

    if (lFrame != ERR)
    {
	if (lFrame >= m_pSeekInfo->size())
	    return 0;
	info_pos = lFrame;
    }
    else
	info_pos = GetPos();

    return m_pSeekInfo->nearestKeyFrame(info_pos);
}

double ASFReadStream::GetFrameTime() const
{
    if (!m_pSeekInfo || !m_pSeekInfo->size())
	return 1.0/15.0;

    return GetLength()/m_pSeekInfo->size();
}

uint_t ASFReadStream::GetLength() const
{
    return (m_pSeekInfo) ? m_pSeekInfo->size() : 0x7FFFFFFF;
}

double ASFReadStream::GetLengthTime() const
{
    if (m_pSeekInfo && m_pSeekInfo->size() > 0)
	return (*m_pSeekInfo)[m_pSeekInfo->size() - 1].object_start_time / 1000.0;

    return m_pParent->m_Header.play_time / 10000000.0;
}

StreamInfo* ASFReadStream::GetStreamInfo() const
{
    if (!m_pSeekInfo)
    {
	printf("ASFReadStream::GetStreamInfo() no seek info\n");
        return 0;
    }
    if (m_StreamInfo.m_p->m_dLengthTime == 0.0)
    {
	uint_t kfmax = 0;
	uint_t kfmin = ~0U;
	uint_t kfchunks = 0;
	int64_t kfsize = 0;
	uint_t fmax = 0;
	uint_t fmin = ~0U;
	uint_t chunks = 0;
	int64_t size = 0;

	for (unsigned i = 0; i < m_pSeekInfo->size(); i++)
	{
	    uint_t l = (*m_pSeekInfo)[i].object_length;
	    if ((*m_pSeekInfo)[i].IsKeyFrame())
	    {
		kfmax = (kfmax > l) ? kfmax : l;
		kfmin = (kfmin < l) ? kfmin : l;
		kfsize += l;
                kfchunks++;
	    }
	    else
	    {
		fmax = (fmax > l) ? fmax : l;
		fmin = (fmin < l) ? fmin : l;
		size += l;
		chunks++;
	    }
	}

	m_StreamInfo.m_p->setKfFrames(kfmax, kfmin, kfchunks, kfsize);
	m_StreamInfo.m_p->setFrames(fmax, fmin, chunks, size);
	m_StreamInfo.m_p->m_dLengthTime = GetLengthTime();
    }

    return new StreamInfo(m_StreamInfo);
}

/**
    We assume that audio data is aligned by samples.
    We can restart reading audio from the middle of object,
    but only from fragment boundaries. It is necessary because
    objects may be quite large.
**/
HRESULT ASFReadStream::Read(uint_t lSamples, void* lpBuffer, uint_t cbBuffer,
			    uint_t* plBytes, uint_t* plSamples)
{
    if (!m_bIsScrambled || !lpBuffer)
	return ReadInternal(lSamples, lpBuffer, cbBuffer, plBytes, plSamples);

    uint_t sampCount = 0, bCount = 0;
    const unsigned short& samp_size = m_Header.hdr.aud.wfex.nBlockAlign;
    uint_t lSamples_actual = cbBuffer / samp_size;
    //cout << "lSampleIn " << lSamples << "  " << cbBuffer << "  " << samp_size << " ln " << lSamples_actual << endl;
    if (lSamples_actual < lSamples)
	lSamples = lSamples_actual;
    while (sampCount < lSamples)
    {
	if (!m_bValidBuf)
	{
	    if ((unsigned)m_iScrOffset >= m_pcScrambleBuf.size())
		m_iScrOffset = 0;

	    uint_t sampRead, bRead;
	    HRESULT res = ReadInternal((m_pcScrambleBuf.size() - m_iScrOffset)
				       / m_pScrambleDef->block_align_1,
				       &m_pcScrambleBuf[m_iScrOffset],
				       m_pcScrambleBuf.size() - m_iScrOffset,
				       &bRead, &sampRead);
	    if (bRead == 0)
		break;
	}

	m_bValidBuf = true;

	uint_t i = m_iScrOffset / m_pScrambleDef->block_align_1;
	for (; i<m_pcScrambleBuf.size()/m_pScrambleDef->block_align_1; i++)
	{
	    //cout << i << "   " << m_iScrOffset << "  " << sampCount << " ls " << lSamples << endl;
	    if (sampCount >= lSamples)
		break;

	    int cnt = m_pScrambleDef->block_size;
	    int row=i / cnt;
	    int col=i % cnt;
	    int idx=row + col * m_pScrambleDef->chunk_size / m_pScrambleDef->block_align_1;
	    //printf("%d bytes in buffer,  block align: %d,   lasttime: %d\n", m_pcScrambleBuf.size()-m_iScrOffset, m_pScrambleDef->block_align_1, m_uiLastTimestamp);
	    //printf("%d bytes in buffer,  block_size:  %d, offset: %d\n", m_pcScrambleBuf.size()-m_iScrOffset, m_pScrambleDef->chunk_size, m_iScrOffset);
	    //printf("%d: writing block %dx%d=%d as %d\n", i, row, col, idx, index);
	    memcpy(lpBuffer,
		   &m_pcScrambleBuf[idx * m_pScrambleDef->block_align_1],
		   m_pScrambleDef->block_align_1);
	    lpBuffer = (char*)lpBuffer + m_pScrambleDef->block_align_1;
	    sampCount++;
	    m_iScrOffset += m_pScrambleDef->block_align_1;
	    if ((unsigned)m_iScrOffset >= m_pcScrambleBuf.size())
	    {
		m_bValidBuf = false;
                break;
	    }
	}
        //break;
    }
    //cout << "bytes " << sampCount * samp_size << "  samples " << sampCount << endl;
    if (plBytes)
	*plBytes = sampCount * samp_size;
    if (plSamples)
	*plSamples = sampCount;
    //cout << " READSAMPLE " << lSamples << "  isaudio " << m_bIsAudio << endl;
    return 0;
}

HRESULT ASFReadStream::ReadInternal(uint_t lSamples, void *lpBuffer,
				    uint_t cbBuffer,
				    uint_t* plBytes, uint_t* plSamples)
{
    uint_t read_bytes = 0, read_samples = 0;
    uint_t buffer_required = 0;
    bool abort_process = false;
    const unsigned short& samp_size = m_Header.hdr.aud.wfex.nBlockAlign;

    if (m_bIsAudio)
    {
	//cout << "***********   READING PACKET " << lSamples << "  " << m_uiSubpointer << "  " << m_iPointer << "  cbBuf " << cbBuffer << endl;

	//if(samp_size)
	//	{
	uint_t lSamples_actual = cbBuffer/samp_size;
	if (lSamples_actual < lSamples)
	    lSamples = lSamples_actual;
	//cout << "cbUf " << cbBuffer << " sampsize " << samp_size << "  lact " << lSamples_actual << endl;

        cbBuffer = lSamples * samp_size;
//	}
//	else
//	    cout<<"WARNING: audio_stream with sample size = 0?"<<endl;
        memset(lpBuffer, 0, cbBuffer);
    }
    uint_t _position2 = 0;
    if (!lpBuffer)
    {
	// if we are only queried for buffer size,
	// we shouldn't change our global pointer
	_position2 = m_uiPosition;
	m_pIterator->undoPosition();
    }
    bool in_fragment = false;
    //uint_t fragment_start_packet=0;
    uint_t fragment_start_pos=0;
    uint_t fragment_read_bytes=0;
    uint_t object_read_size=0;
    if (m_iPointer + m_uiSubpointer)
    {
	if (m_iPointer)
	    in_fragment = true;
	object_read_size = m_iPointer;
	//read_bytes = m_uiSubpointer;
        //if (m_bIsAudio)
	//    cout << "SETTING OBJEC " << object_read_size << endl;
    }
    /**
	This code should be tolerant to most possible errors.
	It only returns less data than requested if it gets
	to the end of stream. In all errorneous situations which
	are possible during reassembly of packets, it simply skips
	suspicious data.
    **/

    int result = 0;

    while (((read_bytes <= cbBuffer) || !lpBuffer)
	  && (read_samples < lSamples) && !abort_process)
    {
	bool success;
	packet p = m_pIterator->getPacket(success);
	if (!success)
	{
	    //cout << "s: success " << success << "  pos: " << m_uiPosition << " rb: " << read_bytes << "  " << lSamples << endl;
	    result = -1;
	    break;
	}
	//cout << "Pos " << (int) p.fragments[m_uiPosition].stream_id
	//    << "  " << m_uiPosition << "   " << p.fragments.size() << endl;

	while (m_uiPosition < p.fragments.size())
	{
            //cout << "Pos " << (int) p.fragments[m_uiPosition].stream_id << endl;
	    //if (m_bIsAudio)
	    //    cout << "XXXXXXXXX " << object_read_size << endl;
	    if (p.fragments[m_uiPosition].stream_id != m_iId)
	    {
		m_uiPosition++;
		continue;
	    }

//	    if(reached<m_uiPosition){reached++; continue;}
	    const fragment& f = p.fragments[m_uiPosition];
	    m_uiPosition++;

	    //cout << " inf: " << in_fragment << "   pos: " << m_uiPosition << endl;
	    //if (m_bIsAudio)
	    //    cout << "datalen " << f.data_length << " objlen " << f.object_length << "  offset " << f.fragment_offset << endl;
	    if (f.data_length < f.object_length)//incomplete object
	    {
		if (lpBuffer && (f.data_length + read_bytes) > cbBuffer)
		{
		    // this fragment won't fit into the buffer
		    //  cout<<"WARNING: this fragment won't fit into the buffer"<<
		    //   "( buffer size "<<cbBuffer<<", read bytes "<<read_bytes<<
		    //   ", fragment data length "<<f.data_length
		    //   <<", position "<<m_uiPosition<<", packet timestamp "<<p.hdr.send_time/1000.0
		    //   <<endl;
		    abort_process = true;
		    //m_pIterator->forgetUndo();
		    m_uiPosition--;
		    break;
		}

		if (!in_fragment && f.fragment_offset)
		{
		    printf("WARNING: incomplete fragment found ( offset %d, length %d )"
			   ", packet timestamp %f\n", f.fragment_offset,
			   f.object_length, p.hdr.send_time / 1000.0);
	    	    read_bytes = fragment_read_bytes;
 		    in_fragment = false;
		    m_pIterator->forgetUndo();
                    result = -1;
		    continue;
		}

		if (in_fragment && f.fragment_offset != object_read_size)
		{
		    printf("WARNING: fragment with unexpected offset while"
			   " reassembling ( expected %d, found %d ), "
			   "packet timestamp %f\n", object_read_size,
			   f.fragment_offset, p.hdr.send_time/1000.0);
		    //m_uiPosition=fragment_start_pos;
		    //_chunk=fragment_start_packet;
		    read_bytes = fragment_read_bytes;
		    m_pIterator->forgetUndo();
		    in_fragment = false;
                    result = -1;
		    continue;
		}

		uint_t to_read = f.data_length - m_uiSubpointer;
		//cout << "to_read " << to_read << "   " << m_uiSubpointer
		//   << " data: " << f.data_length << endl;
		if (lpBuffer)
		    if (to_read > (cbBuffer - read_bytes))
			to_read = cbBuffer - read_bytes;

		if (in_fragment && (object_read_size + to_read > f.object_length))
		{
		    printf("WARNING: overflow while reassembling object ( reassembled "
			   "%d + to_read %d bytes, size %d bytes ), "
			   "packet timestamp %f\n", object_read_size, to_read,
			   f.object_length, p.hdr.send_time / 1000.0);
		    //m_uiPosition=fragment_start_pos;
		    //chunk=fragment_start_packet;
	    	    read_bytes = fragment_read_bytes;
 		    in_fragment = false;
		    m_pIterator->forgetUndo();
		    m_iPointer=0;
		    result = -1;
		    continue;
		}

		if (lpBuffer)
		{
		    //if (m_bIsAudio)
		    //    cout << "copy " << read_bytes << " len " << to_read << "  " << f.pointer << endl;
		    memcpy((char*)lpBuffer + read_bytes,
			   &p[f.pointer + m_uiSubpointer], to_read);
		}
		read_bytes += to_read;
		m_uiSubpointer += to_read;

		//cout << "sub " << m_uiSubpointer << "  " << f.data_length
		//    << "  " << read_bytes << "  to_read " << to_read << endl;

		if (m_uiSubpointer >= f.data_length)
		    m_uiSubpointer = 0;
		else
		    m_uiPosition--;

		if (read_bytes > buffer_required)
		    buffer_required = read_bytes;
	        object_read_size += to_read;

		if (m_bIsAudio)
		{
		    read_samples = read_bytes / samp_size;
		    //cout << "READSAMPLER " << read_samples << "   " << read_bytes << "   " << samp_size << endl;
		}
		if (in_fragment)
		{
		    if (object_read_size == f.object_length) // completed
		    {
			//cout << "packet completed!!! " << f.object_length << endl;
			in_fragment = false;
			m_pIterator->forgetUndo();
			if (!m_bIsAudio)
			    read_samples++;
		    }
		}
		else // just entered fragmented object
		{
                    in_fragment = true;
		    m_pIterator->undoPosition();
		    fragment_start_pos = m_uiPosition - 1;
		    fragment_read_bytes = read_bytes - to_read;
		    object_read_size = to_read;
		}
		m_uiLastTimestamp = f.object_start_time;// - m_pParent->m_Header.play_start_time;

		if (read_samples >= lSamples)
		{
		    //cout << "ABORTING  " << read_samples << "  " << lSamples << endl;
		    abort_process = true;
		    break;
		}
	    }
	    else //complete object
	    {
		if (in_fragment)
		{
		    printf("WARNING: found complete object while reassembling, "
			   "packet timestamp %f\n", p.hdr.send_time/1000.0);
	    	    read_bytes = fragment_read_bytes;
		    in_fragment = false;
		    m_pIterator->forgetUndo();
		    result = -1;
		}
		if (!m_bIsAudio && lpBuffer && (f.data_length>(cbBuffer-read_bytes)))
		// this fragment won't fit into the buffer
	        {
//		    cout<<"WARNING: this fragment won't fit into the buffer"<<
//		    "( buffer size "<<cbBuffer<<", read bytes "<<read_bytes<<
//		    ", fragment data length "<<f.data_length
//		    <<", position "<<m_uiPosition<<", packet timestamp "<<p.hdr.send_time/1000.
//		    <<endl;
	    	    abort_process=true;
		    m_uiPosition--;
		    break;
	        }
		int to_read = f.data_length - m_uiSubpointer;
		if (to_read < 0)
                    to_read = 0;
		if (lpBuffer)
		{
		    if ((unsigned)to_read > (cbBuffer - read_bytes))
			to_read = cbBuffer - read_bytes;

		    memcpy((char*)lpBuffer + read_bytes,
			   &p[f.pointer + m_uiSubpointer], to_read);
		}
		read_bytes += to_read;
		m_uiSubpointer += to_read;

		if (m_uiSubpointer >= f.data_length)
		    m_uiSubpointer = 0;
		else
		    m_uiPosition--;

		if (read_bytes > buffer_required)
		    buffer_required = read_bytes;

		if (m_bIsAudio)
		    read_samples = read_bytes/m_Header.hdr.aud.wfex.nBlockAlign;
		else
		    read_samples++;

		m_uiLastTimestamp = f.object_start_time;//-m_pParent->m_Header.play_start_time;

		if (read_samples>=lSamples)
		{
		    abort_process=true;
		    break;
		}
	    }
	}//while
	if (abort_process)
	    break;
	m_uiPosition=0;
	m_iPointer=0;
	m_pIterator->next();

    }//while

    if (in_fragment)
    {
	// we ran out of space in output buffer
	// while reassembling fragmented object
	if (!m_bIsAudio)
	{
	    m_uiPosition = fragment_start_pos;
	    m_pIterator->undo();
	    read_bytes = fragment_read_bytes;
	}
	else
	{
	    m_pIterator->undo();
	    m_iPointer = object_read_size;
	    //cout << "SETAUDIOPOINTER " << m_iPointer << endl;
	}
    }
    else if (m_bIsAudio)
        m_iPointer = 0;

    //if (m_bIsAudio)
    //    read_samples=read_bytes/m_Header.hdr.aud.wfex.nBlockAlign;

    if (plBytes)
	*plBytes = read_bytes;
    if (plSamples)
	*plSamples = read_samples;
    if (!lpBuffer)
    {
	// if we are only queried for buffer size,
	// we shouldn't change our global pointer
	m_uiPosition = _position2;
	if (plBytes)
	    *plBytes = buffer_required;
	m_pIterator->undo();
    }
    //Debug cout<<"Read(): read "<<read_bytes<<" bytes ( "<<read_samples<<" samples )   result: " << result << endl;
    //cout<<"Read(): read "<<read_bytes<<" bytes ( "<<read_samples<<" samples )   result: " << result << "      " << m_uiLastTimestamp << endl;
    //if (m_bIsAudio)
    //    cout<<"Read(): read "<<read_bytes<<" bytes ( "<<read_samples<<" samples )"<<endl;
    return result;
}

HRESULT ASFReadStream::Seek(framepos_t pos)
{
    chunk_info ch;
    if (m_pIterator->seek((int64_t)pos, &ch))
	return -1;
    m_uiPosition = ch.fragment_id;
    m_iPointer = m_uiSubpointer = 0;
    m_uiLastTimestamp = ch.object_start_time;

    // reset descrambling
    m_iScrOffset = 0;
    m_bValidBuf = false;

    return 0;
}


HRESULT ASFReadStream::SeekTime(double timepos)
{
    Debug printf("ASFReadStream::SeekTime() %f\n", timepos);
    assert(timepos >= 0);
    chunk_info ch;
    if (m_pIterator->seek(timepos, &ch))
	return -1;
    m_uiPosition = ch.fragment_id;
    m_iPointer = m_uiSubpointer = 0;
    m_uiLastTimestamp = ch.object_start_time;

    // reset descrambling
    m_iScrOffset = 0;
    m_bValidBuf = false;

    return 0;
}

double ASFReadStream::GetSampleTime(framepos_t lSample) const
{
    double t = m_uiLastTimestamp / 1000.0;
    if (lSample == ERR)
    {
	if (m_bIsAudio && m_iScrOffset > 0)
	{
	    const double a = m_iScrOffset/(double)m_Header.hdr.aud.wfex.nAvgBytesPerSec;
	    //cout << "t: " << a <<  "  offset: " << m_iScrOffset << "  " << m_Header.hdr.aud.wfex.nAvgBytesPerSec << endl;
            t += a;
	}
    }
    else if (m_pSeekInfo && lSample < m_pSeekInfo->size())
	t = (*m_pSeekInfo)[lSample].object_start_time / 1000.0;

    return t;
}

uint_t ASFReadStream::GetFormat(void *pFormat, uint_t lSize) const
{
    uint_t size;
    const void* fmt;
    if (m_bIsAudio)
    {
	size = m_Header.hdr.data_len;
	fmt = &m_Header.hdr.aud.wfex;
    }
    else
    {
	size = m_Header.hdr.data_len - 11;
	fmt = &m_Header.hdr.vid.biSize2;
    }

    if (pFormat)
	memcpy(pFormat, fmt, (size > lSize) ? lSize : size);
    return size;
}

fourcc_t ASFReadStream::GetFourCC() const
{
    if (GuidIsType(m_Header.hdr.uid_type, GUID_AUDIO_STREAM))
	return streamtypeAUDIO;
    else if (GuidIsType(m_Header.hdr.uid_type, GUID_VIDEO_STREAM))
	return streamtypeVIDEO;

    return 0;
}

uint_t ASFReadStream::GetSampleSize() const
{
    if (m_bIsAudio)
	return m_Header.hdr.aud.wfex.nBlockAlign;
    return 0;
}

void ASFReadStream::ClearCache()
{
//    return m_pParent->m_pInput->clear();
}

HRESULT ASFReadStream::SkipFrame()
{
    Debug printf("Skip frame\n");

    m_pIterator->next();

    return 0;
}

HRESULT ASFReadStream::SkipTo(double pos)
{
    if (pos < (m_uiLastTimestamp / 1000.0))
	return -1;

    for (;;)
    {
	bool success;
	packet p = m_pIterator->getPacket(success);

	if (!success)
	    return -1;
	if ((p.hdr.send_time/*-m_pParent->m_Header.play_start_time*/)/1000.0 > pos)
	    return 0;
	m_pIterator->next();
    }
}
