#include "aviplay_impl.h"
#include "cpuinfo.h"
#include "utils.h"

#include <unistd.h> //close
#include <stdlib.h> //exit
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h> //needed by sys/resource.h in FreeBSD
#include <sys/resource.h> //set/getpriority
#include <sys/ioctl.h>

#ifdef __NetBSD__
#include <sys/sched.h>
#else
#include <sched.h>
#endif

// has to go with mga_vid
#if	HAVE_SYS_IOCCOM_H
#include <sys/ioccom.h>
#endif
#include "../drivers/mga_vid.h"

#include <cstdio>
#include <ostream.h>

using namespace std;

// preffer video playing thread
const int PRIORITY_ADD_VIDEO = 16;
const int PRIORITY_ADD_AUDIO = 15;

extern int gotsigfpe;
extern int AVIPLAY_DEBUG;
#undef Debug
#define Debug if(AVIPLAY_DEBUG)

float AviPlayer::getVideoAsync()
{
    if (m_lTimeStart == 0)
    {
	m_dFrameStart = (!gotsigfpe && m_pVideostream) ? m_pVideostream->GetTime() : 0.;
	m_dLastFrameStart = m_dFrameStart;
	m_lTimeStart = longcount();
	Debug cout << "AviPlayer::getVideoAsync(): resetting" << endl;
    }

    double actual_time;
    if (m_pAudioRenderer && !m_pAudioRenderer->Eof())
    {
	actual_time = m_pAudioRenderer->GetTime();
	//cout << "     audiorender " <<  actual_time << "   " <</* m_pAudioRenderer->GetSampleTime() << */endl;
    }
    else
    {
	actual_time = m_dFrameStart + to_float(longcount(), m_lTimeStart);
	//cout<<"noaudio " << " "<<to_float(time_current, m_lTimeStart)<<" since "<<m_dFrameStart<<endl;
    }
    //Debug cout<<"AviPlayer::getVideoAsync(): actual_time "<<actual_time<<endl;

    double stream_time = (m_pVideostream && !gotsigfpe) ? m_pVideostream->GetTime() : actual_time;
    //cout << "stream time " << stream_time << endl;
    //Debug cout<<"AviPlayer::getVideoAsync(): stream_time "<<stream_time<<endl;
    //cout<<"Async "<<stream_time-actual_time<<"\tvstream "<<stream_time<<"\tastream "<<actual_time<<endl;
    //cout << m_dFrameStart << m_lTimeStart << endl;
    return stream_time - actual_time;
}

/*
 * check for synchronization event
 *
 * might block this task - it will await for broadcast from the main thread
 * returns: 0 - quit
 *          1 - ok could continue
 */
int AviPlayer::checkSync(ThreadId i)
{
    while (!m_bQuit)
    {
	//cout << "Thread: " << i  << " m_bHangup: " << m_iHangup << endl;
	if (m_bHangup)
	{
	    m_ThreadCond[i].Wait(m_ThreadMut[i]);
	    Debug cout << "Thread unlocked " << i << endl;
	}

	if (!m_bInitialized) // FIXME: remove me later
	{
	    Debug cout << "not initialized" << endl;
	    m_ThreadCond[i].Wait(m_ThreadMut[i], 0.2);
	    continue;
	}
	if (!IsOpened())
	{
	    Debug cout << "not opened" << endl;
	    m_ThreadCond[i].Wait(m_ThreadMut[i], 0.2);
	    continue;
	}
	if (!IsValid())
	{
	    Debug cout << "not valid" << endl;
	    return 0;
	}

	break;
    }

    return (!m_bQuit);
}

/*
 * block all running threads - implemented as recursive lock
 */
void AviPlayer::lockThreads(const char *name)
{
    Locker locker(m_LockMutex);

    if (++m_iLockCount > 1)
	return;

    m_bHangup = true;

    Debug cout << "Waiting for main_thread to hang up ("
	<< ((name) ? name : "???") << ")... " << flush;

    if (!gotsigfpe && m_pVideostream)
    {
	m_pVideostream->GetDecoder()->SetEnabled(false);
	m_ThreadMut[THREAD_DECODER].Lock();
	Debug cout << " decoder" << flush;
	m_pVideostream->GetDecoder()->SetEnabled(true);
	m_ThreadMut[THREAD_VIDEO].Lock();
	Debug cout << " video" << flush;
    }

    if (m_pAudioRenderer)
	m_pAudioRenderer->Wake();
    m_ThreadMut[THREAD_AUDIO].Lock();
    Debug cout << " audio" << flush;

    syncFrame();

    // now we have all threads locked
    m_bInitialized = false;

    Debug cout << " OK!" << endl;
}

/*
 * unlock all waiting threads in cond_wait state
 */
void AviPlayer::unlockThreads()
{
    Locker locker(m_LockMutex);

    if (--m_iLockCount > 0)
        return;

    // pausetest
    m_iLockCount = 0;
    m_lTimeStart = 0;
    m_bHangup = false;
    m_bInitialized = true;

    Debug cout << "Unlock threads" << endl;
    m_ThreadCond[THREAD_AUDIO].Broadcast();
    m_ThreadMut[THREAD_AUDIO].Unlock();

    if (!gotsigfpe && m_pVideostream)
    {
	m_ThreadCond[THREAD_VIDEO].Broadcast();
	m_ThreadMut[THREAD_VIDEO].Unlock();

	m_ThreadCond[THREAD_DECODER].Broadcast();
	m_ThreadMut[THREAD_DECODER].Unlock();
    }
}

static void gotsigfpe_handler(int hubba)
{
    if (!gotsigfpe)
	cout << "Got FATAL FPE Signal! - trying to continue with sound stream" << endl;

    gotsigfpe = true;

    // we can't just return from this place - as this would cause endless
    // loop here, so we will wait for another signal (break, kill)
    // end exit whole program
    pause();
    exit(0);
}

// returns true when some frames have been dropped
bool AviPlayer::dropFrame(bool leaveLocked)
{
    //cout << "Oj" <<  m_bVideoAsync << "   " << to_float(longcount(), m_lTimeStart) << endl;
    //cout << "ATime " << m_pAudioRenderer->GetTime() << endl;
    if (m_bVideoAsync || !m_bVideoDropping
	|| to_float(longcount(), m_lTimeStart) < 1.0
	|| m_DropMutex.TryLock() != 0)
	return false;

    bool ret = false;
    // here we try to eliminate all the frames
    // until next key-frame with reasonable close time
    framepos_t orig = m_pVideostream->GetPos();
    framepos_t kpos = orig;
    for (;;)
    {
	framepos_t npos = m_pVideostream->GetNextKeyFrame(kpos);
	double curtime = (m_pAudioRenderer) ? m_pAudioRenderer->GetTime() :
	    m_dFrameStart + to_float(longcount(), m_lTimeStart);
	double nt = m_pVideostream->GetTime(npos) - curtime;

	Debug cout << "AviPlayer::dropFrame() " << endl
	    << "  async  " << getVideoAsync()
	    << "  new_vtime: " << m_pVideostream->GetTime(npos)
	    << "  cur_atime: " << curtime
	    << "  diff " << nt << endl;
	if (nt > 0.1 || npos <= kpos || npos == m_pVideostream->ERR)
	    break;
	//cout << "skip2 " << (npos - kpos) << endl;
        framepos_t posbef = m_pVideostream->GetPos();

	m_bDropping = true;
	m_pVideostream->GetDecoder()->SetEnabled(false);
	m_ThreadMut[THREAD_DECODER].Lock();
	m_bDropping = false;
	kpos = m_pVideostream->SeekToNextKeyFrame();
	m_ThreadCond[THREAD_DECODER].Broadcast();
	m_pVideostream->GetDecoder()->SetEnabled(true);
	m_ThreadMut[THREAD_DECODER].Unlock();
	if (kpos != npos)
	{
	    Debug cout << "AviPlayer::dropFrame()  logical fault compare "
		<< npos << "   " << kpos << "   " << orig << "   " << m_pVideostream->GetPos() << "   before " << posbef << endl;
	    //abort();
	}
    }

    if (kpos > orig && kpos != m_pVideostream->ERR)
    {
	ret = true;
	// and finaly we we have keyframe which is
	// is not to ahead of audio
	Debug cout << "AviPlayer::dropFrame() skipped " << (kpos - orig)
	    << " frames  ( " << orig << "  " << kpos << " )" << endl;
	while (orig++ < kpos)
	{
	    m_Drop.insert(100.0);
	    m_iFrameDrop++;
	}
    }

    if (!ret)
    {
	double async = getVideoAsync();

	//cout << "DROPAVG " << m_Drop.average() << endl;
	// allow 4% frame dropout of decompressed frames
        // it's big waste of CPU to throw out already decompressed images
	if (async < -4.0 || (async < -0.15 && m_Drop.average() < 4))
	{
            if (async < -0.4)
		m_pVideostream->GetDecoder()->FlushCache();
	    CImage *im = m_pVideostream->GetFrame(!m_bVideoBuffered);
	    if (im)
	    {
		im->Release();
		m_Drop.insert(100.0);
		m_iFrameDrop++;
		Debug cout << "Dropped video frames: " << m_iFrameDrop
		    //<< "  a: " << ((m_pAudiostream) ? m_pAudiostream->GetTime() : 0.0)
		    << " atime: " << ((m_pAudioRenderer) ? m_pAudioRenderer->GetTime() : 0.0)
		    << " vtime: " << m_pVideostream->GetTime()
		    << endl;
		ret = true;
	    }
	}
    }
    if (0)
    {
	// old unused version
	// skip even frame decoding
	// but only non-keyframes
	int flags;
	m_pVideostream->GetFrameFlags(&flags);
	if (!flags)
	    m_pVideostream->SkipFrame();
    }
    //cout << "-----DROPDONE------" << endl;
    if (!leaveLocked || !ret)
	m_DropMutex.Unlock();

    return ret;
}

void* AviPlayer::videoThread()
{
    ThreadId th = THREAD_VIDEO;
    m_ThreadMut[th].Lock();
    Debug cout << "Thread video started" << endl;

    static const float NO_SLEEP = 1e9;
    float lastAsync = NO_SLEEP;

#ifdef SCHED_FIFO
    changePriority("Video player", 0, SCHED_FIFO);
#endif
    int mga_vid = open("/dev/mga_vid", O_RDONLY);
    if (mga_vid == -1)
	mga_vid = open("/dev/misc/mga_vid", O_RDONLY);

    float minWait = -0.03;
    float oversleep = 0.0;
    float lastCopyTime = 0.0;
    int64_t checkpoint = 0;
    bool sameFrameTime = true;

    if (mga_vid != -1)
    {
	if (ioctl(mga_vid, MGA_VID_IRQ_ON, 0) < 0)
	{
            // not supported
	    close(mga_vid);
            mga_vid = -1;
	}
        minWait = 0.01; // better precision with interrupt handler
        cout << "Using MGA_VID device for VBI (vsync) synchronization" << endl;
    }

    // video synchronization - NOTE: most of the lines here are
    // either comments or some debugging code - the alghorithm itself
    // it pretty simple and short
    while (checkSync(th))
    {
	//cout << "VTIME: " << m_pVideostream->GetTime() << "  p: "
	//    << m_pVideostream->GetPos()
	//    << "    " << m_pAudiostream->GetTime() << "    " << getVideoAsync()<< endl;
	if (gotsigfpe)
	{
	    m_ThreadCond[th].Wait(m_ThreadMut[th], 1.0);
            continue;
	}
	if (m_pVideostream->Eof())
	{
	    cout << "Video stream eof" << endl;
	    m_ThreadCond[th].Wait(m_ThreadMut[th], 1.0);
#if 1
	    bool autorepeat;
            Get(AUTOREPEAT, &autorepeat, 0);
	    if (autorepeat)
	    {
		m_DropMutex.Lock();
		if (m_pAudioRenderer)
		    m_pAudioRenderer->Reseek(0);
		m_pVideostream->Seek((framepos_t) 0);
		m_lTimeStart = 0;
		m_DropMutex.Unlock();
	    }
#endif
	    continue;
	}

	// wakeup decoder thread in case its sleeping
	m_ThreadCond[THREAD_DECODER].Broadcast();

	float async = getVideoAsync();
        m_fLastSleepTime = 0.0;
	//cout << "   ----ASYNC----       " << async << endl;

	if (lastAsync != async && async > minWait)
	{
	    lastAsync = async;
	    // also used for time in GetPos calculation!

	    if (mga_vid >= 0)
	    {
		int buf[3];
		read(mga_vid, &buf, sizeof(buf));
		// always wait for at least one VBI event
		//cout << buf[0] << " --- " << buf[1] << endl;
		continue;
	    }
	    else
	    {
		if (async > 0.5)
		    async = 0.5;        // max sleep time

                //async -= 0.01;
		// round to the nearest lower waiting time
                // 21ms .. 30ms -> 10ms -> will become 30ms
		//float asyncCalc = int((async + m_fLastSyncTime/2.0) * 100) / 100.0 - 0.001;
		//cout << lastCopyTime << endl;
                float asyncCalc;
		if (sameFrameTime)
		{
		    float x = to_float(longcount(), m_lLastTimeStart);
		    if (x > m_dVframetime) x = m_dVframetime;
		    else if (x < 0.0) x = 0.0;

		    asyncCalc = m_dVframetime - x - 0.011;// - m_fLastSyncTime;
		    if (async > 0.0)// (m_dVframetime - 0.01))
		    {
			asyncCalc += 0.01;
			//cout << "add " << async << "   " << asyncCalc << endl;
		    }
		    else if (async < -0.02)
		    {
			asyncCalc -= 0.01;
		    }
		}
		else
		    asyncCalc = async - 0.011;

		asyncCalc = int(asyncCalc * 100) / 100.0 + 0.008;

		if (asyncCalc > 0.008)
		{
		    int64_t t1 = longcount();
		    m_ThreadCond[th].Wait(m_ThreadMut[th], asyncCalc);
		    m_fLastSleepTime = to_float(longcount(), t1);
		    oversleep = (m_fLastSleepTime - asyncCalc);
		    if ((m_fLastSleepTime - asyncCalc) > 0.013)
		    {
			Debug cout << " Sleep too long!!!  real: " << m_fLastSleepTime  << "  expected: " << asyncCalc << "  " <<  asyncCalc << "  async: " << getVideoAsync() << "  " << oversleep << endl;
		    }
		}
		//else  cout << "nosleep" << endl;
		if (getVideoAsync() > 0.1)
		{
		    m_dVframetime -= m_fLastSleepTime;
		    m_lLastTimeStart = longcount();
		    continue; // we need to sleep some more
		}
	    }
	}
	if (lastAsync == NO_SLEEP)
	{
	    Debug cout << "no sleep - async: " << async << endl;
	}
        else
	    lastAsync = NO_SLEEP;

	//m_lLastTimeStart = longcount();
	//syncFrame();
	m_lLastTimeStart = longcount();
	m_dLastFrameStart = m_pVideostream->GetTime();

	bool dropLocked = false;
	//cout << "ASXXXX  " << async << "   " << GetAvgDrop() << endl;
        // aren't we getting out of sync??
	if (async < -0.07)
	    dropLocked = dropFrame(true);

	CImage *im = m_pVideostream->GetFrame(!m_bVideoBuffered);
	if (dropLocked)
	{
	    m_DropMutex.Unlock();
	    m_dLastFrameStart = m_pVideostream->GetTime();
	}

	//cout << "        spos " << m_pVideostream->GetPos() << endl;
	m_Drop.insert(0.0);
	double nf = m_pVideostream->GetTime() - m_dLastFrameStart;
        m_dVframetime -= nf;
	sameFrameTime = (m_dVframetime > -0.1 && m_dVframetime < 0.1);
	m_dVframetime = nf;

	if (!im)
	    // could be Hangup -> checkSync
	    // but also it might be 'still' image (zero delta frame)
            // thus we have to make normal time calculation
	    continue;

	if (m_bVideoMute)
	{
	    im->Release();
	    continue;
	}
	//cout << " ---------  newtime " << m_pVideostream->GetTime() << "\t\t" << m_dVframetime << endl;

	// draw this frame
	m_iFramesVideo++;
	m_Quality.insert(im->GetQuality() * 100.0);

	int64_t copys = longcount();
	// copy image to graphics card
	const subtitle_line_t* sl = GetCurrentSubtitles();
	for (unsigned i = 0; i < m_VideoRenderers.size(); i++)
	{
	    m_VideoRenderers[i]->Draw(im);
	    if (HasSubtitles())
		m_VideoRenderers[i]->DrawSubtitles(sl);
	    //m_VideoRenderers[i]->Sync(); // syncFrame() executed elsewhere
	}
	lastCopyTime = to_float(longcount(), copys); // copy time
	im->Release();
	m_bCallSync = true;
	syncFrame(); // syncing here - use CPU cache
    }

    if (mga_vid != -1)
    {
	ioctl(mga_vid, MGA_VID_IRQ_OFF, 0);
        close(mga_vid);
    }

    Debug cout << "Thread video finished" << endl;
    return 0;
}

void* AviPlayer::audioThread()
{
    ThreadId th = THREAD_AUDIO;

    m_ThreadMut[th].Lock();
    Debug cout << "Thread audio started" << endl;
    bool paused = false;

    changePriority("Audio decoder ", PRIORITY_ADD_AUDIO);

    while (checkSync(th))
    {
	if (!m_pAudioRenderer)
	{
	    m_ThreadCond[th].Wait(m_ThreadMut[th], 0.3);
            continue;
	}

	//cout << "Audiorenderer " << m_pAudioRenderer->Eof() << "  time " << m_pAudioRenderer->GetTime() << endl;
	if (m_bAudioMute || m_pAudioRenderer->Eof())
	{
	    if (!paused)
	    {
		m_pAudioRenderer->Pause(true);
		paused = true;
		cout << "Audio stream eof" << endl;
	    }
	    //m_pAudioRenderer->Stop();
	    m_ThreadCond[th].Wait(m_ThreadMut[th], 1.0);
            if (!m_pVideostream && m_pAudioRenderer->Eof())
	    {
		//cout << "Audio stream eof" << endl;
		bool autorepeat;
		Get(AUTOREPEAT, &autorepeat, 0);
		if (autorepeat)
		{
		    m_pAudioRenderer->Reseek(0);
		    if (paused)
		    {
                        paused = false;
			m_pAudioRenderer->Pause(paused);
		    }
		    //m_pAudioRenderer->Stop();
		    //m_pAudioRenderer->Start();
		}
	    }
	    continue;
	}

	double async = getVideoAsync();
	double buftime = m_pAudioRenderer->GetBufferTime();

	if (!m_bVideoAsync && to_float(longcount(), m_lTimeStart) > 1.0 && async < -1.0)
	{
	    if (!paused && async < -10.0)
	    {
		// ok even with dropping we could not stay in sync
		// so for a while turn of audio track and wait
                // until video will match audio time
		Debug cout << "Thread audio large async time: " << async << "  bt: " << buftime << endl;
		m_pAudioRenderer->Pause(true);
		paused = true;
                //cout << "pausing" << endl;
	    }
	    else if (buftime > 0.7 && m_pVideostream)
	    {
		//cout << "ADROP - buftime " << buftime << " - " << m_pAudioRenderer->bufferTime() << endl;
		if (dropFrame())
		    continue;
	    }
	}
        else if (paused && async > 0.02)
	    paused = (m_pAudioRenderer->Pause(false) != 0);

	//cout << "atime:" << async << "    btime:" << buftime << endl;
        int r = -1;
	if (buftime < 0.5)
	{
	    double t = (m_pVideostream) ? m_pVideostream->GetTime() : m_pAudioRenderer->GetTime();
            r = m_pAudioRenderer->Extract(t);
	}

	if (r != 0)
	    // either we have enough prebuffered sound or audio renderer
            // is unable to extrack new samples - wait for a while
	    m_ThreadCond[th].Wait(m_ThreadMut[th], 0.1);
    }

    Debug cout << "Thread audio finished" << endl;
    return 0;
}

void* AviPlayer::decoderThread()
{
    ThreadId th = THREAD_DECODER;

    m_ThreadMut[th].Lock();
    // thread started only with (stream != NULL)
    Debug cout << "Thread decoder started" << endl;

    changePriority("Video decoder ", PRIORITY_ADD_VIDEO);

    // we will leave the SIGFPE
    //cout << "Not installing SIGFPE handler..." << endl;
    //signal(SIGFPE, gotsigfpe_handler);

    if (geteuid() == 0)
        cout << "!!!Running video decoder as ROOT - Dangerous!!!" << endl;

    while (checkSync(THREAD_DECODER))
    {
        int r;
	if (!m_bVideoBuffered || m_bDropping ||
	    (r = m_pVideostream->ReadFrame()) < 0)
	{
	    //cout << "RR " << r << endl;
	    m_ThreadCond[th].Wait(m_ThreadMut[th], 1.0);
	}
    }

    Debug cout << "Thread decoder finished" << endl;
    return 0;
}

void AviPlayer::changePriority(const char* taskName, int addPriority, int schedt)
{

#if !defined(__NetBSD__) && !defined(__FreeBSD__)
    if (addPriority != 0)
    {
        if (!taskName)
            taskName = "";
#if 1
	int p = getpriority(PRIO_PROCESS, 0) + addPriority;
	if (p > 20)
	    p = 20;
	setpriority(PRIO_PROCESS, 0, p);
	cout << taskName << "thread lowered priority to "
	    << getpriority(PRIO_PROCESS, 0) << endl;
#else
	// this code doesn't work on Linux!
	// it looks like phtread priority is completely ignored and the only
        // way is to call 'setpriority' as you see above

	int policy;
	struct sched_param sched;
	pthread_getschedparam(pthread_self(), &policy, &sched);
	sched.sched_priority += addPriority;
	pthread_setschedparam(pthread_self(), policy, &sched);
	cout << taskName << " using priority: " << sched.sched_priority << endl;
#endif
    }

    if (schedt != 0 && m_iEffectiveUid == 0)
    {
        // we could use FIFO scheduler with root access privilegies
        seteuid(m_iEffectiveUid);
        setegid(m_iEffectiveGid);

	static const char *sched_txt[] = {
	    "SCHED_OTHER", "SCHED_FIFO", "SCHED_RR"
	};
	struct sched_param schedRec;

	if (sched_getparam(0, &schedRec) != 0) {
	    perror("sched_getparam");
	    return;
	}

	schedRec.sched_priority = sched_get_priority_min(schedt);

	if (sched_setscheduler(0, schedt, &schedRec) != 0) {
	    perror("sched_setscheduler");
	    return;
	}

	cout << "Video thread - using scheduler: "
	    << sched_txt[sched_getscheduler(0)] << endl;
	setuid(getuid());
        setgid(getgid());
    }
#endif // __NetBSD__  & __FreeBSD__

}

void AviPlayer::syncFrame()
{
    if (!gotsigfpe && m_pVideostream && !m_bVideoMute && m_bCallSync)
    {
	m_bCallSync = false;

	double newAudio = 0;
        double async = 0;

	// we will measure time difference between frame and
        // we try to elimineta  50,30,50ms jumps

        int64_t newsync = longcount();
	Debug
	{
	    // just for checking if these two diffs are correct
	    // for SdlAudioRender they are now matching!!
	    // Oss & Sun have to be fixed
	    newAudio = (m_pAudioRenderer) ? m_pAudioRenderer->GetTime() : 0.0;
	    async = getVideoAsync();
	}

	// blit image on screen
	for (unsigned i = 0; i < m_VideoRenderers.size(); i++)
	    m_VideoRenderers[i]->Sync();
	// splited into two parts so we do not create jumping images
	// even in the debug mode
	int64_t newVideo = longcount();
	float newDiff = to_float(newVideo, m_lLastVideoSync);
        m_fLastSyncTime = to_float(newVideo, newsync);
	Debug
	{
	    // after check of ahead might be more presice time for this
	    // however we would have to play some with pausing
	    // so we currently place the image right  here
	    //double d = to_float(newVideo, lastVideo);
	    //if (d < 0.027 || d > 0.043)
	    printf("A-V sync: %+8.5f   sleep: %4.1f  Vd: %4.1f  Ad: %4.1f  VSync: %4.1f\n",
		   async,
		   m_fLastSleepTime * 1000.0,
		   to_float(newVideo, m_lLastVideoSync) * 1000.0,
		   (newAudio - m_dLastAudioSync) * 1000.0,
		   m_fLastSyncTime * 1000.0
		  );

	    float d = newDiff - m_fLastDiff;
	    if (d < 0)
                d = -d;

	    if (d > 0.014)
		printf("Larger diff between frames: %f\n", d);
	    m_dLastAudioSync = newAudio;
	}

	m_lLastVideoSync = newVideo;
	m_fLastDiff = newDiff;
    }
}


// from video sync - just in case I would need some tests....
#if 0
                // code to filter differencies in frame duration
                // estimate duration of current frame
		float estim = to_float(longcount(), m_lLastTimeStart) + asyncCalc + 0.01;
		float d = m_dVframetime - estim;
		//cout << "estim " << estim << "   diff " << d << endl;
		if (d > 0.015)// || (m_fLastDiff - estim) > 0.02)
		{
		    //cout << "XXXXXXXXX+++ " << estim << "    " << m_fLastDiff << "  " << d << "   " << (m_fLastDiff - estim) << endl;
		    asyncCalc += 0.01;
                    estim += 0.01;
		}
		else if (d < -0.015)
		{
		    //cout << "YYYYYYYYYY--- " << estim << "    " << m_fLastDiff << "  " << d << "   " << (m_fLastDiff - estim) << endl;
		    asyncCalc -= 0.01;
		    estim -= 0.01;
		}
#endif

#if 0
		// this code should eliminate time discontinuities
                // however it fails completely with autoquality
                double vtrlimit = int(m_dVframetime * 100 + 0.9) /  100.0 + 0.007;
		cout << "estim " << estim  << "   " << asyncCalc << "   " << m_dVframetime << "   " << vtrlimit << endl;
		if ((m_fLastDiff - estim) > 0.015)
		{
		    asyncCalc += 0.011;
		}
		else if (estim > vtrlimit)
                    asyncCalc -= 0.01;
		//cout << "estim " << estim << "     " << m_fLastDiff << "   "
		//    << m_dVframetime << "   w: " << asyncCalc << "   " << async << endl;
#endif

#if 0
		    // show some stats for  sleep delays

		    // it shows that 0.029ms produces delays 0.040
		    // with much bigger probability than 0.028ms
                    // that's why we add 0.9 to round asyncCalc
		    static int tstcnt= 20;
		    static float tsttm = 0.020;

		    if (tstcnt-- < 0)
		    {
			tstcnt = 20;
                        tsttm += 0.001;
		    }
		    asyncCalc = tsttm;
#endif
