#include "aviplay_impl.h"
#include "configfile.h"
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>

const avm::string AviPlayer::regName = "AviPlayer";
const int AviPlayer::DEFAULT_0 = 0;
const int AviPlayer::DEFAULT_1 = 1;
const int AviPlayer::DEFAULT_RATE = 44100;
const int AviPlayer::DEFAULT_VOLUME = 1000;

// first in the list is default
const char* AviPlayer::videoRendererTxt =
#ifdef USE_SDL
    "SDL_XV,SDL_X11,"
#endif
    "novideo";

const char* AviPlayer::audioRendererTxt =
#ifdef USE_SDL
    "SDL,"
#endif
#ifdef OSS_SUPPORT
    "OSS,"
#endif
#ifdef SUNAUDIO_SUPPORT
    "SUN,"
#endif
    "noaudio";

const char* AviPlayer::subtitleRendererTxt =
#ifdef HAVE_XFT
    "XFT_DRAW,"
#endif
#ifndef WITHOUT_X
    "X11_DRAW,"
#endif
    "nosubtitles";

const AviPlayer::AviPlayerProperties AviPlayer::propertyList[] =
{
    // FIXME convert rest of flags later
    // text, type, defvalue, minval, maxval
    { "property_end" },
    { "AudioRenderer", RA_STRING, audioRendererTxt },
    { "VideoRenderer", RA_STRING, videoRendererTxt },
    { "SubtitleRenderer", RA_STRING, subtitleRendererTxt },
    { "VideoCodecs", RA_STRING, "" },
    { "AudioCodecs", RA_STRING, "" },
    { "UseYUV", RA_BOOL, &DEFAULT_1 },
    { "Autorepeat", RA_BOOL, &DEFAULT_1 },
    { "VideoStream", RA_INT, DEFAULT_0 },
    { "AudioStream", RA_INT, &DEFAULT_0, 0, 32768 },
    { "SubtitleFont", RA_STRING, "adobe-times" },
    { "SubtitleAsync_ms", RA_INT, &DEFAULT_0 },
    { "UseHttpProxy", RA_BOOL, &DEFAULT_0 },
    { "HttpProxy", RA_STRING, "" },
    { "PreserveAspect", RA_BOOL, &DEFAULT_1 },
    { "VideoBuffered", RA_BOOL, &DEFAULT_1 },
    { "VideoDirect", RA_BOOL, &DEFAULT_1 },
    { "VideoDropping", RA_BOOL, &DEFAULT_1 },
    { "QualityAuto", RA_BOOL, &DEFAULT_1 },
    { "DisplayFramePos", RA_BOOL, &DEFAULT_1 },
    { "AudioResampling", RA_BOOL, &DEFAULT_0 },
    { "AudioResamplingRate", RA_INT, &DEFAULT_RATE, 1, 100000 },
    { "AudioPlayingRate", RA_INT, &DEFAULT_RATE, 1, 100000 },
    { "AudioMasterTimer", RA_BOOL, &DEFAULT_0 },
    { "AudioVolume", RA_INT, &DEFAULT_VOLUME, 0, 10000 }, // 1000 is 1.0
    { "Async_ms", RA_INT, &DEFAULT_0 },
    { "AudioUrl", RA_STRING, "." },

};

HRESULT AviPlayer::Set(...)
{
    va_list args;
    Property prop;

    va_start(args, this);

    while ((prop = (Property) va_arg(args, int)) != 0)
    {
	int int_param = 0;
	char* string_param = 0;

	if (prop < LAST_PROPERTY)
	{
	    //cerr << "Property processing: " << propertyList[prop].text << endl;
	    switch (propertyList[prop].varType)
	    {
	    case RA_INT:
	    case RA_BOOL:
		int_param = va_arg(args, int);
		//cout << "INT " << int_param << endl;
		break;
	    case RA_STRING:
		string_param = va_arg(args, char*);
                //cout << "STR " << string_param << endl;
		break;
	    }

	    switch (prop)
	    {
	    case AUDIO_RENDERER:
		break;
	    case VIDEO_RENDERER:
		break;
	    case USE_YUV:
		setUseYUV(int_param);
		break;
	    case VIDEO_QUALITY_AUTO:
		setVideoQualityAuto(int_param);
		break;
	    case VIDEO_BUFFERED:
		setVideoBuffered(int_param);
		break;
	    case VIDEO_DIRECT:
		setVideoDirect(int_param);
		break;
	    case VIDEO_DROPPING:
		m_bVideoDropping = int_param;
		break;
	    case ASYNC_TIME_MS:
		m_fAsync = int_param / 1000.0;
		if (m_pAudioRenderer)
		    m_pAudioRenderer->SetAsync(m_fAsync);
		break;
	    case SUBTITLE_FONT:
                setFont(string_param);
		break;
	    case SUBTITLE_ASYNC_TIME_MS:
                m_fSubAsync = int_param / 1000.0;
		break;
	    case AUDIO_VOLUME:
		m_fVolume = int_param / 1000.0;
		if (m_pAudioRenderer)
		    m_pAudioRenderer->SetVolume(m_fVolume);
		break;
	    case AUDIO_URL:
		setAudioUrl(string_param);
		break;
	    case VIDEO_STREAM:
		lockThreads("SetVideoStream");
                //delete m_pVideostream;
		//m_pVideostream = m_pClip->GetStream(0, IAviReadStream::Video);
		unlockThreads();
		break;
	    case AUDIO_STREAM:
		lockThreads("SetAudioStream");
		setAudioStream(int_param);
		unlockThreads();
                break;
	    default:
		//cout << "Unknown (?unsupported?) propety " << prop << endl;
		break;
	    }

	    switch (propertyList[prop].varType)
	    {
	    case RA_INT:
	    case RA_BOOL:
		Registry::WriteInt(regName, propertyList[prop].text, int_param);
		break;
	    case RA_STRING:
		Registry::WriteString(regName, propertyList[prop].text, string_param);
		break;
	    }
	}
    }
    va_end(args);

    return 0;
}

HRESULT AviPlayer::Get(...) const
{
    va_list args;
    Property prop;

    //printf("Sizeof %d\n", sizeof(Property));
    va_start(args, this);
    while ((prop = (Property) va_arg(args, int)) != 0)
    {
	bool ok = true;
        bool isBool = false;
	avm::string string_param = "_empty_";
	int int_param = 0;

	if (prop < LAST_PROPERTY)
	{
	    //printf("Property %d  %s  (readcnt: %d)\n", prop, propertyList[prop].text, propertyRead[prop]);
	    Property force = prop;

	    isBool = (propertyList[prop].varType == RA_BOOL);

	    if (!propertyRead[prop])
		force = LAST_PROPERTY; // force reading from registry

	    switch (force)
	    {
	    case VIDEO_BUFFERED:
                int_param = m_bVideoBuffered;
		break;
	    case VIDEO_DROPPING:
		int_param = m_bVideoDropping;
		break;
	    case VIDEO_DIRECT:
		int_param = m_bVideoDirect;
                break;
	    case VIDEO_QUALITY_AUTO:
		int_param = m_bQualityAuto;
		break;
	    case ASYNC_TIME_MS:
                int_param = int(m_fAsync * 1000.0);
		break;
	    case SUBTITLE_ASYNC_TIME_MS: // float
		int_param = int(m_fSubAsync * 1000.0);
		break;
	    case AUDIO_VOLUME: // float
		int_param = int(m_fVolume * 1000.0);
		break;

	    default:
		//printf("Default propety: %s (idx = %d)\n", propertyList[prop].text, prop);
		switch (propertyList[prop].varType)
		{
		case RA_BOOL:
		case RA_INT:
		    int_param = Registry::ReadInt(regName,
						  propertyList[prop].text,
						  *(const int*)propertyList[prop].defValue);
		    break;
		case RA_STRING:
		    string_param = Registry::ReadString(regName,
							propertyList[prop].text,
							(const char*)propertyList[prop].defValue);
                    break;
		}
		propertyRead[prop]++;
		break;
	    }
	}
	else
	{
	    switch (prop)
	    {
	    case QUERY_AVG_QUALITY:
		int_param = int(m_Quality.average());
		break;
	    case QUERY_AVG_DROP:
		int_param = int(m_Drop.average());
		break;
	    case QUERY_AUDIO_RENDERERS:
		string_param = audioRendererTxt;
                break;
	    case QUERY_VIDEO_RENDERERS:
		string_param = videoRendererTxt;
                break;
	    case QUERY_SUBTITLE_RENDERERS:
		string_param = subtitleRendererTxt;
                break;
	    case QUERY_EOF:
                isBool = true;
		int_param = (m_pVideostream) ? m_pVideostream->Eof() : true;
		//printf("VSTREAM %d\n", int_param);
		//  audiostream doesn't detect eof - have to use Length FIXME
		if (int_param)
		    int_param = (m_pAudiostream) ? m_pAudiostream->Eof()
			|| m_pAudiostream->GetTime() >= m_pAudiostream->GetLengthTime() : true;
		//printf("???EOF %d %d  %f  %f\n", int_param, m_pAudiostream->Eof(),
		//       m_pAudiostream->GetTime(), m_pAudiostream->GetLengthTime());
		break;
	    case QUERY_VIDEO_WIDTH:
		{
                    int w = m_iWidth, h;
		    if (m_VideoRenderers.size())
			m_VideoRenderers[0]->GetSize(w, h);
                    int_param = w;
		}
                break;
	    case QUERY_VIDEO_HEIGHT:
		{
		    int w, h = m_iHeight;
		    if (m_VideoRenderers.size())
			m_VideoRenderers[0]->GetSize(w, h);
		    int_param = h;
		}
                break;
	    case QUERY_VIDEO_STREAMS:
		int_param = (m_pClip) ? m_pClip->VideoStreamCount() : 0;
		break;
	    case QUERY_AUDIO_STREAMS:
                if (m_pClipAudio)
		    int_param = m_pClipAudio->AudioStreamCount();
		else
                    int_param = (m_pClip) ? m_pClip->AudioStreamCount() : 0;
		break;
	    default:
		printf("Unexpected property value: %d\n", prop);
                abort();
		break;
	    }
	}

	int* ptr = va_arg(args, int*);
	//printf("Ptr:  %p\n", ptr);
	if (ok && ptr)
	{
	    if (isBool)
		*(bool*)ptr = (bool) int_param;
	    else
	    {
		*ptr = (string_param == "_empty_")
		    ? int_param : (int) strdup(string_param.c_str());
	    }
	}
    }
    va_end(args);

    return 0;
}

int AviPlayer::setVideoQualityAuto(bool enabled)
{
    if (m_bQualityAuto != enabled)
    {
	m_bQualityAuto = enabled;
	if (m_pVideostream)
	{
	    IVideoDecoder* vd = m_pVideostream->GetDecoder();
	    if (vd)
	    {
		IVideoDecoder::DecodingMode mode = IVideoDecoder::DIRECT;
		if (m_bVideoBuffered)
		    if (m_bQualityAuto)
			mode = IVideoDecoder::BUFFERED_QUALITY_AUTO;
		    else
			mode = IVideoDecoder::BUFFERED;
		vd->SetDecodingMode(mode);
	    }
	}
    }

    return 0;
}

int AviPlayer::setVideoDirect(bool enabled)
{
    if (m_bVideoDirect != enabled)
    {
	lockThreads("SetVideoDirect");
	if (m_VideoRenderers.size() > 0)
	{
	    m_bVideoDirect = (m_pVideostream && m_pVideostream->GetDecoder()
			      && m_pVideostream->GetDecoder()->SetDirectMemoryAccess((enabled) ? m_VideoRenderers[0] : 0) == 0);
	    printf("Direct mode is %s\n",
		   (m_bVideoDirect) ? "enabled" : "disabled");
	}
	unlockThreads();
    }

    return 0;
}

int AviPlayer::setVideoBuffered(bool enabled)
{
    if (m_bVideoBuffered != enabled)
    {
	m_bVideoBuffered = enabled;
	lockThreads("SetVideoBuffered");

	IVideoDecoder::DecodingMode mode = IVideoDecoder::DIRECT;

	if (m_bVideoBuffered)
	    if (m_bQualityAuto)
		mode = IVideoDecoder::BUFFERED_QUALITY_AUTO;
	    else
		mode = IVideoDecoder::BUFFERED;

	if (m_pVideostream && m_pVideostream->GetDecoder())
            m_pVideostream->GetDecoder()->SetDecodingMode(mode);// == 0)

	unlockThreads();
    }
    return 0;
}

int AviPlayer::setAudioStream(int channel)
{
    IAviReadStream* astream = 0;
    while (!astream && channel >= 0)
    {
        if (m_pClipAudio)
	    astream = m_pClipAudio->GetStream(channel--, IAviReadStream::Audio);
	else if (m_pClip)
	    astream = m_pClip->GetStream(channel--, IAviReadStream::Audio);
    }

    if (astream != m_pAudiostream)
    {
	double pos = 0;
	if (m_pAudioRenderer)
	{
	    pos = m_pAudioRenderer->GetTime();
	    delete m_pAudioRenderer;
	    m_pAudioRenderer = 0;
	}
        if (m_pAudiostream)
	    m_pAudiostream->StopStreaming();
	// NOTE: never call delete on   m_pAudiostream !
	m_pAudiostream = astream;
	if (astream)
	{
	    createAudioRenderer();
	    if (m_pAudioRenderer)
	    {
		m_pAudioRenderer->Start();
		m_pAudioRenderer->Reseek(pos);
	    }
	}
    }
    return 0;
}

int AviPlayer::setAudioUrl(const char* filename)
{
    lockThreads("SetAudioUrl");

    IAviReadFile* newClipAudio = 0;

    try
    {
	newClipAudio = CreateIAviReadFile(filename);
    }
    catch (FatalError& e)
    {
        e.Print();
    }

    if (newClipAudio)
    {
	delete m_pClipAudio;
        m_pClipAudio = newClipAudio;
	int def_audio;
	Get(AUDIO_STREAM, &def_audio, 0);

        setAudioStream(def_audio);
    }

    unlockThreads();
    return 0;
}
