#ifndef AVIPLAY_IMPL_H
#define AVIPLAY_IMPL_H

#include "aviplay.h"
#include "avifile.h"
#include "videodecoder.h"
#include "audiodecoder.h"
#include "renderer.h"
#include "except.h"
#include "Statistic.h"
#include "IAudioRenderer.h"
#include "Locker.h"

/***************************************************

Here aviplay interface goes

****************************************************/


class AviPlayer: public IAviPlayer2
{
public:
    AviPlayer(const char* filename, int _depth, const char*subname, const char* vcodec=0, const char* acodec=0) throw(FatalError);
    ~AviPlayer();
    virtual bool IsOpened() const
    {
	return (m_pClip) ? m_pClip->IsOpened() : true;
    }

    virtual bool IsValid() const
    {
	return (m_pClip) ? m_pClip->IsValid() : false;
    }

    virtual bool IsRedirector() const
    {
	return (m_pClip) ? m_pClip->IsRedirector() : false;
    }

    virtual bool GetURLs(avm::vector<avm::string>& urls) const
    {
	return (m_pClip) ? m_pClip->GetURLs(urls) : false;
    }

    HRESULT SetColorSpace(fourcc_t csp, bool test_only);

    virtual int GetWidth() const {return m_iWidth;}
    virtual int GetHeight() const {return m_iHeight;}
    virtual void SetKillHandler(KILLHANDLER handler, void* arg=0)
    {
	m_pKillhandler=handler;
	m_pKillhandlerArg=arg;
    }
    virtual void SetAudioFunc(AUDIOFUNC func, void* arg=0)
    {
	m_pAudiofunc=func;
	m_pAudiofuncArg=arg;
    }
    virtual double GetPos() const;
    virtual framepos_t GetFramePos() const;

    virtual bool IsStopped() const;
    virtual bool IsPlaying() const;
    virtual bool IsPaused() const;
    virtual void Restart();
    virtual double Reseek(double pos);
    virtual HRESULT ReseekExact(double pos);
    virtual void Start();
    virtual void Stop();
    virtual HRESULT PageUp();
    virtual HRESULT PageDown();
    virtual HRESULT NextFrame();
    virtual void Play();
    virtual void Pause(bool state);
    virtual const subtitle_line_t* GetCurrentSubtitles();
    virtual double GetLengthTime() const;
    virtual double GetAudioLengthTime() const
    {
	return (m_pAudioRenderer) ? m_pAudioRenderer->GetLengthTime() : 0;
    }
    virtual double GetVideoLengthTime() const
    {
	return (m_pVideostream) ? m_pVideostream->GetLengthTime() : 0;
    }
    virtual const char* GetAudioFormat() const
    {
	return (m_pAudioRenderer) ? m_pAudioRenderer->GetAudioFormat() : 0;
    }
    virtual StreamInfo* GetAudioStreamInfo() const;

    virtual const char* GetVideoFormat() const
    {
	if (!m_pVideostream)
	    return "";
	BITMAPINFOHEADER bh;
	m_pVideostream->GetVideoFormatInfo(&bh, sizeof(bh));
	const CodecInfo* ci=CodecInfo::match(bh.biCompression, CodecInfo::Video);

	return (ci) ? ci->GetName() : "unknown";
    }
    virtual StreamInfo* GetVideoStreamInfo() const;
    virtual double GetFps() const
    {
	return (m_pVideostream) ? 1.0 / m_pVideostream->GetFrameTime() : 0.0;
    }
    virtual const char* GetFileName() const
    {
	return m_Fn.c_str();
    }
    virtual const avm::vector<VideoRenderer*>& GetVideoRenderers() const;
    virtual HRESULT SetVideoRenderers(avm::vector<VideoRenderer*>);
    virtual State GetState(double* percent);
    virtual bool HasSubtitles() const;
    virtual HRESULT InitSubtitles(const char* filename);
    virtual HRESULT Set(...);
    virtual HRESULT Get(...) const;
    virtual void SetAsync(float async);
    virtual float GetAsync() const { return m_fAsync; }
    virtual const CodecInfo& GetCodecInfo() const
    {
	return m_pVideostream->GetDecoder()->GetCodecInfo();
    }
    virtual IRtConfig* GetRuntimeConfig() const { return dynamic_cast<IRtConfig*>(m_pVideostream->GetDecoder()); }

    virtual void SetVideo(bool enabled) { m_bVideoMute = enabled; }
    virtual void SetAudio(bool enabled) { m_bAudioMute = enabled; }
    virtual bool GetVideo() const { return m_bVideoMute; }
    virtual bool GetAudio() const { return m_bAudioMute; }

    virtual int Refresh() { return -1; }
    virtual int Resize(int& new_w, int& new_h) { return -1; }
    virtual int ToggleFullscreen(bool maximize = false) { return -1; }

protected:
    enum var_t
    {
	RA_BOOL, RA_INT, RA_STRING
    };

    struct AviPlayerProperties
    {
	const char* text;
	var_t varType;
	const void* defValue;
	int min;
        int max;
    };
    static const int DEFAULT_0;
    static const int DEFAULT_1;
    static const int DEFAULT_RATE;
    static const int DEFAULT_VOLUME;
    static const avm::string regName;
    static const char* videoRendererTxt;
    static const char* audioRendererTxt;
    static const char* subtitleRendererTxt;
    static const AviPlayerProperties propertyList[];

    mutable int AviPlayer::propertyRead[LAST_PROPERTY]; // have we read at least once from Registry::

    avm::vector<VideoRenderer*> m_VideoRenderers; // we could draw image to more places
    IAudioRenderer* m_pAudioRenderer; // not sure about the sound - this will be
                                      // more complicated because of buffer sizes
    KILLHANDLER m_pKillhandler;
    void* m_pKillhandlerArg;
    AUDIOFUNC m_pAudiofunc;
    void* m_pAudiofuncArg;
    IAviReadFile* m_pClip;
    IAviReadFile* m_pClipAudio;
    IAviReadStream* m_pVideostream;
    IAviReadStream* m_pAudiostream;
    Statistic m_Drop;
    Statistic m_Quality;
    avm::string m_Fn;
    avm::string m_Sn;
    avm::string m_vcodec;
    avm::string m_acodec;
    subtitles_t* m_pSubtitles;	// list of subtitles
    subtitle_line_t* m_pSubline;// contains current subtitle line

    bool m_bVideoMute;
    bool m_bAudioMute;
    bool m_bVideoAsync;		// true - do not care about a-v sync
    bool m_bQualityAuto;
    bool m_bVideoBuffered;
    bool m_bVideoDirect;
    bool m_bVideoDropping;
    bool m_bDropping;           // internal flag for temporal stop of decoder

    bool m_bPaused;		// true if we are paused
    bool m_bInitialized;	// true if we are playing OR paused
    bool m_bQuit;		// true signals to all processing threads
				// to terminate

    bool m_bHangup;		// signals main_thread to enter 'waiting' state
				// and set initialized=0
    bool m_bBuffering;

    bool m_bCallSync;           // marks that video update is needed



    float m_fVolume;		// current volume level
    float m_fAsync;		// a-v async time
    float m_fSubAsync;
    int m_iFramesVideo;		// just counters
    int m_iFrameDrop;
    int m_iLockCount;

    int m_iWidth;
    int m_iHeight;
				// these vars are used in syncing video with time
    int64_t m_lTimeStart;	// timestamp of moment when we started playing
				// updated after each reseek or pause
    int64_t m_lLastVideoSync;	// stamp for last video time in sync
    int64_t m_lLastTimeStart;	// timestamp of the last displayed frame
    double m_dFrameStart;	// precise position of video at time_start ( needed
				// for timing of video-only streams )
    double m_dLastFrameStart;	// precise position of video at last frame
    double m_dLastAudioSync;	// stamp for last audio time in sync
    float m_fLastDiff;          // difference between succesive
    float m_fLastSyncTime;      // length of the last Video sync call
    float m_fLastSleepTime;
    PthreadTask* m_VideoThread;	// performs video output and sync
    PthreadTask* m_AudioThread;	// performs audio output and sync
    PthreadTask* m_DecoderThread; // performs video decompression and caching

    uint_t m_iEffectiveUid;	// effective user id
    uint_t m_iEffectiveGid;	// effective group id

    enum ThreadId
    {   // do not changed order
	THREAD_FIRST = 0,
	THREAD_DECODER = THREAD_FIRST, THREAD_AUDIO, THREAD_VIDEO,
	THREAD_LAST,
    };

    PthreadMutex m_ThreadMut[THREAD_LAST]; // 3 mutexes
    PthreadCond m_ThreadCond[THREAD_LAST]; // 3 conditions
    PthreadMutex m_DropMutex;
    PthreadMutex m_LockMutex;

    double m_dVideoSeekDest;
    double m_dVframetime;
    int m_iDepth;
    static void* constructThread(void* arg);
    void* constructThreadfunc();
    void construct();
    void lockThreads(const char *name = 0);
    void unlockThreads(void);
    int checkSync(ThreadId);
    int drawFrame();
    void syncFrame();
    void changePriority(const char* taskName, int add, int schedt = 0);

    void* videoThread();
    void* audioThread();
    void* decoderThread();

    // special way to have function  compatible with C code pthread parameter
    // we could use only one parameter and this seems to be simpliest way
    static void* startVideoThread(void* arg);
    static void* startAudioThread(void* arg);
    static void* startDecoderThread(void* arg);

    float getVideoAsync();
    bool dropFrame(bool leaveLocked = false);

    virtual int setVideoQualityAuto(bool enabled);
    virtual int setVideoBuffered(bool enabled);
    virtual int setVideoDirect(bool enabled);
    virtual int setUseYUV(bool enabled) { return -1; }
    virtual int setAudioStream(int channel);
    virtual int setAudioUrl(const char* filename);
    virtual int setFont(const char* fn) { return -1; }
    void createAudioRenderer();
};

#endif // AVIPLAY_IMPL_H
