/********************************************************

	AVI player object
	Copyright 2000 Eugene Kuznetsov  (divx@euro.ru)

*********************************************************/

#include "aviplay_impl.h"

#ifdef USE_SDL
#include "SdlAudioRenderer.h"
#endif

#ifdef OSS_SUPPORT
#include "OssAudioRenderer.h"
#endif

#ifdef SUNAUDIO_SUPPORT
#include "SunAudioRenderer.h"
#endif

#include "CopyAudioRenderer.h"

#include "fourcc.h"
#include "cpuinfo.h"
#include "utils.h"
#include "version.h"

//#include <iomanip>
#include <iostream>

#include <pthread.h> /// just for init thread - soon to be removed
#include <unistd.h> // geteuid
#include <string.h> //memcpy
#include <stdlib.h> // getenv, free()
#include <stdio.h>

using namespace std;
#include <math.h> // abs(double)

#define __MODULE__ "IAviPlayer"

int AVIPLAY_DEBUG = 0;
int gotsigfpe = 0;
#undef Debug
#define Debug if(AVIPLAY_DEBUG)

#undef min
#define min(X,Y) ((X)<(Y)?(X):(Y))

AviPlayer::AviPlayer(const char* filename, int depth, const char* subname, const char* vcodec, const char* acodec) throw(FatalError)
    :m_Drop("Drop", 50), m_Quality("Quality", 25)
{
    m_pVideostream = 0;
    m_pAudiostream = 0;
    m_bInitialized = false;
    m_bHangup = false;
    m_bPaused = false;
    m_bQuit = false;
    m_bCallSync = false;
    m_bDropping = false;
    m_iLockCount = 0;
    m_lTimeStart = 0;
    m_dFrameStart = 0;
    m_dLastFrameStart = 0;
    m_iFramesVideo = 0;
    m_iFrameDrop = 0;
    m_pClip = 0;
    m_pClipAudio = 0;
    m_fLastSyncTime = 0.0;
    m_pAudiofunc = 0;
    m_pAudioRenderer = 0;
    m_fAsync = 0.0;
    m_fSubAsync = 0.0;
    m_fVolume = 1.0;
    m_dVframetime = 0.04;
    m_Fn = filename;
    m_Sn = subname ? subname : "";
    m_iDepth = depth;
    m_pKillhandler = 0;
    m_pSubtitles = 0;
    m_pSubline = 0;

    m_VideoThread = m_AudioThread = m_DecoderThread = 0;

    m_bQualityAuto = false;
    m_bVideoBuffered = false;
    m_bVideoDirect = false;

    if(vcodec && strlen(vcodec))
	m_vcodec = vcodec;
    if(acodec && strlen(acodec))
	m_acodec = acodec;
	
    memset(propertyRead, 0, sizeof(propertyRead));

    // remember effective ids at the creation time of the object
    // and use the real user id until we really need the euid
    m_iEffectiveUid = geteuid();
    m_iEffectiveGid = getegid();
    if (getuid() != m_iEffectiveUid)
	seteuid(getuid());
    if (getgid() != m_iEffectiveGid)
	setegid(getgid());

    m_bAudioMute = (getenv("AVIPLAY_MUTE_AUDIO") != 0) ? true : false;
    m_bVideoMute = (getenv("AVIPLAY_MUTE_VIDEO") != 0) ? true : false;

    m_bVideoAsync = (getenv("AVIPLAY_VIDEO_ASYNC") != 0) ? true : false;

    AVIPLAY_DEBUG = (getenv("AVIPLAY_DEBUG")   != 0) ? 1 : 0;

    Debug cout << "Debug is on " << AVIPLAY_DEBUG << endl;
    if (m_bVideoAsync)
	Debug cout << "Video is running asynchronously" << endl;
    try
    {
	if (m_pClip == 0)
	    m_pClip = CreateIAviReadFile(filename);
    }
    catch (FatalError& e)
    {
	e.Print();
	throw;
    }

    if (m_pClip->IsOpened())
    {
	if (!m_pClip->IsValid())
	    throw FATAL("Invalid file");
	construct();
    }
    else
    {
	pthread_t th;
	pthread_create(&th, 0, constructThread, this);
    }
}

IAviPlayer::~IAviPlayer()
{
}

AviPlayer::~AviPlayer()
{
    if (gotsigfpe)
    {
	cout << "Quiting whole program as SIGFPE has been caught!" << endl;
        exit(0);
    }

    Stop();

    m_bQuit = true;
    if (IsRedirector())
	return;
    while (!m_bInitialized)
	unlockThreads();

    if (m_pVideostream)
    {
	delete m_VideoThread;
	if (m_pAudioRenderer)
	    delete m_AudioThread;

	IVideoDecoder* vd = m_pVideostream->GetDecoder();
	if (vd)
	{
	    vd->SetEnabled(false);
	    vd->FlushCache();
	}
	delete m_DecoderThread;
	m_pVideostream->StopStreaming();
    }

    while (m_VideoRenderers.size() > 0)
    {
	VideoRenderer* r = m_VideoRenderers.back();
	m_VideoRenderers.pop_back();
	delete r;
    }

    if (m_pAudioRenderer)
	delete m_pAudioRenderer;

    if (m_pAudiostream)
	m_pAudiostream->StopStreaming();

    if (m_pClipAudio)
    {
	cout << "Closing audio clip" << endl;
	delete m_pClipAudio;
    }

    if (m_pClip)
    {
	cout << "Closing clip" << endl;
	delete m_pClip;
    }
    if (m_iFramesVideo)
	cout<<"Played "<<m_iFramesVideo<<" video frames ( "
	    << m_iFrameDrop*100./m_iFramesVideo << "% drop )" << endl;
    if (m_pSubtitles)
	subtitle_close(m_pSubtitles);
    if (m_pSubline)
        subtitle_line_free(m_pSubline);
}

void* AviPlayer::constructThread(void* arg)
{
    AviPlayer* a = (AviPlayer*)arg;
    return a->constructThreadfunc();
}

void* AviPlayer::constructThreadfunc()
{
    while (!m_pClip->IsOpened() && !m_bQuit)
	avm_usleep(100000);

    if (m_bQuit || !m_pClip->IsValid())
	return 0;

    construct();

    return 0;
}

void AviPlayer::construct()
{
    if (IsRedirector())
    {
	cout<<"Redirector"<<endl;
	return;
    }

    int def_audio, asynctm, avol;
    Get(DEFAULT_AUDIO_STREAM, &def_audio,
	ASYNC_TIME_MS, &asynctm,
	AUDIO_VOLUME, &avol,
	0);
    m_fAsync = asynctm / 1000.0;
    m_fVolume = avol / 1000.0;

    while (!m_pAudiostream && def_audio >= 0)
	m_pAudiostream = m_pClip->GetStream(def_audio--, IAviReadStream::Audio);

    bool vqa;
    Get(VIDEO_QUALITY_AUTO, &vqa,
	VIDEO_DROPPING, &m_bVideoDropping,
	VIDEO_BUFFERED, &m_bVideoBuffered,
	VIDEO_DIRECT, &m_bVideoDirect,
	0);

    // Initialize video
    setVideoQualityAuto(vqa);
    m_iWidth = 0;
    m_iHeight = 0;
    if (!m_bVideoMute && m_iDepth)
    {
	m_pVideostream = m_pClip->GetStream(0, IAviReadStream::Video);
	if (!m_pVideostream)
	    cerr<<"Videostream not detected" << endl;
	else
	    try
	{
	    m_dVframetime = m_pVideostream->GetFrameTime();

	    // reading subtitles
	    if (m_Sn.size())
		InitSubtitles(m_Sn.c_str());
	    else if (m_Fn.size() >= 4
		     && ((strncasecmp(m_Fn.c_str() + m_Fn.size() - 4, ".avi", 4) == 0)
			 || (strncasecmp(m_Fn.c_str() + m_Fn.size() - 4, ".asf", 4) == 0)))
	    {
		avm::string subname(m_Fn.c_str(), m_Fn.size() - 4);
		InitSubtitles(subname.c_str());
	    }

	    int result = m_pVideostream->StartStreaming(m_vcodec.size() ? m_vcodec.c_str() : 0);
	    if (result != 0)
		throw FATAL("Failed to initialize decoder object");
	    //cout << "Video Depth " << m_iDepth << endl;
	    // prefer this format
	    // remove 0 to get YUY2 rendering
	    BITMAPINFOHEADER bh;
	    m_pVideostream->GetVideoFormatInfo(&bh, sizeof(bh));
	    m_iWidth = bh.biWidth;
	    m_iHeight = labs(bh.biHeight);
	}
	catch (FatalError& e)
	{
	    e.Print();
	    cerr<<"No video will be available"<<endl;
	    m_pVideostream = 0;
	}
    } else
    {
	cerr<<"Video disabled"<<endl;
    }

    // Initialize audio - thread from SDL is initialized here
    if (m_pAudiostream == 0)
	cout << "Audiostream not detected" << endl;
    else if (!m_bAudioMute)
	createAudioRenderer();

    m_iFramesVideo=0;
    m_iFrameDrop=0;
    m_lTimeStart=0;
    m_bQuit = false;
    //	short fs;
    //	__asm__ __volatile__ ("movw %%fs, %%ax":"=a"(fs));
    //	cout<<"Before pthread_create: fs is "<<fs<<endl;
    //	fs_seg=fs;

    m_bHangup = true;
    if (!m_bVideoMute && m_pVideostream)
    {
	m_DecoderThread	= new PthreadTask(0, startDecoderThread, (void*)this);
	m_VideoThread = new PthreadTask(0, startVideoThread, (void*)this);
    }
    else if (!m_pAudioRenderer)
    {
	cerr << "Cannot play this" << endl;
	m_pClip = 0;
	return;
    }

    //m_pAudioRenderer = 0;
    if (!m_bAudioMute && m_pAudioRenderer)
	m_AudioThread = new PthreadTask(0, startAudioThread, (void*)this);

    // now wait until these new threads hang in the cond_wait
    lockThreads("Init");
}

HRESULT AviPlayer::SetColorSpace(fourcc_t csp, bool test_only)
{
    if (!m_pVideostream || !m_pVideostream->GetDecoder())
	return -1;

    //cout << "SET COLOR SPACE " << hex << csp << dec << "  " << test_only << endl;
    if (!test_only)
	return m_pVideostream->GetDecoder()->SetDestFmt(0, csp);

    IVideoDecoder::CAPS cap;
    switch (csp)
    {
    case fccYUY2:
	cap = IVideoDecoder::CAP_YUY2;
        break;
    case fccI420:
	cap = IVideoDecoder::CAP_I420;
	break;
    case fccYV12:
	cap = IVideoDecoder::CAP_YV12;
        break;
    case fccIYUV:
	cap = IVideoDecoder::CAP_IYUV;
        break;
    case fccUYVY:
	cap = IVideoDecoder::CAP_UYVY;
        break;
    case fccYVYU:
	cap = IVideoDecoder::CAP_YVYU;
	break;
    default:
	cap = IVideoDecoder::CAP_NONE;
    }

    return !(m_pVideostream->GetDecoder()->GetCapabilities() & cap);
}

HRESULT AviPlayer::InitSubtitles(const char* filename)
{
    if (m_pVideostream)
    {
	m_pSubtitles = subtitle_open(m_pSubtitles, filename);
	if (m_pSubtitles)
	{
	    printf("Subtitles from: %s\n", m_pSubtitles->filename);
	    subtitle_set_fps(m_pSubtitles, GetFps());
	}
    }

    return 0;
}

StreamInfo* AviPlayer::GetAudioStreamInfo() const
{
    return (m_pAudiostream) ? m_pAudiostream->GetStreamInfo() : 0;
}

StreamInfo* AviPlayer::GetVideoStreamInfo() const
{
    return (m_pVideostream) ? m_pVideostream->GetStreamInfo() : 0;
}

const subtitle_line_t* AviPlayer::GetCurrentSubtitles()
{
    if (!m_pVideostream || !m_pSubtitles)
	return 0;
    if (m_pSubline == 0)
    {
	m_pSubline = subtitle_line_new();
	if (m_pSubline == 0)
            return 0;
    }

    int r = subtitle_get(m_pSubline, m_pSubtitles,
			 m_pVideostream->GetTime() + m_fSubAsync);
    if (!r)
	return 0;
    return m_pSubline;
}

bool AviPlayer::HasSubtitles() const
{
    return (m_pSubtitles && subtitle_get_lines(m_pSubtitles) > 0);
}

void AviPlayer::Start()
{
    Debug cout << "AviPlayer::start" << endl;

    if (!IsValid()) return;
    if (IsRedirector()) return;
    if (IsPlaying())
    {
	cout<<"Can't start(), already started"<<endl;
	return;
    }
//    if(clip==0)
//    {
//	cout<<"Can't start(), no clip"<<endl;
//	return;
//    }
    m_bQuit = false;
    m_bPaused = false;
    m_bBuffering = false;

    if (m_pAudioRenderer)
    {
        // start audio first
        m_pAudioRenderer->Start();
	//m_pAudioRenderer->Reseek(0);
    }

    if (!gotsigfpe && m_pVideostream)
    {
	m_pVideostream->Seek((framepos_t)0);
	//m_pVideostream->GetDecoder()->SetDecodingMode(IVideoDecoder::REALTIME_QUALITY_AUTO);
    }

    unlockThreads();
}

void AviPlayer::Stop()
{
    Debug cout << "AviPlayer::stop" << endl;
    if (!IsPlaying())
	return;

    lockThreads("Stop");

    if (m_pAudioRenderer)
	m_pAudioRenderer->Stop();

    m_bPaused = false;

    //cout << "KILL handler " << (void*)m_pKillhandler<<endl;
    if (m_pKillhandler)
	m_pKillhandler(0, m_pKillhandlerArg);
}

void AviPlayer::Pause(bool state)
{
    Debug cout << "AviPlayer::pause " << m_bPaused << " -> " << state << endl;
    if (!IsPlaying())
        return;

    if (m_bPaused == state)
	return;

    if (state)
    {
	lockThreads("Pause");
	if (m_pAudioRenderer)
	    m_pAudioRenderer->Pause(state);
	m_bPaused = state;
    }
    else
    {
	if (m_pAudioRenderer)
	{
	    if (m_pAudioRenderer->Pause(state) != 0)
		return;
	}
	m_bPaused = state;

	unlockThreads();
    }
}

void AviPlayer::Play()
{
    Debug cout << "AviPlayer::play " << m_bPaused << endl;
    if (!IsPlaying())
        return;

    if (m_bPaused)
	Pause(!m_bPaused);
}

// in reality this is GetTime!!!
// FIXME - I would propose this name change
double AviPlayer::GetPos() const
{
    if (!gotsigfpe && m_pVideostream && !m_bVideoMute)
    {
	// this is bit more complicate so here goes light explanation:
	// Asf file could display same frame for 5 second - and
	// it doesn't look good when we show jumping numbers
	// so we pick the last remebered frame time and we are
	// calculating time from this frame - we should be preserveing
	// this speed - for paused mode we display time for currently
        // displayed frame
	double len = m_pVideostream->GetLengthTime();
	if (len > 0.)
	{
	    if (m_bPaused || !IsPlaying())
		return m_pVideostream->GetTime();

	    double ct = m_dLastFrameStart + to_float(longcount(), m_lLastTimeStart);
	    return (ct < len) ? ((ct > 0) ? ct : 0) : len;
	}
    }
    if (m_pAudioRenderer)
    {
	if (!m_pAudioRenderer->Eof())
	    return m_pAudioRenderer->GetTime();
        return m_pAudiostream->GetLengthTime();
    }
    return 0.;
}

double AviPlayer::GetLengthTime() const
{
    double len = GetVideoLengthTime();
    double alen = GetAudioLengthTime();
    //cout << "l " << len << "   " << alen << endl;
    if (len == 0)
	len = alen;
    return len;
}

framepos_t AviPlayer::GetFramePos() const
{
    return (!gotsigfpe && m_pVideostream) ? m_pVideostream->GetPos() : 0;
}

double AviPlayer::Reseek(double pos)
{
    Debug cout << "Seek pos: " << pos << endl;

    if (!IsValid() || IsRedirector() || !IsPlaying())
	return -1;

    lockThreads("Reseek");

    if (!gotsigfpe && m_pVideostream)
	pos = m_pVideostream->SeekTimeToKeyFrame(pos);

    Debug cout << "Keyframe pos: " << pos << endl;

    if (m_pAudioRenderer)
        m_pAudioRenderer->Reseek(pos);

    if (m_bPaused)
	drawFrame();

    unlockThreads();

    return pos;
}

HRESULT AviPlayer::ReseekExact(double pos)
{
    if (!IsValid() || IsRedirector() || !IsPlaying())
	return -1;

    // we might not need this in future
    lockThreads("ReseekExact");

    int r = 0;
    double pos2 = pos;

    if (!gotsigfpe && m_pVideostream)
    {
	double cft = m_pVideostream->GetTime();
	double nkft = m_pVideostream->GetTime(m_pVideostream->GetNextKeyFrame());
	if (cft < pos &&  pos < nkft)
	{
	    // do nothing
	    //cout << " NOT SEEKING " << cft << "  " << nkft << endl;
	    unlockThreads();
	    return 0;
	}

	pos2 = m_pVideostream->SeekTimeToKeyFrame(pos);
    }

    Debug cout << "Seek OK (" << pos << ", " << pos2 << ")" << endl;
    if (pos2 < 0.0)
    {
	pos2 = 0.0;
        pos = 0;
	cout << "ERROR: reseek_exact  pos2<0!" << endl;

	r = -1;
    }

    if (m_pAudioRenderer)
	m_pAudioRenderer->Reseek(pos2);

    if (pos2 > pos && pos2 > m_pVideostream->GetTime(0))
    {
	cout << "ERROR: reseek_exact: pos2>pos!" << pos2 << "  " << pos << endl;
	r = -1;
    }

    else if (m_pVideostream)
    {
	if (m_bVideoMute) //?? why
	{
	    //uint_t p=(uint_t)(pos/stream->GetFrameTime());
	    cout << "??? video_mute_seek ???" << pos << endl;
	    m_pVideostream->SeekTime(pos);
	}
	else
	{
	    //m_bInVideoSeek=true;
	}

	drawFrame();
    }

    unlockThreads();
    return r;
}

HRESULT AviPlayer::PageUp()
{
    if (!IsValid() || IsRedirector() || !IsPlaying())
	return -1;

    int r = 0;
    lockThreads("PageUp");
    if (!gotsigfpe && m_pVideostream)
    {
	framepos_t pos = m_pVideostream->GetPos();
        framepos_t npos = m_pVideostream->GetNextKeyFrame();
	if (pos < npos && npos != m_pVideostream->ERR)
	{
	    m_pVideostream->SeekToKeyFrame(npos);
	    if (m_pAudioRenderer)
		m_pAudioRenderer->Reseek(m_pVideostream->GetTime());

	    drawFrame();
	}
	else
            r = -1;
    }
    else if (m_pAudioRenderer)
    {
	m_pAudioRenderer->Reseek(m_pAudioRenderer->GetTime() + 1.0);
    }

    unlockThreads();

    return r;
}

HRESULT AviPlayer::PageDown()
{
    if (!IsValid() || IsRedirector() || !IsPlaying())
	return -1;

    static framepos_t prevnpos = ~0;
    static int64_t prevtime = 0;

    int r = 0;
    lockThreads("PageDown");

    if (!gotsigfpe && m_pVideostream)
    {
        framepos_t cpos = m_pVideostream->GetPos();
	framepos_t npos = m_pVideostream->SeekToPrevKeyFrame();
	if (to_float(longcount(), prevtime) < 0.3)
	{
            framepos_t p = npos;
	    while (npos > 0 && npos >= prevnpos)
	    {
		//cout << "  using trick " << endl;
		npos = m_pVideostream->SeekToPrevKeyFrame();
		if (p == npos)
                    break;
	    }
	}

	//cout << " POSN " << npos << endl;

	if ((cpos - npos < 5) && !IsPaused())
	{
	    m_pVideostream->Seek(npos);
            npos = m_pVideostream->SeekToPrevKeyFrame();
	    //cout << " POSx " << npos << endl;
	}
        prevtime = longcount();
	prevnpos = npos;
	//cout << "Diff " << cpos << "   " << npos << endl;
	if (npos != m_pVideostream->ERR)
	{
	    //m_pVideostream->SeekToKeyFrame(npos);
	    double pos = m_pVideostream->GetTime();
	    if (m_pAudioRenderer)
		m_pAudioRenderer->Reseek(pos > 0.0 ? pos : 0.0);
	}
	drawFrame();
    }
    else if (m_pAudioRenderer)
	m_pAudioRenderer->Reseek(m_pAudioRenderer->GetTime() - 1.);

    unlockThreads();

    return r;
}

HRESULT AviPlayer::NextFrame()
{
    if (!IsValid() || IsRedirector() || !IsPlaying())
	return -1;

    if (m_pVideostream)
    {
	lockThreads("NextFrame");
	drawFrame();
	if (m_pAudioRenderer)
	    m_pAudioRenderer->Reseek(m_pVideostream->GetTime());
	unlockThreads();
    }

    return 0;
}

// used when we are in pause mode or the player is not playing
int AviPlayer::drawFrame()
{
    if (!gotsigfpe && m_pVideostream && !m_bVideoMute)
    {
	CImage* im = m_pVideostream->GetFrame(true); // ReadFrame
        m_fLastDiff = 0.0;
	if (im)
	{
	    const subtitle_line_t* sl = GetCurrentSubtitles();
	    for (unsigned i = 0; i < m_VideoRenderers.size(); i++)
	    {
		m_VideoRenderers[i]->Draw(im);
                if (HasSubtitles())
		    m_VideoRenderers[i]->DrawSubtitles(sl);

		m_VideoRenderers[i]->Sync();
	    }
	    im->Release();
	    m_iFramesVideo++;
	}
	return 0;
    }
    return -1;
}

void AviPlayer::createAudioRenderer()
{
    delete m_pAudioRenderer;
    m_pAudioRenderer = 0;

    WAVEFORMATEX MyOwf;
    memset(&MyOwf, 0, sizeof(MyOwf));
    uint_t forcedFreq = 0;

    bool resamp;
    Get(AUDIO_RESAMPLING, &resamp, 0);
    if (resamp)
    {
	Get(AUDIO_RESAMPLING_RATE, &MyOwf.nSamplesPerSec,
	    AUDIO_PLAYING_RATE, &forcedFreq,
	    0);
    }

    char* arend;
    Get(AUDIO_RENDERER, &arend, 0);
    if (!arend)
	return;

    char* arends = arend;
    while (arends && !m_pAudioRenderer)
    {
	WAVEFORMATEX Owf = MyOwf;
	try
	{
	    cout << "Trying audio renderers: " << arends << endl;
	    if (m_pAudiofunc)
	    {
		// we could run this code sooner - but why...
		m_pAudioRenderer = new CopyAudioRenderer(m_pAudiostream, Owf, m_pAudiofunc, m_pAudiofuncArg, m_acodec.size() ? m_acodec.c_str() : 0);
	    }
	    else if (!strncasecmp(arends, "SDL", 3))
	    {
#ifdef USE_SDL
		m_pAudioRenderer = new SdlAudioRenderer(m_pAudiostream, Owf, forcedFreq, m_acodec.size() ? m_acodec.c_str() : 0);
#else
		cout << "Warning: SDL audio renderer unavailable!" << endl;
#endif
	    }
	    else if (!strncasecmp(arends, "OSS", 3))
	    {
#ifdef OSS_SUPPORT
		m_pAudioRenderer = new OssAudioRenderer(m_pAudiostream, Owf, m_acodec.size() ? m_acodec.c_str() : 0);
#else
		cout << "Warning: OSS audio renderer unavailable!" << endl;
#endif
	    }
	    else if (!strncasecmp(arends, "SUN", 3))
	    {
#ifdef SUNAUDIO_SUPPORT
		cout << "Warning: Sun audio renderer out of date - update me!" << endl;
		//		m_pAudioRenderer = new SunAudioRenderer(m_pAudiostream, Owf, m_acodec.size() ? m_acodec.c_str() : 0);
#else
		cout << "Warning: Sun audio renderer unavailable!" << endl;
#endif
	    }
	    else if (!strncasecmp(arends, "noaudio", 5))
                break;
	}
	catch (FatalError& error)
	{
	    error.Print();
	}
	arends = strchr(arends, ',');
	if (arends)
            arends++;
    }
    if (m_pAudioRenderer)
    {
	m_pAudioRenderer->SetAsync(m_fAsync);
	m_pAudioRenderer->SetVolume(m_fVolume);
    }
    else
	m_pAudiostream = 0;
    free(arend);
}

AviPlayer::State AviPlayer::GetState(double* percent)
{
    if (!m_pClip)
	return Invalid;

    if (!m_pClip->IsOpened())
	return Opening;

    if (m_bBuffering)
    {
	if (percent)
	{
	    if (!gotsigfpe && m_pVideostream)
		*percent = m_pVideostream->CacheSize();
	    else if (m_pAudioRenderer)
		*percent = m_pAudioRenderer->GetCacheSize();
	}
	return Buffering;
    }
    if (IsPaused())
	return Paused;
    if (IsPlaying())
	return Playing;

    return Stopped;
}

bool AviPlayer::IsPaused() const
{
    return (m_pClip && m_bPaused);
}

bool AviPlayer::IsPlaying() const
{
    // when we are in paused mode  initialized is 0
    return m_bInitialized || m_bPaused;
}

bool AviPlayer::IsStopped() const
{
    return (m_pClip && !IsPlaying());
}

const avm::vector<VideoRenderer*>& AviPlayer::GetVideoRenderers() const
{
    return m_VideoRenderers;
}

HRESULT AviPlayer::SetVideoRenderers(avm::vector<VideoRenderer*>)
{
    cerr << "AviPlayer::SetVideoRenderers()  unimplemented yet sorry..." << endl;
    return -1;
}

void AviPlayer::SetAsync(float async)
{
    Set(ASYNC_TIME_MS, int(async * 1000), 0);
}

void AviPlayer::Restart()
{
    if (!IsPlaying() || !m_pVideostream || gotsigfpe)
	return;

    IVideoDecoder* vs = m_pVideostream->GetDecoder();
    if (!vs)
	return;

    if (m_bPaused)
    {
	vs->Restart();
	return;
    }

    lockThreads("Restart");
    vs->Restart();
    unlockThreads();
    ReseekExact(GetPos());
}

IAviPlayer* CreateAviPlayer(const char* filename, int bitdepth, const char* subname, const char* vcodec=0, const char* acodec=0) throw(FatalError)
{
    return new AviPlayer(filename, bitdepth, subname, vcodec, acodec);
}

extern "C" double GetAvifileVersion()
{
    return AVIFILE_VERSION;
}

void* AviPlayer::startVideoThread(void* arg)
{
    AviPlayer* a=(AviPlayer*)arg;
    return a->videoThread();
}

void* AviPlayer::startAudioThread(void* arg)
{
    AviPlayer* a=(AviPlayer*)arg;
    return a->audioThread();
}

void* AviPlayer::startDecoderThread(void* arg)
{
    AviPlayer* a=(AviPlayer*)arg;
    return a->decoderThread();
}
/*
vim: tabstop=8
*/
