/*********************************************************

	Implementation of base audio renderer class

*********************************************************/

#include "IAudioRenderer.h"
#include "AudioQueue.h"
#include "RegAccess.h"

#include "avifile.h"
#include "cpuinfo.h"
#include "except.h"
#include "utils.h"

#include <stdio.h>

#undef Debug
#define Debug if(0)

#define __MODULE__ "IAudioRenderer"

template<class T> inline T mymax(const T x, const T y)
{
    return (x > y) ? x : y;
}

IAudioRenderer::IAudioRenderer(IAviReadStream* as, WAVEFORMATEX& Owf, const char* privname)
{
    m_bInitialized = false;
    m_bPaused = true;
    m_bQuit = false;
    m_dAudioRealpos = 0.0;
    m_lAudioTime = 0;
    m_lTimeStart = 0;
    m_pcLocalFrame = 0;
    m_pQueue = 0;
    m_fVolume = 1.0;
    m_Owf = Owf;
    m_dPauseTime = 0.0;
    m_pAudiostream = as;

    if (!m_pAudiostream)
	throw FATAL("NULL m_pAudiostream");

    // stop any previous streaming of this stream
    m_pAudiostream->StopStreaming();
    int audio_status = m_pAudiostream->StartStreaming(privname);
    if (audio_status != 0)
	throw FATAL("Failed to start streaming");

    // needed for bitrate in nAvgBytesPerSec
    WAVEFORMATEX Inwf;
    m_pAudiostream->GetAudioFormat(&Inwf, sizeof(Inwf));
    m_pAudiostream->GetOutputFormat(&m_Iwf, sizeof(m_Iwf));

    // format can't be changed
    m_Owf.wFormatTag = m_Iwf.wFormatTag;
    m_Owf.nBlockAlign = m_Iwf.nBlockAlign;

    if (m_Owf.nChannels == 0)
	m_Owf.nChannels = m_Iwf.nChannels;

    if (m_Owf.nSamplesPerSec == 0)
	m_Owf.nSamplesPerSec = m_Iwf.nSamplesPerSec;

    if (m_Owf.wBitsPerSample == 0)
	m_Owf.wBitsPerSample = m_Iwf.wBitsPerSample;

    static const char* chs[] =
    {
	"---", "mono", "stereo", "3 channels", "4 channels", "5 channels"
    };

    sprintf(m_pcAudioDesc, " %s %dkbit/s %dHz %s",
	    avm_wave_format_name(Inwf.wFormatTag),
	    (int)(Inwf.nAvgBytesPerSec * 8 + 500) / 1000, // rounding
	    (int)m_Iwf.nSamplesPerSec,
	    (m_Iwf.nChannels >= 0 && m_Iwf.nChannels < 6)
	    ? chs[m_Iwf.nChannels] : "ERROR!!");
    // no need to catch - we allocate m_pQueue as last object
    m_pQueue = new AudioQueue(m_Iwf, m_Owf);
}

IAudioRenderer::~IAudioRenderer()
{
    m_Mutex.Lock();
    if (m_pQueue)
    {
	m_pQueue->Lock();
	delete m_pQueue;
    }
    delete[] m_pcLocalFrame;
}

int IAudioRenderer::Extract(double video_time)
{
    static const uint_t sound_size_max = 176400U;
    // 1s of stereo 16b 44kHz sound

    if (m_bQuit
	|| m_pQueue->GetSize() > sound_size_max
	|| m_pAudiostream->Eof())
	return -1; // reader is supposed to wait!!

    const uint_t one_frame_sound =
	mymax(20000U, m_pAudiostream->GetFrameSize());

    uint_t ocnt;
    {
	Locker locker(m_Mutex);

	if (Eof())
	{
	    printf("Audio stream finished\n");
	    //pause(1);
	    //m_bInitialized = false;
	    //m_bPaused = false;
	    m_pQueue->Clear();
	    return -1;
	}

	if (!m_pcLocalFrame)
	    m_pcLocalFrame = new char[one_frame_sound];

	uint_t samples;

	//for (int i = 0; i < 64; i++)
	//    printf("%d   0x%x\n", i, (unsigned char) m_pcLocalFrame[i]);
	//printf("TIME %f  %f   ft: %f\n", m_pAudiostream->GetTime(),
	//       m_pQueue->GetBufferTime(), m_pAudiostream->GetTime() - m_pQueue->GetBufferTime());

	m_pAudiostream->ReadFrames(m_pcLocalFrame, one_frame_sound,
				   one_frame_sound, samples, ocnt);
	if ((int)ocnt < 0)
	    return -1;
	if (samples > one_frame_sound)
	    printf("OOPS: samples (%d) > one_frame_sound (%d) at %s\n", samples, one_frame_sound, __FILE__);
	if (ocnt > one_frame_sound)
	{
	    printf("OOPS: ocnt (%d)  > one_frame_sound (%d) at %s\n",
		   ocnt, one_frame_sound, __FILE__);
	    ocnt = one_frame_sound;

	}
	Debug printf("doAudioExtract: Read %d samples (%d bytes)\n", samples, ocnt);

	// reseek skiping is handled by AudioQueue
	// we can't just simply remove the packet as this makes
	// the sound asynchronous
	//if (m_iReseekSkip > 0)
	//{
	//    m_iReseekSkip--;
	//    return;
	//}
    }
    m_pQueue->Push(m_pcLocalFrame, ocnt);
    return 0;
}

double IAudioRenderer::GetBufferTime() const
{
    if (!m_bInitialized)
        return 0.0;

    //cout << "GBT " << m_pQueue->GetSize() << "   " << m_pQueue->GetBytesPerSec() << endl;
    double btime = m_pQueue->GetBufferTime() + getRendererBufferTime();

    Debug printf("IAudioRenderer: BufferTime: %f   Queue size: %d   Queue size: %Ld\n",
		 btime, m_pQueue->GetSize(), m_lAudioTime);

    return btime;
}

void IAudioRenderer::Clear()
{
    Locker locker(m_Mutex);
    m_pQueue->Clear();
}

const char* IAudioRenderer::GetAudioFormat() const
{
    return m_pcAudioDesc;
}

double IAudioRenderer::GetCacheSize() const
{
    return m_pAudiostream->CacheSize();
}

double IAudioRenderer::GetLengthTime() const
{
    return m_pAudiostream->GetLengthTime();
}

double IAudioRenderer::GetTime()
{
    if (m_dPauseTime)
	return m_dPauseTime;

    double actual_time = m_dAudioRealpos;

    if (m_lTimeStart == 0)
    {
	if (m_lAudioTime == 0)
	{
	    m_lAudioTime++; // prevent this setting again
	    // check again with lock
	    Debug printf("Reset m_dAudioRealpos = %f (stream: %f) buffered: %f  async: %f\n",
			 m_dAudioRealpos, m_pAudiostream->GetTime(),
			 GetBufferTime(), m_fAsync);

	}
    }
    else
	actual_time += to_float(longcount(), m_lTimeStart);
	//actual_time = m_pAudiostream->GetTime() - GetBufferTime()
	//    + to_float(longcount(), m_lAudioTime);

    //cout << "stream: " << m_pAudiostream->GetTime()
    //    << " buffered " << GetBufferTime()
    //    << " atime " << actual_time  << endl;

    actual_time -= m_fAsync;
    return (actual_time > 0) ? actual_time : 0.0;
}

bool IAudioRenderer::Eof() const
{
    //if (m_pAudiostream->Eof()) printf("Audio EOF --- %f\n", GetBufferTime());
    return m_pAudiostream->Eof() && ((GetBufferTime() - getRendererBufferTime()) < 0.01);
}

HRESULT IAudioRenderer::Pause(bool state)
{
    Locker locker(m_Mutex);

    if (!m_bInitialized)
	return -1;

    if (m_bPaused != state)
    {
	m_bPaused = state;
        if (state)
	    Wake(); // in case audio renderer waits for new data
	pause((state) ? 1 : 0);
	if (state)
            m_dPauseTime = GetTime();
    }
    return 0;
}

HRESULT IAudioRenderer::Reseek(double pos)
{
    Locker locker(m_Mutex);

    double et = m_pAudiostream->GetLengthTime();
    pos = (et > (pos + m_fAsync)) ? (pos + m_fAsync) : et;
    if (pos < 0)
        pos = 0.0;
    HRESULT hr = m_pAudiostream->SeekTime(pos);
    m_pAudiostream->SkipTo(pos);
    if (!m_bInitialized || hr != 0)
	return -1;

    // do not skip sound when seeking to the begining
    // this sound should be clean
    m_bInitialized = false;
    Wake(); // sets m_bInitialized = false;
    m_pQueue->Clear();
    m_lAudioTime = m_lTimeStart = 0;
    m_dAudioRealpos = m_pAudiostream->GetTime();
    m_dPauseTime = 0.0;
    m_bInitialized = true;
    return 0;
}

HRESULT IAudioRenderer::Skip(double skiptime)
{
    Locker locker(m_Mutex);
    return m_pAudiostream->SkipTo(skiptime);
}

void IAudioRenderer::SetAsync(float async)
{
    m_fAsync = async;
}

HRESULT IAudioRenderer::SetVolume(float volume)
{
    if (volume < 0.0 || volume > 1.0)
	return -1;

    m_fVolume = volume;
    return 0;
}

HRESULT IAudioRenderer::SetPlayingRate(int rate)
{
    return -1;
}

HRESULT IAudioRenderer::SetResamplingRate(int rate)
{
    return -1;
}

void IAudioRenderer::Start()
{
    {
	Locker locker(m_Mutex);
	if (m_bInitialized)
	{
	    Debug printf("IAudioRenderer: can't start(), already started\n");
	    return;
	}
	m_bInitialized = true;
    }
    Reseek(0);
    Pause(false);
}

void IAudioRenderer::Stop()
{
    {
	Locker locker(m_Mutex);
	if (!m_bInitialized)
	    return;
	m_pQueue->Clear();
    }
    Pause(true);
    m_bInitialized = false;
}

void IAudioRenderer::Wake()
{
    m_pQueue->Broadcast();
}

/// to be overloaded
void IAudioRenderer::pause(int)
{
}

void IAudioRenderer::reset()
{
}
