#ifndef AVIFILE_RENDERER_H
#define AVIFILE_RENDERER_H

/********************************************************

	Video renderer
	Copyright 2000 Eugene Kuznetsov (divx@euro.ru)

*********************************************************/

/*
 *  One day we may support more renderers
 *  We do not want to have any XFree specific stuff here
 */

#include "default.h"
#include "avm_stl.h"
#include "image.h"
#include "subtitle.h"

class PlayerWidget;
/**
 * Function that returns actual bit depth of specified X display.
 * Should be used instead of any other functions with similar
 * purpose ( DefaultDepth(), x11Depth() in Qt ) when determining
 * bit depth of data to pass to renderer.
 */

struct VideoMode
{
    int width;
    int height;
    float freq;
    avm::string name;
};

class VideoRenderer
{
public:
    enum Property {
	SYNC,
	REFRESH,
	FULLSCREEN,		// true - maximize with FULLSCREEN toggle
	DRAW_CIMAGE,            // CImage*
	DRAW_SUBTITLES,

        WINDOW_SIZE = 100,      // int* width, int* height
	WINDOW_POSITION,        // get*/set int* x, int* y
        WINDOW_CAPTION,         // char*
        SUBTITLE_FONT,          // char*
        SUBTITLE_POSITION,      // get*/set int* y
        SUBTITLE_VISIBLE,
	VIDEO_MODES,		// avm::vector<VideoMode> - list of video modes
	VIDEO_OVERLAYS,		// avm::vector<fcctype_t> - list of overlay

        LAST
    };
    /**
     * Allow YUV renderer to use software YUV->RGB conversion
     * ( slow, only for testing purposes )
     * SDL has very fast implementation - used now
     */
    static bool allow_sw_yuv;
    /**
     * Allow renderers to perform resizing window from caller thread
     */
    static bool direct_resizes;

    virtual ~VideoRenderer() {}

    // possible new interface???
    virtual int Set(...) 					= 0;
    virtual int Get(...) const					= 0;
#if 1

    /**
     * Issue the request to draw the image in the window memory
     * might be empty when direct rendering is being used
     */
    virtual int Draw(const CImage* data)                        = 0;
    /**
     * Draw a line of subtitles below the picture.
     */
    virtual int DrawSubtitles(const subtitle_line_t* sl) { return -1; }
    /**
     * Returns CImage for direct rendering into memory based
     * on current format provided by renderer
     */
    virtual CImage* GetData(int idx)			{ return 0; }
    /**
     * Return actual size of the rendered image
     */
    virtual int GetSize(int& width, int& height) const	{ return -1; }
    /**
     * Return actual windows x,y coordinates
     */
    virtual int GetPosition(int& x, int& y) const	{ return -1; }
    /**
     * Retrieve available video modes
     */
    virtual const avm::vector<VideoMode>& GetVideoModes() const	= 0;
    /**
     * Called to blit image on the screen
     */
    virtual int Sync()						= 0;
    /**
     * Redraw image (resize, move with pause)
     */
    virtual int Refresh()                                       = 0;
    /**
     * Resize renderer window and stretch the image in it.
     */
    virtual int Resize(int& new_w, int& new_h)		{ return -1; }
    /**
     * Set window titlebar
     */
    virtual int SetCaption(const char* title, const char* icon) { return -1; }
    /**
     * Set font for subtitle
     */
    virtual int SetFont(const char* font)		{ return -1; }
    /**
     * Set window position
     */
    virtual int SetPosition(int x, int y)		{ return -1; }
    /**
     * Toggle on/off fullscreen mode. maximize=true means that
     * renderer should stretch the picture to dimensions of 
     * screen when going fullscreen.
     */
    virtual int ToggleFullscreen(bool maximize = false)	{ return -1; }
    /**
     * Function to be called when we need to prevent renderer from
     * doing any internal changes - like resize - implemented with mutex lock
     */
#endif
    virtual int Lock() = 0;
    virtual int TryLock() = 0;
    virtual int Unlock() = 0;
};


#ifndef WITHOUT_X
/**
 * Creates SDL video renderer for RGB data in current depth of
 * display from 2nd argument. Last argument specifies whether
 * this renderer should reserve place for subtitles or not.
 */
VideoRenderer* CreateFullscreenRenderer(PlayerWidget*, void* dpy,
					int width, int height, bool sub = false);
/**
 * Creates SDL video renderer for YUV data in format yuv_fmt.
 * If it fails ( e.g. because yuv_fmt is not supported by X server
 * and VideoRenderer::allow_sw_yuv==false ), returns pointer to
 * 'ordinary' fullscreen renderer and puts 0 into yuv_fmt.
 */    
VideoRenderer* CreateYUVRenderer(PlayerWidget*, void* dpy,
				 int width, int height,
				 fourcc_t yuv_fmt, bool sub = false);
/*
VideoRenderer* CreateXvYUVRenderer(PlayerWidget*, void* dpy,
				   int width, int height,
				   fourcc_t yuv_fmt, bool sub = false);
 */
#endif // WITHOUT_X

int GetPhysicalDepth(void* dpy);

#ifdef HAVE_LIBXXF86DGA
//VideoRenderer* CreateDGARenderer();
#endif

#endif // AVIFILE_RENDERER_H
