
/* MODULE							HTScript.c
**		CALL A SCRIPT TO PRODUCE A DOCUMENT ON FLY
**
** AUTHORS:
**	AL	Ari Luotonen	luotonen@dxcern.cern.ch
**	MD	Mark Donszelmann    duns@vxdeop.cern.ch
**
** HISTORY:
**	31 Oct 93  AL	Written from scratch.
**	 6 Nov 93  MD	Made VMS compatitibility.
**	13 Nov 93  MD	Escaped parameters and keywords correctly for VMS.
**
** BUGS:
**
**
*/

#include <string.h>
#include <stdio.h>

#include "HTUtils.h"
#include "HTRules.h"	/* HTBinDir, HTBINDIR */
#include "HTFile.h"
#include "HTWriter.h"
#include "HTStream.h"
#include "tcp.h"

#define INPUT_BUFFER_SIZE  4096

extern char * HTClientProtocol;
extern char * HTServerProtocol;
extern char * HTRedirectedDoc;

/*
**		SCRIPT OUTPUT
**		=============
*/
struct _HTStream {
    HTStreamClass *isa;
    /* ...and opaque stuff... */
};


/* PRIVATE						HTLoadScriptResult()
**		PUSH DATA FROM SCRIPT OUTPUT FILE TO STREAM
**		USING CORRECT PROTOCOL VERSION (HTTP0/HTTP1).
**		DO REDIRECTION IF REQUESTED BY THE SCRIPT.
** ON ENTRY:
**	fp	open script output file, format:
**
**			Content-Type: .../...
**			<blank line>
**			Document
**		or:
**			Location: ...
**			<blank line>
**
**		which causes the server to send a redirection
**		reply.
**
**	sink	stream to push the data down to.
**
** ON EXIT:
**	returns	HT_LOADED on success.
**		HT_REDIRECTION_ON_FLY if the daemon should redo the
**		retrieve (in case the redirection is a local file).
**		In this case the global HTRedirectedDoc is set to
**		the document name.
**		File is read until the EOF, but left open.
*/
PRIVATE int HTLoadScriptResult ARGS2(FILE *,	fp,
				     HTStream*,	sink)
{
    HTStreamClass targetClass;
    char input_buffer[INPUT_BUFFER_SIZE];
    BOOL reading_headers = YES;
    char *location = NULL;
    char reply[2048];		/* @@ */

    if (!fp || !sink)
	return HT_INTERNAL;

    targetClass = *(sink->isa);	/* Copy pointers to procedures */
    
    for(;;) {
	int status = fread(input_buffer, 1, INPUT_BUFFER_SIZE, fp);
	if (status == 0) { /* EOF or error */
	    if (ferror(fp) == 0) break;
	    if (TRACE) fprintf(stderr,
			       "HTFormat: Read error, read returns %d\n",
			       ferror(fp));
	    break;
	}

	if (reading_headers) {
	    char *cur = input_buffer;
	    char *end = input_buffer + status;
	    
	    while (cur < end) {
		if (0 == strncasecomp(cur, "Location:", 9)) {
		    char *last;

		    cur += 9;	/* Skip "Location:" */
		    while (cur < end &&
			   (*cur == ' ' || *cur =='\t'))
			cur++;
		    if (cur >= end) break;
		    last = strchr(cur, '\n');
		    if (last) {
			if (*(last-1) == '\r')
			    last--;
			location = (char*)malloc(last - cur + 1);
			strncpy(location, cur, last-cur);
			location[last-cur] = (char)0;
		    } /* if valid Location: contents found */
		} /* if Location: field */

		else if (*cur == '\n' ||
			 (*(cur) == '\r' && *(cur+1) == '\n')) {
		    if (*(cur) == '\r')
			cur += 2;
		    else
			cur++;
		    reading_headers = NO;
		    break;
		} /* if end-of-header-section */

		while (cur < end  &&  *cur != '\n') {
		    cur++;
		}
		if (cur < end)
		    cur++;	/* Skip newline */

	    } /* for the entire header field section */

	    if (location) {	/* Redirection */
		if (location[0] == '/') {	/* Local file -- do retrieve */
		    StrAllocCopy(HTRedirectedDoc, location);
		    return HT_REDIRECTION_ON_FLY;
		}
		if (HTClientProtocol) {	/* HTTP1 reply */
		    sprintf(reply,
			    "%s %s\r\n%s\r\n%s\r\n%s %s\r\n\r\n",
			    HTServerProtocol, "302 Found",
			    "MIME-Version: 1.0",
			    "Content-Type: text/html",
			    "Location:", location);
		    (*targetClass.put_string)(sink, reply);
		}
		sprintf(reply, "%s\n%s%s%s\n",
			"<BODY><H1>Redirection</H1>",
			"This document can be found <A HREF=\"",
			location,
			"\">elsewhere.</A></BODY>");
		(*targetClass.put_string)(sink, reply);
		goto done;
	    }
	    else { /* No redirection, just send the document */
		if (HTClientProtocol) {
		    sprintf(reply,
			    "%s 200 Document follows\r\nMIME-Version: 1.0\r\n",
			    HTServerProtocol);
		    (*targetClass.put_string)(sink, reply);
		    (*targetClass.put_block)(sink, input_buffer, status);
		}
		else { /* Old protocol -- strip header lines */
		    (*targetClass.put_block)(sink, cur, end-cur);
		}
	    }
	} /* if reading headers */

	else { /* else reading body */
	    (*targetClass.put_block)(sink, input_buffer, status);
	} /* reading body */

    } /* next bufferload */

  done:
    (*targetClass.free)(sink);
    return HT_LOADED;
}



/* PUBLIC							HTCallScript()
**		CALL A SCRIPT AND SEND RESULTS BACK TO CLIENT
** ON ENTRY:
**	url		translated URL starting "/htbin/".
**			The next component is taken to be the script
**			name, and the rest of it (even if there is nothing)
**			is the first parameter for the script.
**			E.g.
**				/htbin/foo/bar/x/y
**
**			is called as:
**
**			    <HTBinDir>/foo /bar/x/y keywords...
**
**			and:
**				/htbin/foo
**
**			is called as:
**
**			    <HTBinDir>/foo '' keywords...
**
**	keywordvec	array of strings, each of which is one
**			keyword (already unescaped).
**			Escapes special characters to protect them
**			from shell special character translations.
**	soc		socket to write the reply.
**
** ON EXIT:
**	returns		HT_LOADED on success.
*/
PUBLIC int HTCallScript ARGS3(CONST char *,	url,
			      char **,		keywordvec,
			      int,		soc)
{
    static char *script = NULL;		/* Auto-freed every time called */
    static char *name = NULL;		/* Auto-freed			*/
    char *param = NULL;
    int len = 0;
#ifdef VMS
    static char *tmpscriptfile = NULL;	/* Auto-freed			*/
#endif /* VMS */
    static char *tmpfile = NULL;	/* Auto-freed			*/
    FILE *fp = NULL;
    int status;
    HTStream * sink = HTASCIIWriter(soc);

    /*
    ** Get tmp names
    */
#ifdef VMS
    StrAllocCopy(tmpscriptfile, tmpnam(NULL));	/* Also frees the one from previous call */
#endif /* VMS */
    StrAllocCopy(tmpfile, tmpnam(NULL));	/* Also frees the one from previous call */

    /*
    ** Check that it really is a /htbin request
    */
    if (!url  ||  strlen(url) < 8  ||  0 != strncmp(url, "/htbin/", 7)) {
	if (TRACE) fprintf(stderr,
			   "HTCallScript: called with an invalid URL: '%s'\n",
			   (url ? url : "(null)"));
	return HTLoadError(sink, 400, "Bad script execution request");
    }

    /*
    ** Find the script name
    */
    url += 7;			/* Skip "/htbin/" */
    StrAllocCopy(name, url);	/* Also frees the one from previous call */
    if (NULL != (param = strchr(name, '/'))) {
	*(param++) = (char)0;
    }

    /*
    ** Check that /htbin scripts are actually enabled on this server
    */
    if (!HTBinDir || !*HTBinDir) {
	if (TRACE) fprintf(stderr, "%s (requested URL '%s')\n",
			   "HTCallScript: /htbin scripts not enabled", url);
	return HTLoadError(sink, 500,
	     "/htbin scripts are not enabled on this server (no htbin rule)");
    }

    /*
    ** Allocate enough space for system() command line
    */
    len = strlen(HTBinDir) + strlen(name) + 200;
    if (param)
	len += strlen(param);
    if (keywordvec) {
	char **cur = keywordvec;
	while (*cur) {
	    len += strlen(*cur) + 3;	/* Quotes and space */
	    cur++;
	}
    }
    if (script) free(script);	/* From previous call */
    if (!(script = (char*)malloc(len)))
	outofmem(__FILE__, "HTCallScript");

#ifdef VMS
    /*
    ** Make command line for system()
    */
    strcpy(script, HTBinDir);
    strcat(script, "/");
    strcat(script, name);
    sprintf(script,"@%s",HTVMS_name("",script));
    if (param) {
	strcat(script, " \"/");
	strcat(script, param);
	strcat(script, "\"");
    }
    if (keywordvec) {
	CONST char **cur = keywordvec;
	char *dest = &script[strlen(script)];
	while (*cur) {
	    CONST char *src = *cur;
	    *(dest++) = ' ';	/* Space, i.e. field separator		     */
	    *(dest++) = '\"';	/* Quote which escapes everything but itself */
	    while (*src) {
		if (*src == '\"') {
		    *(dest++) = '\"';	/* create double quotes 	*/
		    *(dest++) = '\"';	
		}
		else *(dest++) = *src;	/* Just copy the character	*/
		src++;
	    }
	    *(dest++) = '\"';		/* Close quotes			*/
	    cur++;			/* Next keyword			*/
	} /* while keywords remain */
	*dest = (char)0;
    } /* if there are keywords given */

    /* 
     * for VMS we write redirection into a tmpscriptfile, including a call to
     * the actual script. We then call tmpscriptfile with system instead of 
     * script.
     */
    StrAllocCat(tmpfile,".");
    StrAllocCat(tmpscriptfile,".");
    {
       FILE *tmp = fopen(tmpscriptfile,"w");
       if (tmp == NULL) {
	   char msg[256];
	   if (TRACE) fprintf(stderr,
			   "HTCallScript: could not create temp script file");
	   sprintf(msg, "Script call '%s/%s' failed %s", HTBinDir, name,
		"-- either a bad script name or an internal error");
	   return HTLoadError(sink, 400, msg);
       }       
       fprintf(tmp,"$ define sys$output %s\n",tmpfile);
       fprintf(tmp,"$ define sys$error %s\n",tmpfile);
       fprintf(tmp,"$ %s\n",script);
       fprintf(tmp,"$ deassign sys$output\n");  
       fprintf(tmp,"$ deassign sys$error\n");  
       fclose(tmp);
    }

    /*
    ** Execute script
    */
    if (TRACE) fprintf(stderr, "HTCallScript: %s\n", script);
    sprintf(script,"$ @%s",tmpscriptfile);
    if ((status = system(script)) == 0) {
	char msg[256];
	if (TRACE) fprintf(stderr,
			   "HTCallScript: system() returned %d\n", status);
	sprintf(msg, "Script call '%s/%s' failed %s", HTBinDir, name,
		"-- either a bad script name or an internal error");
	return HTLoadError(sink, 400, msg);
    }
#else /* not VMS */
    /*
    ** Make command line for system()
    */
    strcpy(script, HTBinDir);
    strcat(script, "/");
    strcat(script, name);
    if (param) {
	strcat(script, " '/");
	strcat(script, param);
	strcat(script, "'");
    }
    else strcat(script, " ''");
    if (keywordvec) {
	char **cur = keywordvec;
	char *dest = &script[strlen(script)];
	while (*cur) {
	    CONST char *src = *cur;
	    *(dest++) = ' ';	/* Space, i.e. field separator		     */
	    *(dest++) = '\'';	/* Quote which escapes everything but itself */
	    while (*src) {
		if (*src == '\'') {
		    *(dest++) = '\'';	/* First close current quotes	*/
		    *(dest++) = '\\';	/* Escape quote with backslash	*/
		    *(dest++) = '\'';	/* Quote itself			*/
		    *(dest++) = '\'';	/* Open quotes again		*/
		}
		else *(dest++) = *src;	/* Just copy the character	*/
		src++;
	    }
	    *(dest++) = '\'';		/* Close quotes			*/
	    cur++;			/* Next keyword			*/
	} /* while keywords remain */
	*dest = (char)0;
    } /* if there are keywords given */
    strcat(script, " > ");
    strcat(script, tmpfile);
    strcat(script, " 2>&1");

    /*
    ** Execute script
    */
    if (TRACE) fprintf(stderr, "HTCallScript: %s\n", script);
    if ((status = system(script))) {
	char msg[256];
	if (TRACE) fprintf(stderr,
			   "HTCallScript: system() returned %d\n", status);
	sprintf(msg, "Script call '%s/%s' failed %s", HTBinDir, name,
		"-- either a bad script name or an internal error");
	return HTLoadError(sink, 400, msg);
    }

#endif /* not VMS */

    /*
    ** Send results back to client
    */
    if (!(fp = fopen(tmpfile, "r"))) {
	char msg[256];
	if (TRACE) fprintf(stderr, "%s '%s' %s\n",
			   "HTCallScript: Unable to open result file",
			   tmpfile, "for reading");
	sprintf(msg, "Unable to open generated document file '%s'", tmpfile);
	return HTLoadError(sink, 500, msg);
    }

    status = HTLoadScriptResult(fp, sink);
    fclose(fp);
#ifdef VMS
    delete(tmpfile);
    delete(tmpscriptfile);
#else
    unlink(tmpfile);
#endif /* not VMS */

    return status;
}

