/*
 * Copyright 1991 Pei-Yuan Wei.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */
/*
 * cl_generic.c
 *
 * class	: generic
 * superClass	: cosmic
 */
#include <stdio.h>
#include <ctype.h>
#include <sys/file.h>
#include <stdlib.h>
#include "error.h"
#include "file.h"
#include "hash.h"
#include "mystrings.h"
#include "ident.h"
#include "obj.h"
#include "vlist.h"
#include "attr.h"
#include "packet.h"
#include "class.h"
#include "slotaccess.h"
#include "cgen.h"
#include "cexec.h"
#include "classlist.h"
#include "cl_generic.h"
#include "misc.h"
#include "sys.h"
#include "glib.h"
#include "event.h"

#ifdef hpux
#include <time.h>
#else hpux
#include <sys/time.h>
#endif hpux

extern double cos(), sin();

int flag_cliprompt = 1;

#define DEBUG FALSE
#define RADIAN_TO_DEGREE_RATIO 0.017453293

SlotInfo cl_generic_NCSlots[] = {
	NULL
};
SlotInfo cl_generic_NPSlots[] = {
{
	STR_name,
	PTRS | SLOT_RW,
	(long)""
},{
	STR_parent,
	PTRS | SLOT_RW,
	(long)""
},{
	STR__parent,
	OBJP,
	NULL
},{
	STR_children,
	PTRS | SLOT_RW,
	(long)""
},{
	STR__children,
	OBJL,
	NULL
},{
	STR_message,
	PTRS,
	(long)""
},{
	STR_script,
	PTRS | SLOT_RW,
	(long)"usual();\n"
},{
	STR__script,
	PCOD,
	NULL,
},{
	STR__scriptSize,	/* used to detect ``seg fault'' */
	LONG,			/* should probably store in pcode itself */
	0,			/* at cost of complicating pcode format */
},{
	STR__varList,
	ATTR,
	NULL
},{
	STR_active,
	LONG | SLOT_RW,
	(long)0
},{
	NULL
}
};
SlotInfo cl_generic_CSlots[] = { /* COMMON SLOTS */
{
	STR_class,
	PTRS | SLOT_RW,
	(long)"generic"
},{
	NULL
}
};
SlotInfo cl_generic_PSlots[] = { /* PRIVATE SLOTS */
{
	STR__classInfo,
	CLSI,
	(long)&class_generic
},{
	NULL
}
};

SlotInfo *slots_generic[] = {
	(SlotInfo*)cl_generic_NCSlots,
	(SlotInfo*)cl_generic_NPSlots,
	(SlotInfo*)cl_generic_CSlots,
	(SlotInfo*)cl_generic_PSlots
};

MethodInfo meths_generic[] = {
{
	STR_accessible,
	meth_generic_accessible
},{
	STR_after,
	meth_generic_after
},{
	STR_alarm,
	meth_generic_alarm
},{
	STR_argument,
	meth_generic_argument
},{
	STR_ascii,
	meth_generic_ascii
},{
	STR_asciiVal,
	meth_generic_asciiVal
},{
	STR_bell,
	meth_generic_bell
},{
	STR_bellVolume,
	meth_generic_bellVolume
},{
	STR_clear,
	meth_generic_clear
},{
	STR_cli,
	meth_generic_cli
},{
	STR_clone,
	meth_generic_clone
},{
	STR_clone2,
	meth_generic_clone2
},{
	STR_concatenate,
	meth_generic_concatenate
},{
	STR_cos,
	meth_generic_cos
},{
	STR_countChars,		
	meth_generic_strlen			/* alias */
},{
	STR_countChildren,
	meth_generic_countChildren
},{
	STR_countItems,
	meth_generic_countItems
},{
	STR_countLines,
	meth_generic_countLines
},{
	STR_countWords,
	meth_generic_countWords
},{
	STR_cursorShape,
	meth_generic_cursorShape
},{
	STR_date,
	meth_generic_date
},{
	STR_deepObjectListSend,
	meth_generic_deepObjectListSend
},{
	STR_delay,
	meth_generic_delay
},{
	STR_depth,
	meth_generic_depth
},{
	STR_destroyVariable,
	meth_generic_destroyVariable
},{
	STR_environVar,
	meth_generic_environVar
},{
	STR_filter,
	meth_generic_filter,
},{
	STR_findPattern,
	meth_generic_findPattern,
},{
	STR_float,
	meth_generic_float
},{
	STR_format,
	meth_generic_format,
},{
	STR_freeSelf,
	meth_generic_freeSelf
},{
	STR_get,
	meth_generic_get
},{
	STR_getSelection,
	meth_generic_getSelection
},{
	STR_getVariable,
	meth_generic_getVariable
},{
	STR_gravity,
	meth_generic_gravity
},{
	STR_initialize,
	meth_generic_initialize
},{
	STR_int,
	meth_generic_int
},{
	STR_item,
	meth_generic_item
},{
	STR_isBlank,
	meth_generic_isBlank
},{
	STR_key,
	meth_generic_key
},{
	STR_listAllObjects,
	meth_generic_listAllObjects
},{
	STR_loadFile,
	meth_generic_loadFile
},{
	STR_makeTempFile,
	meth_generic_makeTempFile
},{
	STR_not,
	meth_generic_not
},{
	STR_nthChar,
	meth_generic_nthChar
},{
	STR_nthChild,
	meth_generic_nthChild
},{
	STR_nthItem,
	meth_generic_nthItem
},{
	STR_nthLine,
	meth_generic_nthLine
},{
	STR_nthObjectInList,
	meth_generic_nthObjectInList,
},{
	STR_nthSibling,
	meth_generic_nthSibling
},{
	STR_nthWord,
	meth_generic_nthWord
},{
	STR_objectListAppend,
	meth_generic_objectListAppend
},{
	STR_objectListCount,
	meth_generic_objectListCount
},{
	STR_objectListDelete,
	meth_generic_objectListDelete
},{
	STR_objectListPrepend,
	meth_generic_objectListPrepend
},{
	STR_objectListSend,
	meth_generic_objectListSend
},{
	STR_objectPosition,
	meth_generic_objectPosition
},{
	STR_pipe,
	meth_generic_pipe
},{
	STR_print,
	meth_generic_print
},{
	STR_printf,
	meth_generic_printf
},{
	STR_random,
	meth_generic_random
},{
	STR_saveFile,
	meth_generic_saveFile
},{
	STR_scanf,
	meth_generic_scanf,
},{
	STR_selectionInfo,
	meth_generic_selectionInfo
},{
	STR_set,
	meth_generic_set
},{
	STR_setMouse,
	meth_generic_setMouse
},{
	STR_setSelection,
	meth_generic_setSelection
},{
	STR_setVariable,
	meth_generic_setVariable
},{
	STR_sin,
	meth_generic_sin
},{
	STR_sleep,
	meth_generic_sleep
},{
	STR_str,
	meth_generic_str
},{
	STR_strlen,
	meth_generic_strlen
},{
	STR_system,
	meth_generic_system
},{
	STR_target,
	meth_generic_target
},{
	STR_time,
	meth_generic_time
},{
	STR_tool,
	meth_generic_tool
},{
	STR_version,
	meth_generic_version
},{
	STR_violaPath,
	meth_generic_violaPath
},{
	STR_watch,
	meth_generic_watch
},{
	STR_write,
	meth_generic_print		/* alias */
},{
	STR_writeln,
	meth_generic_writeln
},{
	NULL
}
};

ClassInfo class_generic = {
	slots_generic,		/* class slot information	*/
	meths_generic,		/* class methods		*/
	STR_generic,		/* class identifier number	*/
	&class_cosmic,		/* super class info		*/
};

/*
 * accessible(filepath)
 * 
 * Determine the accessibility of a file.
 *
 * Result: full file path on success, or "" on failure
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_accessible(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int accessible;
	char *path;
	result->type = PKT_STR;
	if (path = saveString(expandPath(argv[0].info.s, buff))) {
		accessible = access(path, R_OK);
		if ((accessible & R_OK) == 0) {
			result->info.s = path;
			return 1;
		}
		free(path);
	}
	result->info.s = "";
	return 0;
}

/*
 * after(milliseconds, object, [message list])
 *
 * Schedule to send messages to an object after a period of time.
 *
 * Result/Return: 1 if successful, 0 if error occured
 */
int meth_generic_after(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int msec;
	VObj *obj;
	Packet *messages;
	int size;

	result->type = PKT_INT;

	msec = PkInfo2Int(&argv[0]);
	obj = PkInfo2Obj(&argv[1]);

	if (!msec || !obj) {
		result->info.i = 0;
		return 0;
	}

	size = sizeof(struct Packet) * (argc - 2);
	messages = (Packet*)malloc(size);
	if (!messages) return 0;
	bcopy(&argv[2], messages, size);
	/* it's event loop's responsibility to free the message list */

	if (scheduleEvent(msec, sendMessagePackets, obj, argc - 2, messages)) {
		result->info.i = 1;
		return 1;
	}
	result->info.i = 1;
	return 0;
}

/*
 * alarm(?time, object, message)
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_alarm(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	if (argv[0].type == PKT_INT) {
		sys_alarm((unsigned)(argv[0].info.i));
		return 1;
	}
	return 0;
}

/*
 * argument();
 *
 * Return the string following the -ar option.
 *
 * Result: version string
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_argument(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	extern char *passthru_argument;	/* defined in cexec.c */
	result->info.s = passthru_argument;
	result->type = PKT_STR;
	return 1;
}

/*
 * ascii(ASCII code)
 *
 * Result: corresponding ASCII character
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_ascii(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_CHR;
	result->info.c = (char)argv[0].info.i;
	return 1;
}

/*
 * asciiVal(ASCII character)
 *
 * Result: corresponding ASCII code
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_asciiVal(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_INT;
	result->info.i = (int)argv[0].info.c;
	return 1;
}

/*
 * bell();
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_bell(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	SLBell();
	return 1;
}

/*
 * bellVolume(0-100);
 *
 * Result: volume value
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_bellVolume(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_INT;
	result->info.i = SLBellVolume(PkInfo2Int(argv));
	return 1;
}

/*
 * XXX
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_clear(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * cli();
 *
 * Invoke Command Line Interface to Viola language interpreter
 * 
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_cli(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{

#define LINE_LENGTH 300

	char c, cmdLine[LINE_LENGTH];
	int i, balance, finish = 0;

	do {
		if (flag_cliprompt) printf("<%s>: ", GET_name(VCurrentObj));
		i = 0;
		balance = 0;

		for (;;) {
			c = getchar();
			if (i < LINE_LENGTH - 1) {
			}
			cmdLine[i++] = c;
			if (c == ';' && (balance == 0)) {
				cmdLine[i] = '\0';
				break;
			} else if (c == '{' || c == '(') {
				balance++;
			} else if (c == '}' || c == ')') {
				balance--;
			} else if (c == EOF) {
				cmdLine[--i] = '\0';
				finish = 1;
				break;
			}
		}
/*		printf(">>>%s<<<\n", cmdLine);*/

		if (!AllBlank(cmdLine)) {
			result = execScript(VCurrentObj, result, cmdLine);

			if (flag_cliprompt && result) {
				dumpPacket(result);
				printf("\n");
			}
		}
	} while (!finish);

	return 1;
}

/*
 * clone(clone name suffix)
 * 
 * Make a clone self
 *
 * Result: clone object, and optinally name it
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_clone(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	VObj *cloneObj;

	if (!meth_generic_clone2(self, result, argc, argv)) return 0;
	cloneObj = result->info.o;
	if (cloneObj) {
		sendInitToChildren(cloneObj);
		result->type = PKT_OBJ;
		result->info.o = cloneObj;
		return 1;
	}
	result->type = PKT_OBJ;
	result->info.o = NULL;
	return 0;
}

int meth_generic_clone2(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	VObj *cloneObj, *obj;
	VObjList *olist, *newolist;
	Packet lresult;

	if (!meth_cosmic_clone2(self, result, argc, argv)) return 0;
	cloneObj = result->info.o;

	newolist = NULL;
	olist = GET__children(self);
	if (olist) {
		for (; olist; olist = olist->next) {
			if (!callMeth(olist->o, &lresult, argc, argv, 
				      STR_clone2)) {
				fprintf(stderr, 
				    "clone (at children cloning) failed\n");
				return 0;
			}
			newolist = appendObjToList(newolist, lresult.info.o);
			SET__parent(lresult.info.o, cloneObj);
			SET_parent(lresult.info.o, GET_name(cloneObj));
		}
	}
	SET__children(cloneObj, newolist);
	SET_children(cloneObj, saveString(OListToStr(newolist)));

	SET__varList(cloneObj, NULL);

	result->type = PKT_OBJ;
	result->info.o = cloneObj;
	return 1;
}

/*
 * concatenate(arglist)
 *
 * Result: concatenation of arguments as strings
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_concatenate(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp;

	result->type = PKT_STR;
	cp = PkInfos2Str(argc, argv);
	if (result->info.s = saveString(cp)) return 1;
	return 0;
}

/*
 * cos(degree)
 *
 * Cosine at the given degree.
 * 
 * Result: cosine value at given degree
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_cos(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_FLT;
	result->info.f = cos((double)(PkInfo2Flt(&argv[0]) * 
					RADIAN_TO_DEGREE_RATIO));
	return 1;
}

/*
 * countChildren();
 *
 * Equivalent to objectListCount("children")
 *
 * Result: number of children
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_countChildren(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	VObjList *olist = GET__children(self);
	int i;

	result->type = PKT_INT;
	if (olist) {
		for (i = 0; olist; olist = olist->next, i++);
		result->info.i = i;
		return 1;
	}
	result->info.i = 0;
	return 0;
}

/*
 * countItems(str)
 *
 * Comma seperate items.
 *
 * Result: number of items
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_countItems(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp = PkInfo2Str(&argv[0]);

	result->type = PKT_INT;
	result->info.i = numOfChar(cp, ',') + 1;

	return 1;
}

/*
 * countLines(str)
 *
 * Counts number of '\n' charcters.
 *
 * Result: number of lines
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_countLines(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp = PkInfo2Str(&argv[0]);

	result->type = PKT_INT;
	result->info.i = numOfChar(cp, '\n');

	return 1;
}

/*
 * countWords(str)
 *
 *
 *
 * Result: number of words
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_countWords(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * cursorShape([busy|idle])
 *
 * Change cursor shape.
 *
 * Result: current cursor shape
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_cursorShape(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *shape = PkInfo2Str(argv);

	result->type = PKT_STR;

	if (argc == 0 || !GET_window(self)) {
		result->info.s = "?";
		return 0;
	}
	
	if (!STRCMP(shape, "busy")) {
		result->info.s = "busy";
		GLChangeToBusyMouseCursor(GET_window(self));
	} else {
		result->info.s = "idle";
		GLChangeToNormalMouseCursor(GET_window(self));
	}
	return 1;
}

/*
 * date() 
 *
 * Result: date string, like ``Sat Feb 16 19:59:04 1991''
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_date(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	time_t theTime;
	time(&theTime);
	result->info.s = saveString(ctime(&theTime));
	result->type = PKT_STR;
	return 1;
}

/* deepObjectListSend(objectListName, arg1, arg2, ..., argn)
 *
 * Recursively send the arguments to each objects in the object list.
 *
 * Result: cleared
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_deepObjectListSend(self, result, argc, argv)
      VObj *self;
      Packet *result;
      int argc;
      Packet argv[];
{
      VObjList *olist = NULL;
      char *listName;

      listName = PkInfo2Str(&argv[0]);
      if (!STRCMP(listName, "children")) {
              olist = GET__children(self);
      }
      while (olist) {
              if (olist->o) {
                      sendMessagePackets(olist->o, &argv[1], argc - 1);
                      callMeth(olist->o, result, argc, argv, 
                               STR_deepObjectListSend);
              } else {
                      /* error */
              }
              olist = olist->next;
      }
      return 1;
}

/*
 * XXX
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_delay(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * XXX
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_depth(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * destroyVariable(attribute)
 *
 *
 * Result: the gotten attribute
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_destroyVariable(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * environVar(attribute)
 *
 * Get environment variable.
 *
 * Result: the gotten attribute
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_environVar(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	extern char **environ;

	result->type = PKT_STR;
	if (getEnvironVars(environ, PkInfo2Str(&argv[0]), buff)) {
		result->info.s = saveString(buff);
		return 1;
	}
	result->info.s = "";
	return 0;
}

/*
 * XXX
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_field(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * XXX
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_fieldList(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * filter(text)
 *
 * Result: replaces escape code into escape char
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_filter(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *inStr = PkInfo2Str(&argv[0]);
	char *outStr;
	char c, *cp;

	outStr = saveString(inStr);
	cp = outStr;

	result->info.s = outStr;
	result->type = PKT_STR;

	while (c = *inStr++) {
		if (c == '\\') {
			switch (c = *inStr++) {
			case 'n': 
				*cp++ = '\n'; 	/* newline */
				break;
			case 't':
				*cp++ = '\t'; 	/* horizontal tab */
				break;
			case 'b': 
				*cp++ = '\b'; 	/* backspace */
				break;
			case 'r': 
				*cp++ = '\r';	/* carriage */
				break;
			case 'f': 
				*cp++ = '\f'; 	/* form feed */
				break;
			case '\\': 
				*cp++ = '\\'; 	/* backslash */
				break;
			case '\'': 
				*cp++ = '\''; 	/* single quote */
				break;
			case '\"': 
				*cp++ = '\"';
				break;
			case EOF:
				break;
			default:
				*cp++ = '\\';
				*cp++ = c;
			}
		} else {
			*cp++ = c;
		}
	}
	*cp = '\0';
	return 1;
}

/*
 * findPattern(text, pattern)
 *
 * Result: position where the matching string pattern ends, or -1 if not found
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_findPattern(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp;
	char *inStr, *patStr;
	int ii = 0, pi = 0;
	int inLength, patLength;

	inStr = PkInfo2Str(&argv[0]);
	patStr = PkInfo2Str(&argv[1]);
	inLength = strlen(inStr);
	patLength = strlen(patStr);

	result->type = PKT_INT;
	for (cp = inStr; *cp; cp++) {
		if (*cp == patStr[pi]) {
			if (++pi >= patLength) {
				result->info.i = ii;
				return 1;
			}
			if (ii >= inLength) {
				result->info.i = -1;
				return 0;
			}
		} else {
			pi = 0;
		}
		ii++;
	}
	result->info.i = -1;
	return 0;
}

/*
 * float(argument)
 *
 * Convert argument to floating point type.
 *
 * Result: arguement in float
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_float(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_FLT;
	result->info.f = PkInfo2Flt(argv);
	return 1;
}

/* 
 * format(text, style [,style info])
 *
 * Simple text formatting kludge
 *
 * Result: arguement in float
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_format(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	if (!STRCMP(PkInfo2Str(&argv[1]), "split")) {
	}
/*
	result->type = PKT_STR;
	result->info.s = PkInfo2Flt(argv);
	return 1;
*/
	clearPacket(result);
	return 0;
}

/*
 * freeSelf()
 *
 * Free object attributes and.
 * 
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_freeSelf(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	meth_cosmic_freeSelf(self, result, argc, argv);
	return 1;
}

/*
 * XXX
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_gravity(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * get(attribute)
 *
 * Get an attribute: active, children, class, message, name, parent, script.
 *
 * Result: the gotten attribute
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_get(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int labelID = getIdent(PkInfo2Str(argv));

	switch (labelID) {
	case STR_name:
		result->info.s = GET_name(self);
		result->type = PKT_STR;
		return 1;

	case STR_class:
		result->info.s = GET_class(self);
		result->type = PKT_STR;
		return 1;

	case STR_classScript:
		result->info.s = GET_classScript(self);
		result->type = PKT_STR;
		return 1;

	case STR_parent:
		result->info.s = GET_parent(self);
		result->type = PKT_STR;
		return 1;

	case STR_children:
		result->info.s = GET_children(self);
		result->type = PKT_STR;
		return 1;

	case STR_message:
		result->info.s = GET_message(self);
		result->type = PKT_STR;
		return 1;

	case STR_script:
		result->info.s = GET_script(self);
		result->type = PKT_STR;
		return 1;

	case STR_active:
		result->info.i = GET_active(self);
		result->type = PKT_INT;
		return 1;
	}
	return 0;
}

int meth_generic_getSelection(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	return GLGetSelection(result);
}

/*
 * getVariable(variabel name)
 *
 * Get a named variable in the object.
 *
 * Result: the gotten attribute
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_getVariable(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp = PkInfo2Str(&argv[0]);

	result->type = PKT_STR;
	if (!cp) {
		result->info.s = "";
		return 0;
	}
	if (getVariable(GET__varList(self), cp, &(result->info.s))) {
		return 1;
	}
	result->info.s = "";
	return 0;
}

/*
 * initialize()
 *
 * Initializes object.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_initialize(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp;

	/* make internal pointer link to parent object */
	if (!GET__parent(self)) {
		if (cp = GET_parent(self)) 
			SET__parent(self, findObject(getIdent(cp)));
	}

	/* make internal pointer link to children objects */
	if (!GET__children(self)) {
		cp = GET_children(self);
		if (cp) if (*cp) SET__children(self, strOListToOList(cp));
	}
	clearPacket(result);

	return 1;
}

/*
 * int(float|str)
 *
 * Convert single argument to integer.
 *
 * Result: integer type of the argument
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_int(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_INT;
	result->info.i = PkInfo2Int(argv);
	return 1;
}

/*
 * isBlank(str)
 *
 *
 * Result: 1 or 0
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_isBlank(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_INT;
	result->info.i = AllBlank(PkInfo2Str(&argv[0])) ? 1 : 0;
	return 1;
}

/* XXX
 * item(str, n1 [,n2])
 *
 * Extract the item(s) ranged by n1 and n2. Comma (,) is the seperating 
 * character.
 *
 * Result: string containing the items
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_item(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 *
 * Result: key
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_key(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->info.c = keyStat_key;
	result->type = PKT_CHR;
	return 1;
}

/*
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_listAllObjects(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 1;
}

/*
 * objectListAppend(objList, obj)
 *
 * Append an object to a named object list.
 *
 * Result: object count, -1 if the list is not found
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_objectListAppend(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *listName;
	VObjList *olist = NULL;

	result->type = PKT_OBJ;
	listName = PkInfo2Str(&argv[0]);
	if (!STRCMP(listName, "children")) {
		VObj *obj = PkInfo2Obj(&argv[1]);
		olist = GET__children(self);
		if (obj) {
			SET__children(self, appendObjToList(olist, obj));
			result->info.o = obj;
			return 1;
		}
	}
	result->info.o = NULL;
	return 0;
}

/*
 * objectListCount(objList)
 *
 * Count the number of objects in a named object list.
 *
 * Result: object count, -1 if the list is not found
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_objectListCount(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i;
	char *listName;
	VObjList *olist = NULL;

	listName = PkInfo2Str(&argv[0]);
	if (!STRCMP(listName, "children")) {
		olist = GET__children(self);
	}
	result->type = PKT_INT;
	if (olist) {
		for (i = 0; olist; olist = olist->next, i++);
		result->info.i = i;
		return 1;
	}
	result->info.i = -1;
	return 0;
}

/*
 * objectListDelete(objList, obj)
 *
 * Delte an object from a named object list.
 *
 * Result: object count, -1 if the list is not found
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_objectListDelete(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	VObjList *olist = NULL;
	char *listName = PkInfo2Str(argv);

	clearPacket(result);
	if (!STRCMP(listName, "children")) {
		VObj *obj = PkInfo2Obj(&argv[1]);
		olist = GET__children(self);
		if (olist && obj) {
			olist = removeVObjListNode(&olist, obj);
			SET__children(self, olist);
			return 1;
		}
	}
	return 0;
}

/*
 * objectListPrepend(objList, obj)
 *
 * Prepend an object to a named object list.
 *
 * Result: object count, -1 if the list is not found
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_objectListPrepend(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *listName;
	VObjList *olist = NULL;

	result->type = PKT_OBJ;
	listName = PkInfo2Str(&argv[0]);
	if (!STRCMP(listName, "children")) {
		VObj *obj = PkInfo2Obj(&argv[1]);
		olist = GET__children(self);
		if (obj) {
			SET__children(self, prependVObjListNode(olist, obj));
			result->info.o = obj;
			return 1;
		}
	}
	result->info.o = NULL;
	return 0;
}

/*
 * objectListSend(objectListName, arg1, arg2, ..., argn)
 *
 * Send the arguments to each objects in the object list.
 *
 * Result: cleared
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_objectListSend(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	VObjList *olist = NULL;
	char *listName;

	listName = PkInfo2Str(&argv[0]);
	if (!STRCMP(listName, "children")) {
		olist = GET__children(self);
	}
	while (olist) {
		sendMessagePackets(olist->o, &argv[1], argc - 1);
		olist = olist->next;
	}
	clearPacket(result);
	return 1;
}

/*
 * objectPosition(obj, objList)
 *
 * Return the position an object in the named object list.
 * 
 * Result: position (starting from 0), -1 if object is not found in list
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_objectPosition(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	VObjList *olist = NULL;
	VObj *obj;
	char *listName;
	int i;

	obj = PkInfo2Obj(&argv[0]);
	listName = PkInfo2Str(&argv[1]);

	if (!STRCMP(listName, "children")) {
		olist = GET__children(self);
	}

	result->type = PKT_INT;
	for (i = 0; olist; olist = olist->next, i++) {
		if (olist->o == obj) {
			result->info.i = i;
			return 1;
		}
	}
	result->info.i = -1;
	return 0;
}

/*
 * loadFile(fileName)
 *
 * Loads a file and returns its content.
 *
 * Result: loaded file
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_loadFile(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp, *retStrp;

	result->type = PKT_STR;
	if (cp = saveString(expandPath(PkInfo2Str(&argv[0]), buff))) {
		if (loadFile(cp, &retStrp) != -1) {
			if (retStrp) {
				result->info.s = retStrp;
				free(cp);
				return 1;
			} else {
				sprintf(buff, "loadFile \"%s\" failed.\n", cp);
				messageToUser(self, MESSAGE_ERROR, buff);
			}
		} else {
			sprintf(buff,
				"Unable to open file '%s'. aborted.\n", cp);
			messageToUser(self, MESSAGE_ERROR, buff);
		}
		free(cp);
	} else {
		sprintf(buff, "no such file\n");
		messageToUser(self, MESSAGE_ERROR, buff);
	}
	result->info.s = "";
	return 0;
}

/*
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_makeTempFile(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->info.s = saveString(tmpnam(NULL));
	result->type = PKT_STR;
	return result->info.s ? 1 : 0;
}

/*
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_not(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * nthChar(string, n1, [n2]) 
 *
 * Result: character range from n1 to n2
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_nthChar(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	if (argc == 2) {
		int n;

		result->type = PKT_CHR;
		n = PkInfo2Int(&argv[1]);
		result->info.c = PkInfo2Str(&argv[0])[n];
	} else if (argc == 3) {
		int i, n1, n2;
		char *cp, *str;

		result->type = PKT_STR;
		n1 = PkInfo2Int(&argv[1]);
		n2 = PkInfo2Int(&argv[2]);
		str = PkInfo2Str(&argv[0]);
		cp = (char*)malloc(sizeof(char) * (n2 - n1 + 2));
		if (!cp) {
			result->info.s = NULL;
			return 0;
		}
		for (i = 0; n1 <= n2; n1++) {
			if (!str[n1]) break;
			cp[i++] = str[n1];
		}
		cp[i] = '\0';
		result->info.s = cp;
	}
	return 1;
}

/*
 * nthChild(n)
 * Returns the n'th object in self's children list.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_nthChild(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i = 0, n;
	char *listName;
	VObjList *olist;

	if (argc != 1) {
		clearPacket(result);
		return 0;
	}
	n = PkInfo2Int(&argv[0]);
	listName = PkInfo2Str(&argv[1]);

	olist = GET__children(self);

	result->type = PKT_OBJ;
	while (olist) {
		if (i++ >= n) {
			result->info.o = olist->o;
			return 1;
		}
		olist = olist->next;
	}
	result->info.o = NULL;
	return 0;
}

/*
 * nthItem(string, n)
 *
 * 
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_nthItem(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * nthLine(str, n1 [,n2])
 *
 * Extracts the specified lines (from line n1 to n2) from the string.
 * Line numbering start from 0.
 *
 * Result: the specified line(s)
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_nthLine(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int li, hi, lines, size;
	char *str;

	result->type = PKT_STR;
	str = PkInfo2Str(&argv[0]);
	if (argc == 2) {
		li = hi = PkInfo2Int(&argv[1]);
	} else if (argc == 3) {
		li = PkInfo2Int(&argv[1]);
		hi = PkInfo2Int(&argv[2]);
	} else {
		/* incorrect number of arguments */
		result->info.s = "";
		return 0;
	}
	if (result->info.s = getLines(li, hi, str, &size)) return 1;

	result->info.s = "";
	return 0;
}

/*
 * nthObjectInList(n, objectList)
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_nthObjectInList(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i = 0, n;
	char *listName;
	VObjList *olist;

	if (argc != 2) {
		clearPacket(result);
		return 0;
	}
	n = PkInfo2Int(&argv[0]);
	listName = PkInfo2Str(&argv[1]);

	if (!STRCMP(listName, "children")) {
		olist = GET__children(self);
	}

	result->type = PKT_OBJ;
	while (olist) {
		if (i++ >= n) {
			result->info.o = olist->o;
			return 1;
		}
		olist = olist->next;
	}
	result->info.o = NULL;
	return 0;
}


/*
 * nthSibling(n)
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_nthSibling(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i = 0, n;
	char *listName;
	VObjList *olist;
	VObj *parent;

	if (argc != 1) {
		clearPacket(result);
		return 0;
	}
	n = PkInfo2Int(&argv[0]);

	parent = GET__parent(self);
	if (parent) {
		olist = GET__children(parent);
		result->type = PKT_OBJ;
		while (olist) {
			if (i++ >= n) {
				result->info.o = olist->o;
				return 1;
			}
			olist = olist->next;
		}
		result->info.o = NULL;
	}
	return 0;
}

/*
 * nthWord(str, n1, n2)
 *
 * Result: the word string if successful, or "" if failed.n
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_nthWord(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *str;
	int n1, n2;

	str = PkInfo2Str(&argv[0]);
	if (!str) {
		result->type = PKT_STR;
		result->info.s = "";
		return 0;
	} else {
		str = saveString(str);
		result->type = PKT_STR;
		if (argc == 2) {
			n1 = n2 = PkInfo2Int(&argv[1]);
		} else if (argc == 3) {
			n1 = PkInfo2Int(&argv[1]);
			n2 = PkInfo2Int(&argv[2]);
		} else {
			result->info.s = "";
			return 0;
		}
		extractWord(str, n1, n2, buff);
		result->info.s = saveString(buff);
		free(str); /* argh! */
	}
	return 1;
}

int meth_generic_pipe(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int c;
	FILE *fp;
	char *buffp;

	clearPacket(result);

	fp = popen(PkInfo2Str(&argv[0]), PkInfo2Str(&argv[1]));
	if (!fp) {
		return 0;
	} else {
		buffp = buff;
		while ((c = fgetc(fp)) != EOF) {
			*buffp = c;
			buffp++;
		}
		*buffp = '\0';
		pclose(fp);
		result->info.s = saveString(buff);
		result->type = PKT_STR;
		return 1;
	}
}

/*
 * print(argument> [, ...])
 *
 * Print arguments to standard output.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_print(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i;

	clearPacket(result);
	for (i = 0; i < argc; i++) {
		switch (argv[i].type) {
		case PKT_OBJ:
			if (argv[i].info.o)
				printf("%s", GET_name(argv[i].info.o));
			else 
				printf("(NULL)");
		break;
		case PKT_INT:
			printf("%d", argv[i].info.i);
		break;
		case PKT_FLT:
			printf("%f", argv[i].info.f);
		break;
		case PKT_CHR:
			printf("%c", argv[i].info.c);
		break;
		case PKT_STR:
			if (argv[i].info.s)
				printf("%s", argv[i].info.s);
			else 
				printf("(NULL)");
		break;
		case PKT_ARY:
			if (argv[i].info.y) {
				int n;
				Array *array = argv[i].info.y;
				for (n = 0; n < array->size; n++)
					printf("%d ", array->info[n]);
			}
		break;
		case PKT_ATR: {
			Attr *attrp;
			attrp = argv[i].info.a;
			for (; attrp; attrp = attrp->next) {
				printf("id=%d val=%d:", 
					attrp->id, attrp->val);
				dumpPacket((Packet*)attrp->val);
				printf("\n");
			}
		}
		break;
		default:
			printf("??");
			dumpPacket(&argv[i]);
			return 0;
		}
	}
	return 1;
}

/*
 * printf(argument [, ...])
 *
 * Print arguments to standard output.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_printf(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i;

	clearPacket(result);
	for (i = 0; i < argc; i++) {
		switch (argv[i].type) {
		case PKT_OBJ:
			printf("%s", GET_name(argv[i].info.o));
		break;
		case PKT_INT:
			printf("%d", argv[i].info.i);
		break;
		case PKT_FLT:
			printf("%f", argv[i].info.f);
		break;
		case PKT_CHR:
			printf("%c", argv[i].info.c);
		break;
		case PKT_STR:
			printf("%s", argv[i].info.s);
		break;
		case PKT_ARY:
			if (argv[i].info.y) {
				int n;
				Array *array = argv[i].info.y;
				for (n = 0; n < array->size; n++)
					printf("%d ", array->info[n]);
			}
		break;
		default:
			printf("??");
			dumpPacket(&argv[i]);
			return 0;
		}
	}
	return 1;
}


int meth_generic_random(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_INT;
	result->info.i = random((unsigned)PkInfo2Int(&argv[0]));
	return 1;
}

/*
 * saveFile(fileName, str)
 *
 * Save a file and returns its content.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_saveFile(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp;

	result->type = PKT_INT;
	if (cp = saveString(expandPath(PkInfo2Str(&argv[0]), buff))) {
		if (saveFile(cp, PkInfo2Str(&argv[1])) != -1) {
			free(cp);
			result->info.i = 1;
			return 1;
		}
	}
	sprintf(buff, "saveFile(\"%s\", data) failed.\n", cp);
	messageToUser(self, MESSAGE_ERROR, buff);
	result->info.i = 0;
	return 0;
}

/*
 * scan(argument [, ...])
 *
 * Print arguments to standard output.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_scan(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i;
	clearPacket(result);
	return 0;
}

/*
 * scanf(argument [, ...])
 *
 * Print arguments to standard output.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_scanf(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i;

	clearPacket(result);
	for (i = 0; i < argc; i++) {
		switch (argv[i].type) {
		case PKT_OBJ:
			printf("%s", GET_name(argv[i].info.o));
		break;
		case PKT_INT:
			printf("%d", argv[i].info.i);
		break;
		case PKT_FLT:
			printf("%f", argv[i].info.f);
		break;
		case PKT_CHR:
			printf("%c", argv[i].info.c);
		break;
		case PKT_STR:
			printf("%s", argv[i].info.s);
		break;
		case PKT_ARY:
			if (argv[i].info.y) {
				int n;
				Array *array = argv[i].info.y;
				for (n = 0; n < array->size; n++)
					printf("%d ", array->info[n]);
			}
		break;
		default:
			printf("??");
			dumpPacket(&argv[i]);
			return 0;
		}
	}
	return 1;
}

/*
 * selectionInfo()
 *
 * Result: [0] object
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_selectionInfo(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	Packet *packet0 = makePacket();
	Attr *attrp;

	result->type = PKT_ATR;
	result->info.a = attrp = makeAttr(0, packet0);
	if (xselectionObj) {
		packet0->info.o = xselectionObj;
		packet0->type = PKT_OBJ;
		return 1;
	} else {
		packet0->info.o = NULL;
		packet0->type = PKT_OBJ;
		return 0;
	}
}

/*
 * returns non-zero if set operation succeded, zero otherwise.
 *
 *
 * Result: ...
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_set(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int labelID = getIdent(PkInfo2Str(argv));
	char *str;

	switch (labelID) {
	case STR_name:
		result->info.s = SaveString(PkInfo2Str(&argv[1]));
		SET_name(self, result->info.s);
		result->type = PKT_STR;
		putHashEntry(objID2Obj, storeIdent(GET_name(self)), (int)self);
		return 1;

	case STR_parent: {
		HashEntry *entry;
		VObj *obj;

		result->info.s = SaveString(PkInfo2Str(&argv[1]));
		SET_parent(self, result->info.s);
		result->type = PKT_STR;
		if (entry = getHashEntry(symStr2ID, (int)result->info.s))
			if (obj = findObject(entry->val)) {
				SET__parent(self, obj);
				return 1;
			}
		SET__parent(self, NULL);
		return 1;
		}

	case STR_children: {
		char *cp = PkInfo2Str(&argv[1]);
		VObjList *objl;

		result->type = PKT_INT;
		if (cp) {
			objl = strOListToOList(cp);
		}
		if (objl) {
			/* free olist (GET__children); */
			SET_children(self, cp);
			SET__children(self, objl);
			result->info.i = 1;
			return 1;
		} else {
			printf("set(\"children\", ...) failed. list unchanged\n");
			SET_children(self, "");
			SET__children(self, NULL);
			result->info.i = 0;
			return 0;
		}
	}

	case STR_message:
		result->info.s = SaveString(PkInfo2Str(&argv[1]));
		SET_message(self, result->info.s);
		result->type = PKT_STR;
		return 1;

	case STR_script: {
		union PCode *pcode = GET__script(self);
		result->info.s = SaveString(PkInfo2Str(&argv[1]));
		SET_script(self, result->info.s);
		result->type = PKT_STR;
		if (pcode) {
			free(pcode);
			SET__script(self, NULL);
		}
		return 1;
	}

	case STR_active:
		result->info.i = PkInfo2Int(&argv[1]);
		SET_active(self, result->info.i);
		result->type = PKT_INT;
		return 1;
	}
	clearPacket(result);
	return 0;
}

/*
 * setSelection(value)
 *
 * Result: 
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_setSelection(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	GLSetSelection(self, saveString(PkInfo2Str(&argv[0])));
	return 1;
}

/*
 * setVariable(attribute, value)
 *
 * Result: 
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_setVariable(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	clearPacket(result);
	return 0;
}

/*
 * setMouse(x, y)
 *
 * Result: the root offset
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_setMouse(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
/*	Window w = PkInfo2Int(argv);*/
	int x = PkInfo2Int(&argv[0]);
	int y = PkInfo2Int(&argv[1]);
/*
	if (!w) {
		w = rootWindow;
	}
*/
	/* relative to root */
	clearPacket(result);
	XWarpPointer(display, None, rootWindow, 0,0,0,0, x, y);

	return 1;
}

/*
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_sin(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_FLT;
	result->info.f = sin((double)(PkInfo2Flt(&argv[0]) * 
					RADIAN_TO_DEGREE_RATIO));
	return 1;
}

/*
 * sleep(sec)
 *
 * Result: seconds slept
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_sleep(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_INT;
	result->info.i = sleep(PkInfo2Int(&argv[0]));
	return 1;
}

/*
 * str(argument [, ...])
 *
 * Result: arguments in string type
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_str(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_STR;
	result->info.s = PkInfos2Str(argc, argv);
	return 1;
}

/*
 * strlen(str)
 */
/*
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_strlen(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	result->type = PKT_INT;
	result->info.i = strlen(PkInfos2Str(argc, argv));
	return 1;
}

/*
 * system(cmdline1 [, cmdline2 , ...])
 *
 * Call to the system(3) call.
 *
 * Result: return value from system()
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_system(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i;

	result->type = PKT_INT;
	for (i = 0; i < argc; i++) {
		result->info.i = system(PkInfo2Str(&argv[i]));
	}
	return 1;
}

/*
 * target() 
 *
 * Return the object that was mark()'ed.
 *
 * Result: target object
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_target(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	char *cp;

	result->type = PKT_OBJ;
	if (argc == 0) {
		result->info.o = VTargetObj;
		return 1;
	}
	cp = PkInfo2Str(&argv[0]);
	if (cp) 
		if (result->info.o = findObject(getIdent(cp))) {
			VTargetObj = result->info.o;
			sendMessage1(VResourceObj, "targetSet");
			return 1;
		}
	return 0;
}

/*
 * time() 
 *
 * Result: elapsed seconds since 00:00:00 GMT, January 1, 1970
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_time(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	time_t theTime;
	result->info.i = time(&theTime);
	result->type = PKT_INT;
	return 1;
}

/*
 * tool([tool name]);
 *
 * Tool name: "action", "move", "reparent", "resize", "target"
 * if no argument is given, the current tool is returned.
 *
 * Result: name of the selected/current tool
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_tool(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	int i;
	char *cp;

	result->type = PKT_STR;
	if (argc == 0) {
		result->info.s = toolID2Str[currentTool];
		return 1;
	}
	if (cp = PkInfos2Str(argc, argv)) {
		for (i = 0; toolID2Str[i]; i++) {
			if (!STRCMP(cp, toolID2Str[i])) {
				currentTool = i;
				result->info.s = toolID2Str[currentTool];
				return 1;
			}
		}
	}
	result->info.s = "";
	return 0;
}

/*
 * version();
 *
 * Return the Viola version identifier string.
 *
 * Result: version string
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_version(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	extern char *version;	/* defined in main.c */
	result->info.s = version;
	result->type = PKT_STR;
	return 1;
}

/*
 * violaPath(str)
 *
 * Set a new path for searching viola object files.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_violaPath(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	if (argc == 0) {
		int i = 0;
		extern char *viola_path[];

		result->type = PKT_STR;
		buff[0] = '\0';
		while (viola_path[i]) {
			strcat(buff, viola_path[i]);
			if (viola_path[++i]) strcat(buff, " ");
		}
		result->info.s = saveString(buff);
	} else if (argc == 1) {
		result->type = PKT_INT;
		if ((result->info.i = setViolaPath(PkInfo2Str(&argv[0]))) > 0)
			return 1;
	}
	return 0;
}

/*
 * watch(flag)
 *
 * Specifies variable tracking flags. NOT YET IMPLEMENTED.
 *
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_watch(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	extern int flag_vwatch; /* defined in cexec.c */

	clearPacket(result);
	flag_vwatch = PkInfo2Int(&argv[0]);

	return 1;
}

/*
 * 
 * 
 * Result: unaffected
 * Return: 1 if successful, 0 if error occured
 */
int meth_generic_writeln(self, result, argc, argv)
	VObj *self;
	Packet *result;
	int argc;
	Packet argv[];
{
	if (meth_generic_print(self, result, argc, argv)) {
		printf("\n");
		return 1;
	} else {
		return 0;
	}
}

