/*
 * ast.c
 */
/*
 * Copyright 1991 Pei-Yuan Wei.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */
#include <stdio.h>
#include <strings.h>
#include "y.tab.h"
#include "ast.h"

AST *ASTStack;

int ASTStackSize = DEFAULT_AST_STACK_SIZE; 
	/* overiden with option, if large script is anticipated */
int ASTStackIdx = 0;

int ASTTypeStrSize;
char *ASTTypeStr[] = {
	"EXPR_PLUS",	/* the EXPR's must be grouped in the front. */
	"EXPR_MINUS",   /* because codeGen() depends on it */
	"EXPR_MOD",
	"EXPR_MULT",
	"EXPR_DIV",
	"EXPR_EQ",
	"EXPR_NE",
	"EXPR_LT",
	"EXPR_LE",
	"EXPR_GT",
	"EXPR_GE",
	"EXPR_AND",
	"EXPR_OR",
	"EXPR_CMP",	/* end of EXPR's grouping */
	"LIST",
	"LISTC",
	"STACK",
	"MOVTO",
	"MOVTO_LIST",
	"CALL",
	"CHAR",
	"IF",
	"INTEGER",
	"FLOAT",
	"STMTS",
	"STRING",
	"WHILE",
	"RETURN",
	"SWITCH",
	"CASE",
	"CASE_SEG",
	"CASE_CLAUSE",
	"DEFAULT_CLAUSE",
	"BREAK",
	"BODY",
	"EXPR",
	"DO",
	"FOR",
	"INC_PRE",
	"INC_POST",
	"DEC_PRE",
	"DEC_POST",
	"UMINUS",
	"PLUS_MOVTO",
	"MINUS_MOVTO",
	"MOD_MOVTO",
	"MULT_MOVTO",
	"DIV_MOVTO",
	"PERSISTENT",
	"REF",
	"LIST_REF",
	NULL
};

char myinput()
{
	char c = yyscript[yyscriptidx++]; 
	if (c == '\n') yylineno++; 
	return c;
}

int init_ast()
{
	for (ASTTypeStrSize = 0; ASTTypeStr[ASTTypeStrSize]; ASTTypeStrSize++);

	do {
		if (ASTStackSize < 50) {
			fprintf(stderr, "can't get enough memory...\n");
			return 0;
		}

		ASTStack = (AST*)malloc(ASTStackSize * sizeof (struct AST));
		if (!ASTStack) {
			ASTStackSize = ASTStackSize / 2;
			fprintf(stderr,
				"trying to malloc for AST stack (size=%d)\n",
				ASTStackSize);
		}
	} while (!ASTStackSize);

	return 1;
}

AST *makeAST(type)
     int type;
{
	if (++ASTStackIdx >= ASTStackSize) {
		/* try to make a larger AST stack... */
		fprintf(stderr, 
			"internal error: AST stack overflow: ASTStackIdx=%d, ASTStackSize=%d.\n", 
			ASTStackIdx, ASTStackSize);
		fprintf(stderr,
			"use ``-astack'' option to allocate larger stack.\n");
		/* XXX unimplemented */
		exit(1);
	}
	bzero(&ASTStack[ASTStackIdx], sizeof(struct AST));
	ASTStack[ASTStackIdx].type = type;
	ASTStack[ASTStackIdx].lineno = yylineno;

	return &ASTStack[ASTStackIdx];
}

int countASTSiblings(ast)
     AST *ast;
{
	int i;
	for (i = 0; ast; ast = ast->next) i++;
	return i;
}

void printAST(ast, tab)
     AST *ast;
     int tab;
{
	int i;
	AST *children;

	for (; ast; ast = ast->next) {

		fprintf(stderr, "%d\t", ast->lineno);
		for (i = 0; i < tab; i++) putc(' ', stderr);
		fprintf(stderr, "%s\t", ASTTypeStr[ast->type]);
		switch (ast->type) {
		case AST_CALL:
		case AST_REF:
			fprintf(stderr,
				"info=%d", ast->attribute.info.i);
		break;

		case AST_STRING:
			fprintf(stderr,	
				"info=\"%s\"", (char*)(ast->attribute.info.s));
		break;

		case AST_CHAR:
			fprintf(stderr, 
				"info='%c' (%d)",
				(char)(ast->attribute.info.c),
				(int)(ast->attribute.info.c));
		break;

		case AST_FLOAT:
			fprintf(stderr,
				"info=%f", (float)(ast->attribute.info.f));
		break;

		case AST_INTEGER:
		default:
			fprintf(stderr,
				"info=%d", (int)(ast->attribute.info.i));
		break;
		}
		putc('\n', stderr);

		if (ast->children) printAST(ast->children, tab + 4);
	}
}

