/* 
 *
 *  process-query.c - check and preprocess query  // Nov 16 93
 *  (C) Christian Neuss (neuss@igd.fhg.de)
 *      Fraunhofer IGD 
 *  
 *  arguments:
 *  arg1 = where to look for the .context file
 *  arg2 = query string
 *  
 *  ouput:
 *  A number of name value/pairs will be printed on stdout.
 *  They can be entered in a shell script via "eval".
 *
 */

/* 
 * Syntax: [GETDEF] w1 [{AND|OR} w2]* [IN f1 [f2]*] [CONTEXT {icib|...}] 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h> 
#include <assert.h>
#include <ctype.h> 
#include <sys/param.h> /* MAXPATHLEN */
 
#ifdef EBUG
#  define dprintf fprintf
#else
#  define dprintf if(0)fprintf
#endif

static char *contextNames[] = {
"ICIB", "VIDEO", "TELECOM", "IT", "DEFACTO", "OTHER", "ORG", ""
};

#define assert2(ex,str)  {if (!(ex)){(void)fprintf(stderr,\
  "Error: %s, file \"%s\", line %d\n",str, __FILE__, __LINE__);return(1);}}

static int process_query(char file[], char query[]);
static int parse(
  char query[], char words[], char bool[], 
	char fields[], char context[], char *pdef
);
static int get_context(char file[], char context[], char dir[]);

int main(int argc, char *argv[])
{
  if(argc!=3){
    printf("%s: wrong argument count!\n",argv[0]);
    return 1;
  }else{
    return process_query(argv[1], argv[2]);
  }
}

int process_query(char file[], char query[])
{
  char *base, isdef, buf[MAXPATHLEN]; int found=0;
  char words[512], fields[512], context[512], bool[512], dir[MAXPATHLEN];
  words[0] = bool[0] = fields[0] = context[0] = dir[0] = '\0';

  /* 1. parse query */
  if(parse(query,words,bool,fields,context ,&isdef)==0){
    dprintf(stderr,"%s\n",isdef?"(GETDEF)":"(No getdef)");
    dprintf(stderr,"%s\n",words);
    dprintf(stderr,"%s\n",bool);
    dprintf(stderr,"%s\n",fields);
    dprintf(stderr,"%s\n",context);
  }else{
    printf("err=\"%s\"; \n",words);
    return 1;
  }

  /* 2. get context */
  if(strlen(context)>0){
    assert2(*file=='/',"\"file\" doesn't start with '/'");
    strcpy(buf,file);
    for(base=buf;*base;base++)if(*base=='/')found=1;
    if(found){
      for(;*base!='/';base--);
      *base='\0';
      base++;
    }else{
      base=buf; 
    }
    strcat(buf,"/.context");
    dprintf(stderr,"context file: [%s]\n",buf);
    if(get_context(buf,context,dir)){
      printf("err=\"%s\"; ",dir);
      return 1;
    }else{
      dprintf(stderr,"context read: [%s]\n",dir);
    }
  }

  /* 3. print out the results */
  printf("theWords=\"%s\"; ",words);
  printf("theBool=\"%s\"; ",bool);
  printf("theFields=\"%s\"; ",fields);
  printf("theDir=\"%s\"; ",dir);
  printf("theDef=\"%s\"; ",isdef?"GETDEF":"");
  return 0;
}

typedef enum {
  T_DEF, T_AND, T_OR, T_IN, T_CONTEXT, T_OTHER} myTokenType;
typedef enum {
  S_START, S_KEY, S_ANDOR, S_IN, S_CONTEXT, S_FIELD,
  S_DEF, S_ERR, S_FINAL} myStateType;
  
static myTokenType token(char word[]);
static char* currentpos(char query[], int pos);

static int 
parse(
  char query[], char words[], char bool[], char fields[], 
	char context[], char *pdef
)
{
  char buf[512], *p, *q, *errmsg="No error", isDef=0;
  myStateType state=S_START;
  *pdef=0; /* set to false */
  
  strcpy(buf, query);
  assert2(query,"parse called with query string NULL");
  p=buf;
  for(;p;p=q){
    myTokenType tok;
    q=index(p,' ');
    if(q) {
		  *q++='\0';
			/* compress multiple blanks in query into one */
			while(*q==' ') q++;
		}
    tok = token(p);
    switch(state){
    case S_START: /* start of query analysis */
      switch(tok){
      case T_DEF: 
        dprintf(stderr," state -> DEF\n");
        state=S_DEF; break;
      case T_OTHER: 
        dprintf(stderr," state -> KEY\n");
        if(words[0]!='\0') strcat(words," "); 
        strcat(words,p); 
				state=S_KEY; break;
      default: 
        errmsg = "Search word expected"; 
        state=S_ERR; break;
      }
      break;
    case S_DEF: /* last thing read was "GETDEF" */
      isDef = 1; 
      switch(tok){
      case T_OTHER: 
        dprintf(stderr," state -> KEY\n");
        if(words[0]!='\0') strcat(words," "); 
        strcat(words,p); 
				state=S_KEY; break;
      default: 
        errmsg = "Search word expected"; 
        state=S_ERR; break;
      }
      break;
    case S_KEY: /* last thing read was a searchword */
      switch(tok){
      case T_AND: 
        dprintf(stderr," state -> ANDOR\n");
        strcat(bool,"&"); state=S_ANDOR; break;
      case T_OR: 
        dprintf(stderr," state -> ANDOR\n");
        strcat(bool,"+"); state=S_ANDOR; break;
      case T_CONTEXT: 
        dprintf(stderr," state -> CONTEXT\n");
        state=S_CONTEXT; break;
      case T_IN: 
        dprintf(stderr," state -> IN\n");
        if(isDef) {
          errmsg = "No fields allowed with GETDEF query"; 
          state=S_ERR; break;
        } else {
          state=S_IN; break;
        }
      default: 
        errmsg = "Keyword AND, OR, IN or CONTEXT expected"; 
        state=S_ERR; break;
      }
      break;
    case S_ANDOR: /* last thing read was AND or OR */
      switch(tok){
      case T_OTHER: 
        dprintf(stderr," state -> KEY\n");
        if(words[0]!='\0') strcat(words," "); 
        strcat(words,p); 
				state=S_KEY; break;
      default: 
        errmsg = "No keywords (AND, OR, IN..) allowed here"; 
        state=S_ERR; break;
      }
      break;
    case S_IN:  /* last thing read was IN */
      switch(tok){
      case T_OTHER: 
        dprintf(stderr," state -> FIELD\n");
        strcat(fields,p); state=S_FIELD; break;
      default: 
        errmsg = "Field name(s) expected"; 
        state=S_ERR; break;
      }
      break;
    case S_FIELD: /* last thing read was a word after IN */
      switch(tok){
      case T_OTHER: 
        dprintf(stderr," state -> FIELD\n");
        strcat(fields,p); state=S_FIELD; break;
      case T_CONTEXT: 
        dprintf(stderr," state -> CONTEXT\n");
        state=S_CONTEXT; break;
      default: 
        errmsg = "Field name or 'CONTEXT' expected"; 
        state=S_ERR; break;
      }
      break;
    case S_CONTEXT: /* last thing read was a word after IN */
      switch(tok){
      case T_OTHER:   
        dprintf(stderr," state -> FINAL\n");
        strcat(context,p); state=S_FINAL; break;
      default: 
        errmsg = "End of query expected"; 
        state=S_ERR; break;
      }
      break;
    default: /* whass goin on here? */
      errmsg = "unknown state encountered"; 
      state=S_ERR; break;
      break;
    } /* end of state switch */

    if(state==S_ERR){
      break; /* leave loop */
    }
  }
  if((state!=S_KEY) && (state!=S_FIELD) && (state!=S_FINAL)){
    char *cur, str[1024];
    cur=currentpos(query,p-buf);
    sprintf(words,"Syntax error while parsing query:\\\\n%s\\\\n%s",
		        cur,errmsg);
    return 1;
  }
  *pdef=isDef;
  return 0;
}

static int get_context(char file[], char context[], char dir[])
{
  FILE *fp; char buf[1024], found=0;
  
  sprintf(dir,"context file %s: '%s' not found",file,context);

  if(!strcasecmp(context,"ICIB")){ 
    strcpy(dir,"/icib");  /* sigh.. CN */
    return 0;
  }
  
  if ((fp=fopen(file,"r"))==NULL){
      sprintf(dir,"Cannot open context file %s",file);
      return 1;
  }
  
  for(;fgets(buf,1023,fp)!=NULL;){
    char *p,*q;
    for(p=buf;(*p)&&(*p!='/');p++);
    if(*p=='\0'){
      continue; 
    }
    q=index(buf,' ');
    if(q) *q='\0';
    else continue;

    dprintf(stderr,"%s <-> %s\n",context,buf); 
    if(!strcasecmp(context,buf)){ 
      found=1;
			/* dispose of trailing whitespace */
			for(q=p;(*q)&&!isspace(*q);q++);
      *q='\0';
      strcpy(dir,p);
      break;
    }
  }
  
  fclose(fp);
  return found? 0 : 1;
}

static myTokenType 
token(char word[])
{
   if(!strcasecmp(word,"GETDEF"))  return T_DEF;
   if(!strcasecmp(word,"AND"))     return T_AND;
   if(!strcasecmp(word,"OR"))      return T_OR;
   if(!strcasecmp(word,"IN"))      return T_IN;
   if(!strcasecmp(word,"CONTEXT")) return T_CONTEXT;
   return T_OTHER;
}

static char* 
currentpos(char query[], int pos)
{
   static char theMessage[1024]; 
   int i;
   sprintf(theMessage,"%s\\\\n", query); 
   for(i=0;i<pos;i++)strcat(theMessage,".");
   strcat(theMessage,"^\\\\n"); 
   return theMessage;
}