// Copyright 1993 by Peter Bentley <pete@tecc.co.uk>
//
// Permission to use, copy, modify, distribute, and sell this software and its
// documentation for any purpose is hereby granted without fee, provided that
// the above copyright notice appear in all copies and that both that
// copyright notice and this permission notice appear in supporting
// documentation, and that the name of Peter Bentley not be used in
// advertising or publicity pertaining to distribution of the software without
// specific, written prior permission.  Peter Bentley makes no representations
// about the suitability of this software for any purpose.  It is provided
// "as is" without express or implied warranty.
//
// PETER BENTLEY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
// INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
// EVENT SHALL PETER BENTLEY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
// DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
// PERFORMANCE OF THIS SOFTWARE.
//
// winflex.cpp - Main part of WinFlex, a Windows FlexFax client
//
#include <zapp.hpp>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <dir.h>
#include "winsock.h"
#include "tcsock.h"
#include "pstring.h"					// Inline string functions
#include "ffclient.h"					// FlexFax client definitions
#pragma hdrstop
#include "winflex.h"

static char RCSid[] = "$Id: winflex.cpp,v 1.1 1993/09/02 12:33:58 pete Exp pete $";

/////////////////////////////////////////////////////////////////////////////
// wfSettings.  Settings specific to WinFlex
//
struct wfSettings {
	int		autoPrint;		// Look for faxes printed to a spool file
	zString	spoolDir;		// Directory for spool files
	zString	spoolFile;		// File name attached to printer
    zString	spoolPref;		// Prefix part of spool file name

	wfSettings() {				// Zero out all the strings
		spoolDir = "";
		spoolFile = "";
        spoolPref = "";
		autoPrint = 0;
	};
	~wfSettings() { };

	void readFromIni(char *s ="WinFlex", char *iFile = "WIN.INI") {
		char	buffer[128];
		GetPrivateProfileString(s, "SpoolDir", "", buffer, 127, iFile);
		spoolDir = buffer;
		GetPrivateProfileString(s, "SpoolFile", "", buffer, 127, iFile);
		spoolFile = buffer;
		GetPrivateProfileString(s, "SpoolPrefix", "WFSP", buffer, 127, iFile);
		spoolPref = buffer;
		autoPrint = GetPrivateProfileInt(s, "AutoPrint", 0, iFile);
	}
	void writeToIni(char *s ="WinFlex", char *iFile = "WIN.INI") {
		WritePrivateProfileString(s, "SpoolDir", spoolDir, iFile);
		WritePrivateProfileString(s, "SpoolFile", spoolFile, iFile);
		WritePrivateProfileString(s, "SpoolPrefix", spoolPref, iFile);
		WritePrivateProfileString(s, "AutoPrint", autoPrint ? "1" : "0", iFile);
	}
};

/////////////////////////////////////////////////////////////////////////////
// ffSetDialogue.  Form dialogue for editting the global settings
//
class ffSetDialogue : public tcForm {
  protected:
	zEditLine		*hostEdit, *userEdit, *fullEdit, *emailEdit;
	zEditLine		*templEdit, *dirEdit, *fileEdit;
	zStaticText		*dirText, *fileText;
	zCheckBox		*autoCheck;
	ffSettings		*fset;
    wfSettings		*wset;

  public:
	ffSetDialogue(zWindow *parent, ffSettings *f, wfSettings *w);
	~ffSetDialogue() {
		// Used to delete the control, but zApp seems to do this for us...
		// Leastways, we get a GPF if we try to delete them. :-(
	}
	int	storeData() {						// Successful completion
		int rc = tcForm::storeData();		// Tell form to store its data

		if (rc != 0) {
			if (wset->autoPrint)			// Validate printer setup
				wset->autoPrint = testPrinterSetup();

			fset->writeToIni();				// Save to WIN.INI as well
			wset->writeToIni();
		}
		return rc;
	}
	int			handleCheck(zNotifyEvt *);	// Handle the checkbox
	int 		testPrinterSetup();
	int			findPrinter(char *port);
};

ffSetDialogue::ffSetDialogue(zWindow *parent, ffSettings *f, wfSettings *w)
: tcForm(parent, zResId("WinFlex_Setup"))
{
	fset = f;							// Save the settings pointers
	wset = w;

	// Attach zEditLines to the appropriate controls in the dialogue
	hostEdit = new zEditLine(this, IDD_FAXHOST, &fset->faxHost);
	userEdit = new zEditLine(this, IDD_FAXUSER, &fset->faxUser);
	fullEdit = new zEditLine(this, IDD_FAXFULLNAME, &fset->faxFullName);
	emailEdit = new zEditLine(this, IDD_FAXEMAIL, &fset->faxEmail);
	templEdit = new zEditLine(this, IDD_TEMPLATE, &fset->coverTemplate);

	dirEdit = new zEditLine(this, IDD_SPOOLDIR, &wset->spoolDir);
    dirText = new zStaticText(this, IDD_DIRTEXT);
	fileEdit = new zEditLine(this, IDD_SPOOLFILE, &wset->spoolFile);
    fileText = new zStaticText(this, IDD_FILETEXT);
	autoCheck = new zCheckBox(this, IDD_AUTOPRINT, &wset->autoPrint);

	autoCheck->setNotifyClicked(this, (NotifyProc) &ffSetDialogue::handleCheck);

	setControlsToDefault();					// Set control contents
	show();									// Display everything
	handleCheck(NULL);						// Test check box state
	modal();								// Go...
}


int ffSetDialogue::handleCheck(zNotifyEvt *) {

	if (autoCheck->check()) {
		dirEdit->enable();
		dirText->enable();
		fileEdit->enable();
		fileText->enable();
	}
	else {
		dirEdit->disable();
        dirText->disable();
		fileEdit->disable();
        fileText->disable();
    }
	return 0;
}

int ffSetDialogue:: testPrinterSetup() {
	char buffer[128];
	char portName[128];
	struct stat sbuf;

    // First off, check whether the spool directory exists
	if (stat(wset->spoolDir, &sbuf) < 0) {		// Can't stat it
		if (errno == ENOENT) {
			zMessage msg(this,
						 "The directory you wish to spool to does not exist.\n"
						 "Shall I create it?",
						 "Create directory?",
						 MB_YESNO | MB_ICONQUESTION);
			if (msg.isYes()) {
//				statusLine("Creating directory...");
				if (mkdir(wset->spoolDir) < 0) {
					zMessage m (this, "Unable to create directory\n"
								"Auto-printing disabled.",
								"Error",
								MB_ICONEXCLAMATION);
					return 0;
				}
//				statusLine("Directory created OK...");
			}
			else {
				zMessage m(this,
						   "No spool directory.\nAuto-printing disabled.",
						   "Information",
						   MB_ICONINFORMATION);
				return 0;
			}
		}
		else {
			zMessage msg(this, "Unable to stat spool directory!\n"
						 "Auto-printing disabled",
						 "Information", MB_ICONINFORMATION);
			return 0;
		}
	}
	else if (!(sbuf.st_mode & S_IFDIR)) {
		zMessage msg(this, "The name you entered for the spool directory "
					 "exists but does not refer to a directory.\n"
					 "Auto-printing disabled",
					 "Information",
					 MB_ICONSTOP);
		return 0;
	}

	// See if there is an entry in [ports] for the spool file
	sprintf(portName, "%s\\%s", (char *) wset->spoolDir,
			(char *) wset->spoolFile);
	strupr(portName);
	GetProfileString("ports", portName, "XXXX", buffer, 127);
	if (!strcmp(buffer, "XXXX")) {
		zMessage msg(this, "No printer port exists for the file name "
					 "you specified.\n"
					 "Shall I create an entry in WIN.INI for you?",
					 "Create port?",
					 MB_YESNO | MB_ICONQUESTION);
		if (msg.isYes()) {
//			statusLine("Creating WIN.INI entry...");
			if (!WriteProfileString("ports", portName, "")) {
				zMessage m(this, "Unable to create entry in WIN.INI?!?",
							"Error", MB_ICONEXCLAMATION);
				return 0;
			}
			else {
				WriteProfileString(NULL, NULL, NULL);
//				statusLine("Entry created...");
			}
		}
		else {
			zMessage msg(this, "You will need to edit your WIN.INI "
						 "file and add a line to the [ports] section "
						 "before printing to the fax will work.",
						 "Information",
						 MB_ICONINFORMATION);
		}
	}

	// See if there are any printers attached to the port we specified
	if (!findPrinter(portName)) {
		zMessage msg(this, "You will have to attach a PostScript printer to "
				"the printer port (using Control Panel) before you can "
				"send faxes by printing to it.",
				"Information",
				MB_ICONINFORMATION);

    }
	return 1;
}

int ffSetDialogue::findPrinter(char *port) {
	char	key[1024];
    char	value[80];

//	statusLine("Port: '%s'", port);
	GetProfileString("devices", NULL, "", key, 1024);
	for (char *s = key; *s != '\0'; s += strlen(s) + 1) {
		GetProfileString("devices", s, "", value, 80);
//		statusLine("String: '%s' : '%s'", s, value);
		char *iniport = strchr(value, ',');
		if (iniport == NULL) continue;
		iniport++;			// Skip over the comma
//		statusLine("INI port: '%s'", iniport);
		if (!stricmp(iniport, port)) {
			sprintf(key, "The printer currently attached to port '%s' "
					"is '%s'", iniport, s);
			zMessage msg(this, key, "Information", MB_ICONINFORMATION);
			return 1;
		}
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// ffDestDialogue.  Form dialogue for editting an ffDest.
//

class ffDestDialogue : public tcForm {
  protected:
	zEditLine		*numEdit, *recEdit, *companyEdit, *regardEdit, *commentEdit;
	zCheckBox		*coverCheck;
	zGroupBox		*coverBox;
	zStaticText		*nameText, *companyText, *regardText, *commentText;
	zRadioGroup		*sizeGroup, *resGroup;
	int				currSize, currRes;
	ffDest			*dest;
	
  public:
	ffDestDialogue(zWindow *parent, ffDest *d);
	int handleCheck(zNotifyEvt *);			// Handle the checkbox
	int storeData();
};

ffDestDialogue::ffDestDialogue(zWindow *parent, ffDest *d)
: tcForm(parent, zResId("WinFlex_Recipient"))
{
	dest = d;

	// Attach zApp controls
	numEdit = new zEditLine(this, IDD_FAXNO, &dest->teleNum);
	recEdit = new zEditLine(this, IDD_RECIPIENT, &dest->recipName,
							FLD_NOTREQUIRED);
	companyEdit = new zEditLine(this, IDD_COMPANY, &dest->recipCompany,
								FLD_NOTREQUIRED);
	regardEdit = new zEditLine(this, IDD_REGARDING, &dest->regarding,
								FLD_NOTREQUIRED);
	commentEdit = new zEditLine(this, IDD_COMMENTS, &dest->comments,
								FLD_NOTREQUIRED);

	coverCheck = new zCheckBox(this, IDD_SENDCOVER, &dest->doCover);
    dest->doCover = TRUE;				// Default from WIN.INI???
	coverBox = new zGroupBox(this, IDD_COVER_GRP);
	nameText = new zStaticText(this, IDD_NAME_TEXT);
	companyText = new zStaticText(this, IDD_COMP_TEXT);
	regardText = new zStaticText(this, IDD_REGARD_TEXT);
	commentText = new zStaticText(this, IDD_COMM_TEXT);

	sizeGroup = new zRadioGroup(this, IDD_SIZE_A4, IDD_SIZE_LETTER, &currSize);
	currSize = IDD_SIZE_A4;				// Default from WIN.INI ???
	resGroup = new zRadioGroup(this, IDD_RES_NORMAL, IDD_RES_FINE, &currRes);
	currRes = IDD_RES_FINE;				// Default from WIN.INI ???

	coverCheck->setNotifyClicked(this,
								 (NotifyProc) &ffDestDialogue::handleCheck);

	setControlsToDefault();				// Set up data in controls
	show();								// Display...
	handleCheck(NULL);					// Test check box state
	modal();							// Go...
}

int ffDestDialogue::handleCheck(zNotifyEvt *) {

	if (coverCheck->check()) {
		recEdit->enable();
		companyEdit->enable();
        regardEdit->enable();
		commentEdit->enable();
		nameText->enable();
		companyText->enable();
        regardText->enable();
		commentText->enable();
		coverBox->enable();
	}
	else {
		recEdit->disable();
		companyEdit->disable();
        regardEdit->disable();
		commentEdit->disable();
		nameText->disable();
		companyText->disable();
		regardText->disable();
		commentText->disable();
		coverBox->disable();
	}
    return 0;
}

int	ffDestDialogue::storeData() {					// Successful completion

	int rc = tcForm::storeData();	// Tell form to store its data
	if (rc != 0) {
		// Yes, these are hard coded.  Feel free to port the
        // page size database & send me the diffs....
		switch (currSize) {
			case IDD_SIZE_LETTER:
				dest->pWidth = 215;
				dest->pLen = 279;
				break;
			case IDD_SIZE_A4:
				dest->pWidth = 209;
				dest->pLen = 296;
			default:
               	break;
		}
		switch (currRes) {
			case IDD_RES_NORMAL:
				dest->vRes = 98;
               	break;
			case IDD_RES_FINE:
               	dest->vRes = 196;
			default:
               	break;
		}
	}
	return rc;
   }

/////////////////////////////////////////////////////////////////////////////
// Event structure WM_SPOOLERSTATUS events (trivial)

class zSpoolerEvt : public zEvent {
  public:
	WORD jobStatus() 	{ return parm1();	};	// Job status
	WORD jobsLeft()		{ return loParam();	};	// Jobs in print queue
};


/////////////////////////////////////////////////////////////////////////////
// WinFlexApp.  The application level class.  Pretty primitive so
// far --- has a zTextPane for scrolling output and a minimal menu...
//

class WinFlexApp : public zAppFrame {
	zTextPane		*tp;
	ffSettings		fsettings;
	wfSettings		wsettings;

	int				jobSeq;				// Print job sequence number

  public:
	WinFlexApp(char *title);
	~WinFlexApp();
	int				command(zCommandEvt *ev);		// Simple commands
	int				fileFax(zCommandEvt *);			// Fax command handler
	void			sendFax(const char *fileName);	// Send a fax
	int				spoolerStatus(zSpoolerEvt *);	// Get print jobs
	void			Status(char *txt) {
		tp->put(txt);
        tp->put("\n");
	}
	void			makeSpoolName(zString &name) {
		char buffer[10];
		sprintf(buffer, "%03d", ++jobSeq);
		name = wsettings.spoolDir;
		name &= "\\";
		name &= wsettings.spoolPref;
		name &= buffer;
		name &= ".PS";
	}
};

/////////////////////////////////////////////////////////////////////////////

WinFlexApp::WinFlexApp(char *title)
  : zAppFrame(0, new zSizer, zSTDFRAME, title) {

	  menu(new zMenu(this, zResId("WinFlex_Menu")));	// Create menu
	  // Send File|Fax menu commands to the sendFax() member.
	  menu()->setCommand(this, (NotifyProc) &WinFlexApp::fileFax,IDM_SENDFAX);
	  tp = new zTextPane(this, new zSizeWithParent);	// Create text pane
	  tp->show();										// Display text pane
	  fsettings.readFromIni();							// Read settings
	  wsettings.readFromIni();

	  zIcon *ico = new zIcon(zResId("WinFlex_Icon"));	// Set up icon
	  setIcon(ico);

	  jobSeq = 0;										// No jobs sent yet

	  if (wsettings.autoPrint) app->setHandler(this, (NotifyProc) &WinFlexApp::spoolerStatus,
											WM_SPOOLERSTATUS);
};

WinFlexApp::~WinFlexApp() {

	if (tp != NULL) delete tp;							// Delete
}

/////////////////////////////////////////////////////////////////////////////
// sendFax().  Send a fax.  Get a file name (modally), then a single
// destination.  Then create a ffJobDialogue to monitor (and start off)
// the job submission
//

static char *filespecs[] = {					// Data for file open dialogue
	"PostScript(tm) Files (*.ps)", "*.ps",
	"All Files(*.*)", "*.*",
	NULL, NULL
};

int WinFlexApp::fileFax(zCommandEvt *) {

	if (fsettings.faxHost == "") {				// Quick sanity check
		zMessage msg(this, "No Fax host configured yet.\n"
				 "Please use the Settings dialogue", "Error", MB_ICONHAND);
		return 1;
	}

	// Get a file name
	zFileOpenForm ofrm(this, "Select a file to fax", NULL, filespecs);
	if (!ofrm.completed()) {			// Cancelled
		statusLine("Aborted...");
	}
	else {
		sendFax(ofrm.name());
    }
    return 1;
}

void WinFlexApp::sendFax(const char *fileName) {

	ffFile *file = new ffPsFile(fileName);		// Assume it's PostScript
	
	ffDest *dest = new ffDest;					// Get a destination
	ffDestDialogue dfrm(this, dest);
	if (dfrm.completed()) {
		dfrm.storeData();
    }
	else {										// Cancelled. Delete ffFile
		delete file;
		delete dest;
		statusLine("Aborted...");
		return;
	}

	ffSendJob job(file, dest, &fsettings);			// Set up the job

	ffJobDialogue	dlg(this, &job);				// ...and send it

	statusLine("Send complete");
}


/////////////////////////////////////////////////////////////////////////////
// Command message handler
//

int WinFlexApp::command(zCommandEvt *ev) {

	switch (ev->cmd()) {
	  case IDM_FILEEXIT:				// File|Exit menu option selected
		app->quit();					// Kill the application
		break;

	  case IDM_FILEABOUT:				// File|About menu option
		{
			tcForm dlg(this, zResId("WinFlex_About"));	// About box
			dlg.show();					// Show it modally
			dlg.modal();
		}
		break;

	  case IDM_FILESETUP:				// File|Setup menu option
		{
			if (wsettings.autoPrint)
				app->removeHandler(this,
								   (NotifyProc) &WinFlexApp::spoolerStatus,
								   WM_SPOOLERSTATUS);
			ffSetDialogue dlg(this, &fsettings, &wsettings);
			if (dlg.completed()) dlg.storeData();	// Update if OK pressed
		  	if (wsettings.autoPrint)
				app->setHandler(this,
								(NotifyProc) &WinFlexApp::spoolerStatus,
								WM_SPOOLERSTATUS);
		}
		break;

	  case IDM_STATUSSEND:
		{
			tp->clearRect();							// Clear text window
			tp->moveHome();
			ffSendStatusJob job(&fsettings);			// Set up the job
			ffJobDialogue	dlg(this, &job);			// ...and do it
		}
		break;

	  case IDM_STATUSRECV:
		{
			tp->clearRect();							// Clear text window
			tp->moveHome();
			ffRecvStatusJob job(&fsettings);			// Set up the job
			ffJobDialogue	dlg(this, &job);			// ...and do it
		}
		break;

		default:
        	return 0;					// Unknown event
	}
	return 1;							// Handled OK
}

/////////////////////////////////////////////////////////////////////////////
// Spooler status message handler
//

int WinFlexApp::spoolerStatus(zSpoolerEvt *ev) {
	int verbose = 0;

	if (!wsettings.autoPrint) {					// Sanity check
		statusLine("Internal oddity.  spoolerStatus called when disabled.");
		return 1;
	}

	if (ev->jobsLeft() != 0) {			// Queue is not empty yet
		if (verbose) statusLine("Waiting for print queue to empty...");
	}
	else {								// Queue is empty. Look for a job
		if (ev->jobStatus() != 0) {		// Failed job...
			if (verbose) statusLine("Ignoring failed print job...");
		}
		else {
			time_t		now;
			struct stat	sbuf;
			zString		dataFile;			// Data from Print Manager
            zString		newFile;			// Renamed file

			dataFile = wsettings.spoolDir;
			dataFile &=  "\\";
			dataFile &= wsettings.spoolFile;

			if (stat((char *) dataFile, &sbuf) < 0) {	// Anything for us?
				if (verbose) statusLine("No valid file (%s) spooled for the fax...",
										(char *) dataFile);
				return 1;
			}
			time(&now);
			if ((sbuf.st_mtime > now) || ((now - sbuf.st_mtime) > 120)) {
				statusLine("Insane time on print spool file...");
				unlink(dataFile);		// Nuke it. (This is a bit fascist)
				return 1;
			}
			int renames = 0;
			makeSpoolName(newFile);
			while (rename(dataFile, newFile) < 0) {
					if ((errno == EACCES) && (++renames < 16)) {
					makeSpoolName(newFile);
					continue; 			// File exists.  Try again
					}
					statusLine("Failed to move print spool file '%s': %s",
								(char *) dataFile, strerror(errno));
					return 1;
			}
			statusLine("Faxing %s...", (char *) newFile);
			sendFax(newFile);
            statusLine("Deleting spool file %s...", (char *) newFile);
			unlink(newFile);
		}
	}
	return 1;
}

WinFlexApp *mainWnd;					// For statusLine()


void zApp::main() {						// zApp entry point

	mainWnd = new WinFlexApp("WinFlex");	// Create an application
	mainWnd->show();						// Display it
	go();									// Run...
	delete mainWnd;							// Now vape it
}

/////////////////////////////////////////////////////////////////////////////
// Global statusLine function.
// This version vsprintf's its arguments into a buffer and then'
// gets the application window to display them
//

int statusLine(const char *fmt, ...) {
	char	buffer[256];
	va_list	args;

	va_start(args, fmt);
	vsprintf(buffer, fmt, args);
	va_end(args);

	mainWnd->Status(buffer);

	return 1;
}
