// Copyright 1993 by Peter Bentley <pete@tecc.co.uk>
//
// Permission to use, copy, modify, distribute, and sell this software and its
// documentation for any purpose is hereby granted without fee, provided that
// the above copyright notice appear in all copies and that both that
// copyright notice and this permission notice appear in supporting
// documentation, and that the name of Peter Bentley not be used in
// advertising or publicity pertaining to distribution of the software without
// specific, written prior permission.  Peter Bentley makes no representations
// about the suitability of this software for any purpose.  It is provided
// "as is" without express or implied warranty.
//
// PETER BENTLEY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
// INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
// EVENT SHALL PETER BENTLEY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
// DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
// PERFORMANCE OF THIS SOFTWARE.
//
#include <zapp.hpp>
#include <string.h>
#include <stdarg.h>
#include "winsock.h"
#include "tcsock.h"

static char RCSid[] = "$Id: tcsock.cpp,v 1.1 1993/09/02 12:33:58 pete Exp pete $";

/////////////////////////////////////////////////////////////////////////////
// tcSocket Member variables and functions
//
int 			tcSocket::instCount = 0;	// Instance count
tcSockManager	*tcSocket::tcMgr = NULL;	// Global manager object

/////////////////////////////////////////////////////////////////////////////
// Contstructor & destructor
//
tcSocket::tcSocket(int type) {

	if (instCount++ == 0) {			// First instance
		tcMgr = new tcSockManager;	// Create a manager object
	}

	sock = socket(AF_INET, type, 0);	// Get a SOCKET from Winsock
	Init(sock);
}


tcSocket::tcSocket(SOCKET s) {

	if (instCount++ == 0) {			// First instance
		tcMgr = new tcSockManager;	// Create a manager object
	}

	sock = s;
	Init(sock);
}

void tcSocket::Init(SOCKET sock) {

	if (sock == INVALID_SOCKET) {
    	statPrint("Socket creation failure: %d", WSAGetLastError());
	}
	else {
        tcMgr->addSock(this);			// Set up in manager list
		evMask = 0;  					// Clear event mask
        updateMask();
	}
}

tcSocket::~tcSocket() {

	tcMgr->delSock(this);				// Remove entry from manager
	Close();

	if (--instCount == 0) {				// Last instance
		delete tcMgr;					// Delete the manager
		tcMgr = NULL;
	}
}


/////////////////////////////////////////////////////////////////////////////
// Connect: Connect to a remote server
//

int tcSocket::Connect(const char *pName, int iPort) {
	sockaddr_in	addr;
	hostent     *hent;

	statPrint("Connect %s:%d", pName, iPort);
	hent = gethostbyname(pName);
    if (hent == NULL)
    {
    	statPrint("Host name lookup failure for '%s'\n", pName);
        return SOCKET_ERROR;
    }

    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = *((long *) hent->h_addr);
    addr.sin_port = htons(iPort);
    statPrint("IP Address = %s\n", inet_ntoa(addr.sin_addr));

	addMask(FD_CONNECT);				// Get connection notifications
	int rc = connect(sock, (sockaddr *) &addr, sizeof(addr));
    if ((rc == SOCKET_ERROR) && (WSAGetLastError() != WSAEWOULDBLOCK))
    {
		Close();
        return SOCKET_ERROR;
	}
	return 0;
}

int tcSocket::Connect(const char *host, const char *service, const char *protocol) {

	servent *ent = getservbyname(service, protocol);
	if (ent == NULL) {
		Close();
		statPrint("Unable to find service name '%s'", service);
		return SOCKET_ERROR;
	}
	return Connect(host, ntohs(ent->s_port));
}

/////////////////////////////////////////////////////////////////////////////
// Listen().  Listen for incoming connections
//
int tcSocket::Listen(int port) {
	sockaddr_in	addr;

	addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    addr.sin_port  = htons(port);
	if ((bind(sock, (sockaddr *) &addr, sizeof(addr)) == SOCKET_ERROR) ||
		(listen(sock, 5) == SOCKET_ERROR)) {
		Close();
        return SOCKET_ERROR;
    }
	addMask(FD_ACCEPT);			// Get notification when connections arrive
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// Accept().  Accept a new connection
//
tcSocket *tcSocket::Accept() {

	SOCKET ns = accept(sock, NULL, NULL);
	if (ns == INVALID_SOCKET) {
		return NULL;
	}
	return new tcSocket(ns);			// Create new socket
}


/////////////////////////////////////////////////////////////////////////////
// Default action functions for socket events
//
void tcSocket::Connected(int error) {
	statPrint("Connected: error %d", error);
}

void tcSocket::Accepted(int error) {
	statPrint("Accepted: error %d", error);
}

void tcSocket::Closed(int error) {
	statPrint("Closed: error %d", error);
}

void tcSocket::Readable(int error) {
	statPrint("Readable: error %d", error);
}

void tcSocket::Writable(int error) {
	statPrint("Writable: error %d", error);
}

void tcSocket::OOBData(int error) {
	statPrint("OOBData: error %d", error);
}

/////////////////////////////////////////////////////////////////////////////
// Status printing
//
void	tcSocket::printStatus(const char *txt) {
	statusLine(txt);				// Go to default status line
}

void	tcSocket::statPrint(const char *fmt, ...) {
	char	buffer[256];
	va_list	args;

	va_start(args, fmt);
	vsprintf(buffer, fmt, args);
	va_end(args);
	printStatus(buffer);
}

/////////////////////////////////////////////////////////////////////////////
// tcBufSocket member functions
//

tcBufSocket::tcBufSocket(int rbs, int wbs)
: tcSocket(SOCK_STREAM) {				// Only makes sense with TCP sockets

	Init(rbs, wbs);
}

tcBufSocket::tcBufSocket(SOCKET s, int rbs, int wbs) : tcSocket(s) {

	Init(rbs, wbs);
}

void tcBufSocket::Init(int rbs, int wbs) {

	rbSize = wbSize = nrBuf = nwBuf = 0;	// Initialise
	if ((rBuf = new char[rbs]) != NULL) rbSize = rbs;
	if ((wBuf = new char[wbs]) != NULL) wbSize = wbs;
}

tcBufSocket::~tcBufSocket() {

	if (rBuf != NULL) delete rBuf;
	if (wBuf != NULL) delete wBuf;
}


int tcBufSocket::send(const char *data, int amt) {

	if (amt == -1) amt = strlen(data);		// Default amount
	if (amt > wrSpace()) {
		statPrint("Out of write buffer space");
		return SOCKET_ERROR;
	}
	if (amt > 0)  {
		memcpy(wBuf + nwBuf, data, amt);		// Copy the data in
		nwBuf += amt;
    }
	addMask(FD_WRITE);						// Get message when writable
	return wrSpace();
}

int tcBufSocket::recv(char *buf, int amt) {

	if (amt > nrBuf) amt = nrBuf;			// Limit to amount buffered
	if (amt != 0) {
		memcpy(buf, rBuf, amt);				// Give data to user
		rdPullUp(amt);						// Re-align the data 
		if (rdSpace() > 0) addMask(FD_READ); // Ask for more
	}
	return amt;
}

void tcBufSocket::Readable(int error) {		// Called by socket manager

	if ((error == 0) && (rdSpace() != 0)) {	// Try to fill buffer
		int amt = ::recv(sock, rBuf + nrBuf, rdSpace(), 0);
		if (amt < 0) {
			error = WSAGetLastError();		// Save error number
		}
		else if (amt > 0) {
			nrBuf += amt;					// Update count
			if (rdSpace() == 0)				// No space for more data!
				delMask(FD_READ);
			dataRead();						// Call callback
		}
	}
	if ((error != 0) && (error != WSAEWOULDBLOCK)) {
		readError(error);					// Pass on to error handler
	}
}

void tcBufSocket::Writable(int error) {		// Called by socket manager

	if (error != 0) statusLine("Writable(%d) called", error);
	if ((error == 0) && (nwBuf != 0)) {		// Try to write data
		int amt = ::send(sock, wBuf, nwBuf, 0);
		if (amt < 0) {
			error = WSAGetLastError();		// Save error value
		}
		else if (amt > 0) {					// Actually wrote something
			wrPullUp(amt);
		}
	}
	if ((error != 0) && (error != WSAEWOULDBLOCK)) {
		writeError(error);					// Pass on to error handler
	}
	if (nwBuf == 0) {
		delMask(FD_WRITE);					// Nothing more to write
		dataWritten();						// Call callback
	}
}

void tcBufSocket::readError(int error) {

	statPrint("Read error %d", error);
}

void tcBufSocket::writeError(int error) {

	statPrint("Write error %d", error);
}

/////////////////////////////////////////////////////////////////////////////
// Line buffered sockets...
//


// Read a line of data out of the buffer
int tcLineBufSocket::readLine(char *buffer, int len) {

	if (nrBuf == 0) return -1;			// No data
	char *s = (char *) memchr(rBuf, '\n', nrBuf);	// Find end of line
	if (s == NULL) return -1;			// TODO: semantics...
	int linelen = (s - rBuf) + 1;		// Space needed to hold line
	if (linelen > len) return -1;		// Too big... TODO: Something better
	if (recv(buffer, linelen) != linelen) {
		statPrint("Internal error in readLine()!");
		return 0;
	}
	linelen--;							// Get rid of the \n
	buffer[linelen] = '\0';
	return (linelen);
}

void tcLineBufSocket::dataRead() {		// Called by tcBuffSocket

	if (memchr(rBuf, '\n', nrBuf) != NULL) {	// Got a line?
		while (memchr(rBuf, '\n', nrBuf) != NULL) lineRead();
	}
	else {								// Test for overflow
		if (rdSpace() == 0) {			// TODO: Finish this
			statPrint("Warning! Read buffer overflow!");
		}
	}
}



/////////////////////////////////////////////////////////////////////////////
// tcSockManager member functions
//
/////////////////////////////////////////////////////////////////////////////
// Constructor & Destructor
//

tcSockManager::tcSockManager() {

	// Tell zApp to pass all WM_SOCKET messages send to the application's
	// root window to our sockHandler() member (q.v.)
	app->setHandler(this, (NotifyProc) &tcSockManager::sockHandler,	WM_SOCKET);

	// Start Winsock.  TODO: handle version numbers properly.
	int rc = WSAStartup(0x0101, &wsData);
	if (rc != 0) {
		statusLine("Socket startup failure: %d", WSAGetLastError());
	}

	// Zero out the list of sockets
	for (int i = 0; i < SOCKET_MAX; i++) {
		socklist[i] = NULL;
	}
}


tcSockManager::~tcSockManager() {

	// Tell zApp we don't want no more messages
	app->removeHandler(this, (NotifyProc) &tcSockManager::sockHandler,
						WM_SOCKET);

	WSACleanup();					// Close down Winsock
}

/////////////////////////////////////////////////////////////////////////////
// sockHandler().  This function gets called whenever the winsock DLL
// dispatches a WM_SOCKET message to the application.  This function
// uses the SOCKET handle to find a tcSocket, then invokes an appropriate
// member function in that object

int	tcSockManager::sockHandler(sockEvent *ev) {

	tcSocket *s = findSock(ev->sock());		// Find the correct tcSocket
	if (s == NULL) {
// This is not worth logging.  If a socket with FD_CLOSE set is
// closed from the Winsock end at the same time as the remote end,
// then an FD_CLOSE may well be delivered after the socket has gone...
//		statusLine("Message received for unknown socket %u", ev->sock());
		return 0;
	}
	else {						// Got one.  Pick a member function to call
		switch (ev->event()) {
			case FD_READ:		s->Readable(ev->error()); break;
			case FD_WRITE:		s->Writable(ev->error()); break;
			case FD_OOB:		s->OOBData(ev->error()); break;
			case FD_ACCEPT:		s->Accepted(ev->error()); break;
			case FD_CONNECT:	s->Connected(ev->error()); break;
			case FD_CLOSE:		s->Closed(ev->error()); break;
		}
		return 1;
	}
}


/////////////////////////////////////////////////////////////////////////////
// Utility functions for managing the list of sockets.
// Currently these use a linear search down an array of tcSockets, which
// is nearly optimal if only one (or maybe two) tcSockets exist,
// but is cr*p if there are lots.

// Find a socket in the list
tcSocket *tcSockManager::findSock(SOCKET s) {
	for (int i = 0; i < SOCKET_MAX; i++) {
		if ((socklist[i] != NULL) && (socklist[i]->winSock() == s)) {
			return socklist[i];
		}
	}
	return NULL;
}

// Add a socket to the list
BOOL tcSockManager::addSock(tcSocket *s) {
	for (int i = 0; i < SOCKET_MAX; i++) {
		if (socklist[i] == NULL) {
			socklist[i] = s;
			return TRUE;
		}
	}
	return FALSE;
}

// Remove a socket from the list
void tcSockManager::delSock(tcSocket *s) {
	for (int i = 0; i < SOCKET_MAX; i++) {
		if (socklist[i] == s){
			socklist[i] = NULL;
		}
	}
}

