/*	$Header: /usr/people/sam/fax/contrib/faxview/RCS/ImageView.c++,v 1.19 93/09/15 16:29:03 sam Exp $
/*
 * Copyright (c) 1990, 1991, 1992, 1993 Sam Leffler
 * Copyright (c) 1991, 1992, 1993 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */
#include "ImageView.h"
#include "Bitmap.h"
#include "tiffio.h"
#include <stdlib.h>
#include <math.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

extern	Display* xDisplay;
extern	int xScreen;
extern	Visual* xVisual;

static const float PAGEWIDTH = 8.5;
static const float PAGELENGTH = 11.;

ImageView::ImageView(fxBool forceSmall, Widget w) : widget(w)
{
    dispResolution = (forceSmall ? 66. : 86.);
    width = (int)(PAGEWIDTH*dispResolution);
    height = (int)(PAGELENGTH*dispResolution);
    if (DisplayHeight(xDisplay, xScreen) < height ||
      DisplayWidth(xDisplay, xScreen) < width) {
	dispResolution = 66.;
	width = (int)(PAGEWIDTH*dispResolution);
	height = (int)(PAGELENGTH*dispResolution);
    }
    raster = new u_char[width * height];
    bitstep0 = new u_char*[width];
    bitstep1 = new u_char*[width];
    rowoff = new u_short[width];
    haveImage = FALSE;
    needCmap = FALSE;
    isopen = FALSE;
    putImage = TRUE;
    filterHeight = 0;
    stepDstWidth = 0;
    stepSrcWidth = 0;
    contrast = EXP;
    photometric = (u_short) -1;

    Arg args[2];
    XtSetArg(args[0], XtNwidth, width);
    XtSetArg(args[1], XtNheight, height);
    XtSetValues(widget, args, 2);

    ximage = XCreateImage(xDisplay, xVisual,
	/*depth*/ 8, ZPixmap, /*offset*/ 0,
	(char*) raster, width, height,
	/*bitmap pad*/ 8, /*bytes/line*/ 0);
    ximage->byte_order = MSBFirst;
}

ImageView::~ImageView()
{
    delete raster;
    delete bitstep0;
    delete bitstep1;
    delete rowoff;
    XDestroyImage(ximage);
}

void
ImageView::open()
{
    u_long masks[1];
    Colormap defCmap = DefaultColormap(xDisplay, xScreen);
    u_long xcmap[32];
    if (XAllocColorCells(xDisplay, defCmap, False, masks, 0, xcmap, 32) < 0) {
	fprintf(stderr, "Unable to allocate colormap.\n");
	exit(-1);
    }
    for (int i = 0; i < 32; i++) {
	if (xcmap[i] > 511) {
	    fprintf(stderr, "Allocated too large a colormap index (%u)\n",
		xcmap[i]);
	    exit(-2);
	}
	cmap[i] = (u_char) xcmap[i];
    }
    /*
     * Create an appropriate GC
     */
    XGCValues gc_values;
    gc_values.function = GXcopy;
    gc_values.plane_mask = AllPlanes;
    gc_values.foreground = XBlackPixel(xDisplay, xScreen);
    gc_values.background = XWhitePixel(xDisplay, xScreen);
    xWinGc = XCreateGC(xDisplay, XtWindow(widget),
        GCFunction | GCPlaneMask | GCForeground | GCBackground, &gc_values);
    /*
     * Create the pixmap and load the image
     */
    xImagePixmap = XCreatePixmap(xDisplay, RootWindow(xDisplay, xScreen),
	ximage->width, ximage->height, 8);
    isopen = TRUE;
}

void
ImageView::update()
{
    XEvent ev;
    ev.type = Expose;
    ev.xexpose.x = ev.xexpose.y = 0;
    ev.xexpose.width = width;
    ev.xexpose.height = height;
    expose(&ev);
}

void
ImageView::expose(XEvent* event)
{
    if (haveImage) {
	if (putImage) {
	    XPutImage(xDisplay, xImagePixmap, xWinGc, ximage,
		0, 0, 0, 0, ximage->width, ximage->height);
	    putImage = FALSE;
	}
	if (needCmap)
	    setupCmap();
	int x = event->xexpose.x;
	int y = event->xexpose.y;
#define MIN(a, b)       (((a) < (b)) ? (a) : (b))
	int w = MIN(event->xexpose.width, width);
	int h = MIN(event->xexpose.height, height);

	XCopyArea(xDisplay, xImagePixmap, XtWindow(widget),
	    xWinGc, x, y, w, h, x, y);
    }
}

/*
 * Install the specified image.  lpi specifies
 * the lines/inch of the source image.  The
 * image is resized to fit the display page using
 * a 3 x N box filter, where N depends on the
 * image height.  Typically this is either 3 x 3
 * or 3 x 2, for 196 lpi and 98 lpi images,
 * respectively.
 */
void
ImageView::setImage(const Bitmap& b, float lpi, fxBool doUpdate)
{
    float pagelen = b.r.h / lpi;
    u_int dheight = (u_int)(pagelen * dispResolution);
    if (dheight > height)
	dheight = height;
    filterHeight = (u_short) fceil(float(b.r.h) / float(dheight));
    setupStepTables(b.r.w);

    int step = b.r.h;
    int limit = dheight;
    int err = 0;
    u_int sy = 0;
    u_char* row = raster + (height-1)*width;
    for (u_int dy = 0; dy < dheight; dy++) {
	/*
	 * Calculate the number of source rows that must be
	 * sampled to calculate the filtered destination.
	 * We record the pointers to the rows and also use
	 * this information to select a colormap segment
	 * for mapping the final pixel value.
	 */
	u_char* rows[10];
	int nrows = 1;
	rows[0] = (u_char*) b.addr(0,sy);
	err += step;
	while (err >= limit) {
	    err -= limit;
	    sy++;
	    if (err >= limit)
		rows[nrows++] = (u_char*) b.addr(0,sy);
	}
	u_char* cm = (nrows == filterHeight ? cmap : cmap+16);
	u_short x = width-1;
	switch (nrows) {
	case 1: setrow1(row, cm, rows); break;
	case 2: setrow2(row, cm, rows); break;
	case 3: setrow3(row, cm, rows); break;
	case 4: setrow4(row, cm, rows); break;
	default:
	    fprintf(stderr, "nrows = %d\n", nrows);
	    break;
	}
	row -= width;
    }
    for (; dy < height; dy++) {
	whiterow(row);
	row -= width;
    }
    needCmap = TRUE;
    haveImage = TRUE;
    putImage = TRUE;
    if (isOpened() && doUpdate)
	update();
}

#define	UNROLL(op) {				\
    u_char* bits0;				\
    u_char* bits1;				\
    u_char* src0;				\
    u_int x;					\
    for (x = width; x >= 4; x -= 4) {		\
	op(0); op(1); op(2); op(3);		\
	bs0 += 4, bs1 += 4, ro += 4, row += 4;	\
    }						\
    switch (x) {				\
    case 3: op(2);				\
    case 2: op(1);				\
    case 1: op(0);				\
    }						\
}

#define	ROWOP1(n) {				\
    bits0 = bs0[n];				\
    bits1 = bs1[n];				\
    src0 = rows[0] + ro[n];			\
    row[n] = cm[bits0[src0[0]]+bits1[src0[1]]];	\
}

void
ImageView::setrow1(u_char* row, u_char* cm, u_char* rows[])
{
    u_char** bs0 = bitstep0;
    u_char** bs1 = bitstep1;
    u_short* ro = rowoff;

    UNROLL(ROWOP1);
}

#define	ROWOP2(n) {				\
    bits0 = bs0[n];				\
    bits1 = bs1[n];				\
    src0 = rows[0] + ro[n];			\
    src1 = rows[1] + ro[n];			\
    ix = bits0[src0[0]] + bits1[src0[1]]	\
       + bits0[src1[0]] + bits1[src1[1]];	\
    row[n] = cm[ix];				\
}

void
ImageView::setrow2(u_char* row, u_char* cm, u_char* rows[])
{
    u_char** bs0 = bitstep0;
    u_char** bs1 = bitstep1;
    u_short* ro = rowoff;

    u_short ix;
    u_char* src1;
    UNROLL(ROWOP2);
}

#define	ROWOP3(n) {				\
    bits0 = bs0[n];				\
    bits1 = bs1[n];				\
    src0 = rows[0] + ro[n];			\
    src1 = rows[1] + ro[n];			\
    src2 = rows[2] + ro[n];			\
    ix = bits0[src0[0]] + bits1[src0[1]]	\
       + bits0[src1[0]] + bits1[src1[1]]	\
       + bits0[src2[0]] + bits1[src2[1]];	\
    row[n] = cm[ix];				\
}

void
ImageView::setrow3(u_char* row, u_char* cm, u_char* rows[])
{
    u_char** bs0 = bitstep0;
    u_char** bs1 = bitstep1;
    u_short* ro = rowoff;

    u_short ix;
    u_char* src1;
    u_char* src2;
    UNROLL(ROWOP3);
}

#define	ROWOP4(n) {				\
    bits0 = bs0[n];				\
    bits1 = bs1[n];				\
    src0 = rows[0] + ro[n];			\
    src1 = rows[1] + ro[n];			\
    src2 = rows[2] + ro[n];			\
    src3 = rows[3] + ro[n];			\
    ix = bits0[src0[0]] + bits1[src0[1]]	\
       + bits0[src1[0]] + bits1[src1[1]]	\
       + bits0[src2[0]] + bits1[src2[1]]	\
       + bits0[src3[0]] + bits1[src3[1]];	\
    row[n] = cm[ix];				\
}

void
ImageView::setrow4(u_char* row, u_char* cm, u_char* rows[])
{
    u_char** bs0 = bitstep0;
    u_char** bs1 = bitstep1;
    u_short* ro = rowoff;

    u_short ix;
    u_char* src1;
    u_char* src2;
    u_char* src3;
    UNROLL(ROWOP4);
}

#include "bitcount.h"

/*
 * Calculate the horizontal stepping tables according
 * to the widths of the source and destination images.
 */
void
ImageView::setupStepTables(u_short sw)
{
    if (stepSrcWidth != sw || stepDstWidth != width) {
	int step = sw;
	int limit = width;
	int err = 0;
	u_int sx = 1;
	for (u_int x = 0; x < width; x++) {
	    bitstep0[x] = bitcount0[sx & 7];
	    bitstep1[x] = bitcount1[sx & 7];
	    rowoff[x] = sx >> 3;
	    err += step;
	    while (err >= limit) {
		err -= limit;
		sx++;
	    }
	}
	stepSrcWidth = sw;
	stepDstWidth = width;
    }
}

/*
 * Fill a row with white.
 */
void
ImageView::whiterow(u_char* row)
{
    u_short c = 0;
    if (photometric == PHOTOMETRIC_MINISBLACK)
	c += 3*filterHeight;
    u_short x = width-1;
    do
	row[x] = c;
    while (x-- != 0);
}

static int clamp(float v, int low, int high)
    { return (v < low ? low : v > high ? high : (int)v); }

static void
expFill(float pct[], u_int p, u_int n)
{
    u_int c = (p * n) / 100;
    for (u_int i = 1; i < c; i++)
	pct[i] = 1-fexp(i/float(n-1))/ M_E;
    for (; i < n; i++)
	pct[i] = 0.;
}

/*
 * Setup the two colormaps -- one for rows crafted
 * from filterHeight source rows and one for those
 * crafted from filterHeight-1 source rows.
 */
void
ImageView::setupCmap()
{
    if (filterHeight > 0) {
	// 3 is for the fixed width horizontal filtering
	setupCmap(3*filterHeight+1, 0);
	setupCmap(3*(filterHeight-1)+1, 16);
	needCmap = FALSE;
    } else
	needCmap = TRUE;
}

void
ImageView::setupCmap(u_int n, u_short base)
{
    float pct[40];			// known to be large enough
    pct[0] = 1;				// force white
    u_int i;
    switch (contrast) {
    case EXP50: expFill(pct, 50, n); break;
    case EXP60:	expFill(pct, 60, n); break;
    case EXP70:	expFill(pct, 70, n); break;
    case EXP80:	expFill(pct, 80, n); break;
    case EXP90:	expFill(pct, 90, n); break;
    case EXP:	expFill(pct, 100, n); break;
    case LINEAR:
	for (i = 1; i < n; i++)
	    pct[i] = 1-float(i)/(n-1);
	break;
    }
    XColor xc[64];
    switch (photometric) {
    case PHOTOMETRIC_MINISBLACK:
	for (i = 0; i < n; i++) {
	    xc[i].pixel = (u_long) cmap[base+i];
	    u_short c = clamp(255*pct[(n-1)-i], 0, 255);
	    xc[i].red = xc[i].green = xc[i].blue = c<<8;
	    xc[i].flags = DoRed | DoGreen | DoBlue;
	}
	break;
    case PHOTOMETRIC_MINISWHITE:
	for (i = 0; i < n; i++) {
	    xc[i].pixel = (u_long) cmap[base+i];
	    u_short c = clamp(255*pct[i], 0, 255);
	    xc[i].red = xc[i].green = xc[i].blue = c<<8;
	    xc[i].flags = DoRed | DoGreen | DoBlue;
	}
	break;
    }
    (void) XStoreColors(xDisplay, DefaultColormap(xDisplay, xScreen), xc, n);
}

/*
 * Set the current contrast.
 */
void
ImageView::setContrast(Contrast c, fxBool doUpdate)
{
    contrast = c;
    if (doUpdate)
	setupCmap();
    else
	needCmap = TRUE;
}

/*
 * Set the current photometric
 * interpretation for the image.
 */
void
ImageView::setPhotometric(u_short photo, fxBool doUpdate)
{
    photometric = photo;
    if (doUpdate)
	setupCmap();
    else
	needCmap = TRUE;
}

/*
 * Flip an image in Y.
 */
void
ImageView::flip(fxBool doUpdate)
{
    int h = height;
    int mid = h / 2;
    if (h & 1) mid++;
    u_char* tmp = new u_char[width];
    int ytop = height-1;
    int ybot = 0;
    int rowbytes = width * sizeof (u_char);
    for (; h > mid; h--) {
	memcpy(tmp, &raster[ytop*width], rowbytes);
	memcpy(&raster[ytop*width], &raster[ybot*width], rowbytes);
	memcpy(&raster[ybot*width], tmp, rowbytes);
	ytop--, ybot++;
    }
    delete tmp;
    putImage = TRUE;
    if (isOpened() && doUpdate)
	update();
}

/*
 * Flip an image in X.
 */
void
ImageView::reverse(fxBool doUpdate)
{
    int mid = width / 2;
    for (int y = height-1; y >= 0; y--) {
	u_char* data = &raster[y*width];
	int top = width-1;
	int bot = 0;
	for (int x = width; x > mid; x--) {
	    u_short b = data[top];
	    data[top] = data[bot];
	    data[bot] = b;
	    top--, bot++;
	}
    }
    putImage = TRUE;
    if (isOpened() && doUpdate)
	update();
}
