/*	$Header: /usr/people/sam/fax/contrib/faxview/RCS/FAXViewer.c++,v 1.20 93/09/15 16:29:01 sam Exp $
/*
 * Copyright (c) 1990, 1991, 1992, 1993 Sam Leffler
 * Copyright (c) 1991, 1992, 1993 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */
#include "tiffio.h"
#include <osfcn.h>

#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xproto.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Label.h>
#include <X11/cursorfont.h>
#define XK_MISCELLANY
#include <X11/keysymdef.h>

#include "FAXViewer.h"
#include "Bitmap.h"

const int ORIENT_NONE		= 0x0;
const int ORIENT_FLIP		= 0x1;
const int ORIENT_REVERSE	= 0x2;
const int ORIENT_OPS		= 0x3;

Display* xDisplay;
int	xScreen;
Visual*	xVisual;
FAXViewer* app;

/*
 * Xt data structures
 */
String FAXViewer::buttonStrings[] =
    { "File", "Page", "View" };

XrmOptionDescRec FAXViewer::shellOptions[] = {
    { "-help",		"*help",		XrmoptionNoArg,
      (caddr_t) "True" },
    { "-big",		"*small",		XrmoptionNoArg,
      (caddr_t) "False" },
    { "-small",		"*small",		XrmoptionNoArg,
      (caddr_t) "True" },
    { "-lsb",		"*fillOrder",		XrmoptionNoArg,
      (caddr_t) FILLORDER_LSB2MSB },
    { "-msb",		"*fillOrder",		XrmoptionNoArg,
      (caddr_t) FILLORDER_MSB2LSB },
    { "-uudecode",	"*uudecode",		XrmoptionNoArg,
      (caddr_t) "True" },
};

typedef struct {
    Boolean help;
    Boolean small;
    Boolean uudecode;
    int fillOrder;
} AppData, *AppDataPtr;

AppData appData;

XtResource FAXViewer::clientResources[] = {
    { "help", XtCBoolean, XtRBoolean, sizeof(Boolean),
      XtOffset(AppDataPtr, help), XtRImmediate, (XtPointer) False },
    { "fillOrder", XtCString, XtRInt, sizeof (int),
      XtOffset(AppDataPtr, fillOrder), XtRImmediate, (XtPointer) -1 },
    { "small", XtCBoolean, XtRBoolean, sizeof (Boolean),
      XtOffset(AppDataPtr, small), XtRImmediate, (XtPointer) False },
    { "uudecode", XtCBoolean, XtRBoolean, sizeof (Boolean),
      XtOffset(AppDataPtr, uudecode), XtRImmediate, (XtPointer) False },
};

Arg FAXViewer::formArgs[] = { { XtNresizable, True } };
Arg FAXViewer::listArgs[] = {
    { XtNresizable,	False },
    { XtNdefaultColumns,3 },
    { XtNforceColumns,	True },
    { XtNlist,		(int) buttonStrings },
    { XtNnumberStrings, XtNumber(buttonStrings) },
    { XtNtop,		XtChainTop },
    { XtNleft,		XtChainLeft },
    { XtNbottom,	XtChainTop },
    { XtNright,		XtChainLeft }
};
Arg FAXViewer::labelArgs[] = {
    { XtNresizable,	True },
    { XtNwidth,		80 },
    { XtNborderWidth,	0 },
    { XtNjustify,	XtJustifyRight },
    { XtNtop,		XtChainTop },
    { XtNleft,		XtChainLeft },
    { XtNbottom,	XtChainTop },
    { XtNright,		XtChainLeft }
};
Arg FAXViewer::imageArgs[] = {
    { XtNresizable,	True },
    { XtNborderWidth,	1 },
    { XtNtop,		XtChainTop },
    { XtNleft,		XtChainLeft },
    { XtNbottom,	XtChainTop },
    { XtNright,		XtChainLeft }
};

static void NextProc(Widget w, XEvent* ev, String* params, Cardinal nparams);
static void PrevProc(Widget w, XEvent* ev, String* params, Cardinal nparams);
static void QuitProc(Widget w, XEvent* ev, String* params, Cardinal nparams);
static void ResizeProc(Widget w, XEvent* ev, String* params, Cardinal nparams);
XtActionsRec FAXViewer::actionsTable[] = {
    { "quit",		(XtActionProc) QuitProc },
    { "nextPage",	(XtActionProc) NextProc },
    { "prevPage",	(XtActionProc) PrevProc },
    { "notifyresize",	(XtActionProc) ResizeProc }
};
char FAXViewer::translationsTable[] = "\
    <Key>q:     quit()			\n\
    <Key>Q:     quit()			\n\
    <Message>WM_PROTOCOLS: quit()	\n\
    <Key>n:     nextPage()		\n\
    <Key>space:	nextPage()		\n\
    <Key>Next:	nextPage()		\n\
    <Btn1Down>:	nextPage()		\n\
    <Key>p:     prevPage()		\n\
    <Key>Prior:	prevPage()		\n\
    <Btn2Down>:	prevPage()		\n\
    <Configure>:notifyresize()		\n\
    <Expose>:	notifyresize()		\
";

static int XTiffErrorHandler(Display* display, XErrorEvent* ev);

FAXViewer::FAXViewer()
{
    tif = 0;
    pageView = 0;
    curDirnum = -1;
    maxDirnum = -1;
    curPageNum = -1;
    removeFile = FALSE;
    pages = new Page;
    pages[0].print = TRUE;
}

FAXViewer::~FAXViewer()
{
}

static fxBool
isFAXImage(TIFF* tif)
{
    u_short w;
    if (TIFFGetField(tif, TIFFTAG_SAMPLESPERPIXEL, &w) && w != 1)
	return (FALSE);
    if (TIFFGetField(tif, TIFFTAG_BITSPERSAMPLE, &w) && w != 1)
	return (FALSE);
    if (!TIFFGetField(tif, TIFFTAG_COMPRESSION, &w) ||
      (w != COMPRESSION_CCITTFAX3 && w != COMPRESSION_CCITTFAX4))
	return (FALSE);
    if (!TIFFGetField(tif, TIFFTAG_PHOTOMETRIC, &w) ||
      (w != PHOTOMETRIC_MINISWHITE && w != PHOTOMETRIC_MINISBLACK))
	return (FALSE);
    return (TRUE);
}

void
FAXViewer::usage(const char* appName)
{
    fprintf(stderr, "usage: %s [-b] [-s] [-u] [file.tif]\n", appName);
    exit(-1);
}

void
FAXViewer::initialize(int argc, char** argv)
{
    shellWidget = XtInitialize(argv[0], "FAXViewer",
	shellOptions, XtNumber(shellOptions),
	(Cardinal*) &argc, argv);
    XSetErrorHandler(XTiffErrorHandler);
    XtGetApplicationResources(shellWidget, (XtPointer) &appData,
        (XtResourceList) clientResources, (Cardinal) XtNumber(clientResources),
        (ArgList) NULL, (Cardinal) 0);
    if (appData.help)
        usage(argv[0]);
    if (argc == 1) {			// decode standard input
	if (appData.uudecode) {
	    uudecode(stdin, filename);
	    removeFile = TRUE;		// remove temp file on exit
	} else
	    usage(argv[0]);
    } else
	filename = argv[1];
    tif = TIFFOpen(filename, "r");
    if (!tif || !isFAXImage(tif)) {
	fprintf(stderr, "%s: Not a FAX image.\n", (char*) filename);
	exit(-2);
    }
    TIFFSetWarningHandler(0);
    xDisplay = XtDisplay(shellWidget);
    xScreen = DefaultScreen(xDisplay);
    getVisual();
    setupPageMenu();
}

/*
 * Locate an appropriate visual.
 */
void
FAXViewer::getVisual()
{
    int n;
    XVisualInfo templ;
    templ.screen = xScreen;
    XVisualInfo* vlist = XGetVisualInfo(xDisplay, VisualScreenMask, &templ, &n);
    if (n == 0) {
        fprintf(stderr, "xtiff: No visual list available\n");
        exit(0);
    }
    for (int i = 0; i < n; i++) {
	const XVisualInfo& vl = vlist[i];
        if (vl.depth == 8 &&
	  (vl.c_class == GrayScale || vl.c_class == PseudoColor) && 
	  vl.visual->map_entries >= (1 << vl.depth)) {
            xVisual = vl.visual;
            xRedMask = vl.red_mask;
            xGreenMask = vl.green_mask;
            xBlueMask = vl.blue_mask;
	    break;
        }
    }
    XFree((char *) vlist);
    if (!xVisual) {
	fprintf(stderr, "Can not locate 8-bit color visual for display.\n");
	exit(-1);
    }
}

void
FAXViewer::setupPageMenu()
{
    // construct page-number <-> directory map
    maxPageNum = -1;
    minPageNum = 10000;
    do {
	short pn, total;
	if (!TIFFGetField(tif, TIFFTAG_PAGENUMBER, &pn, &total))
	    pn = TIFFCurrentDirectory(tif);		// makeup page number
	if (pn > maxDirnum) {
	    maxDirnum = pn;
	    pages = (Page*)realloc(pages, (maxDirnum+1) * sizeof (Page));
	}
	pages[pn].print = TRUE;
	pages[pn].read = FALSE;
	pages[pn].orient = 0;
	if (pn < minPageNum)
	    minPageNum = pn;
	if (pn > maxPageNum)
	    maxPageNum = pn;
    } while (TIFFReadDirectory(tif));
}

inline int DEC(char c) { return ((c - ' ') & 077); }

void
FAXViewer::uudecode(FILE* in, fxStr& temp)
{
    char buf[1024];
    do {
	if (!fgets(buf, sizeof (buf), in)) {
	    fprintf(stderr, "Missing \"begin\" line\n");
	    exit(-1);
	}
    } while (strncmp(buf, "begin ", 6));
    temp = "/tmp/faxvXXXXXX";
    int fd = mkstemp(temp);
    if (fd == -1) {
	fprintf(stderr, "Could not create temp file \"%s\"\n", (char*)temp);
	exit(-2);
    }
    FILE* out = fdopen(fd, "w");
    for (;;) {
	if (!fgets(buf, sizeof (buf), in)) {
	    fprintf(stderr, "Warning, improperly formatted data\n");
	    break;
	}
	char* cp = buf;
	int n = DEC(*cp);
	if (n <= 0)
	    break;
	int c;
	for (cp++; n >= 3; cp += 4, n -= 3) {
	    c = (DEC(cp[0])<<2) | (DEC(cp[1])>>4); putc(c, out);
	    c = (DEC(cp[1])<<4) | (DEC(cp[2])>>2); putc(c, out);
	    c = (DEC(cp[2])<<6) |  DEC(cp[3]);	   putc(c, out);
	}
	if (n >= 1)
	    c = (DEC(cp[0])<<2) | (DEC(cp[1])>>4), putc(c, out);
	if (n >= 2)
	    c = (DEC(cp[1])<<4) | (DEC(cp[2])>>2), putc(c, out);
	if (n >= 3)
	    c = (DEC(cp[2])<<6) |  DEC(cp[3]),	   putc(c, out);
    }
    while (fgets(buf, sizeof (buf), in))	// flush input
	;
    fclose(out);
}

void
FAXViewer::selectPage(u_short pn)
{
#ifdef notdef
    if (curPageNum != -1)
	pages[curPageNum].menu->setState(0);
    pages[pn].menu->setState(1);
    if (pn == minPageNum)
	prevPage->disableByClick(0);
    else
	prevPage->enableByClick(0);
    if (pn == maxPageNum)
	nextPage->disableByClick(0);
    else
	nextPage->enableByClick(0);
#endif
    curPageNum = pn;
}

void
FAXViewer::gotoDirectory(int dirnum)
{
    if (dirnum == curDirnum)
	return;
    if (!TIFFSetDirectory(tif, dirnum)) {
	fprintf(stderr, "Can not access directory %d in TIFF file.\n", dirnum);
	return;
    }
    curDirnum = dirnum;
    u_short pn, total;
    if (!TIFFGetField(tif, TIFFTAG_PAGENUMBER, &pn, &total))
	pn = dirnum;			// makeup page number
    selectPage(pn);
    beginSlowOperation();
    u_short photometric = PHOTOMETRIC_MINISWHITE;
    (void) TIFFGetField(tif, TIFFTAG_PHOTOMETRIC, &photometric);
    pageView->setPhotometric(photometric, FALSE);
    float yres;
    if (!TIFFGetField(tif, TIFFTAG_YRESOLUTION, &yres)) {
	TIFFWarning(TIFFFileName(tif), "No y-resolution, assuming 98 lpi");
	yres = 98;					/* XXX */
    }
    u_short unit;
    TIFFGetField(tif, TIFFTAG_RESOLUTIONUNIT, &unit);
    if (unit == RESUNIT_CENTIMETER)
	yres *= 25.4;
    u_long w, h;
    (void) TIFFGetField(tif, TIFFTAG_IMAGEWIDTH, &w);
    (void) TIFFGetField(tif, TIFFTAG_IMAGELENGTH, &h);
    if (!page || page->r.w != w || page->r.h != h) {
	if (page)
	    delete page;
	page = new Bitmap((u_int) w, (u_int) h);
    }
    u_short orient;
    if (!TIFFGetField(tif, TIFFTAG_ORIENTATION, &orient))
	orient = ORIENTATION_TOPLEFT;
    switch (orient) {
    case ORIENTATION_BOTRIGHT:
    case ORIENTATION_RIGHTBOT:	/* XXX */
    case ORIENTATION_LEFTBOT:	/* XXX */
	orient = ORIENTATION_BOTLEFT;
	break;
    case ORIENTATION_TOPRIGHT:
    case ORIENTATION_RIGHTTOP:	/* XXX */
    case ORIENTATION_LEFTTOP:	/* XXX */
    default:
	orient = ORIENTATION_TOPLEFT;
	break;
    }
    if (appData.fillOrder != -1)
	TIFFSetField(tif, TIFFTAG_FILLORDER, appData.fillOrder);
    u_long rowsperstrip;
    TIFFGetFieldDefaulted(tif, TIFFTAG_ROWSPERSTRIP, &rowsperstrip);
    u_long y = 0;
    for (u_int strip = 0; strip < TIFFNumberOfStrips(tif); strip++) {
	(void) TIFFReadEncodedStrip(tif, strip,
	    (u_char*) page->addr(0,y), (u_long) -1);
	y += rowsperstrip;
    }
    pageView->setImage(*page, yres, FALSE);
    if (!pages[pn].read) {
	switch (orient) {
	case ORIENTATION_TOPRIGHT:
	case ORIENTATION_RIGHTTOP:
	    pages[pn].orient = ORIENT_FLIP|ORIENT_REVERSE;
	    break;
	case ORIENTATION_LEFTTOP:
	case ORIENTATION_TOPLEFT:
	    pages[pn].orient = ORIENT_FLIP;
	    break;
	case ORIENTATION_BOTRIGHT:
	case ORIENTATION_RIGHTBOT:
	    pages[pn].orient = ORIENT_REVERSE;
	    break;
	}
	pages[pn].read = TRUE;
    }
    if (pages[pn].orient & ORIENT_FLIP)
	pageView->flip(FALSE);
    if (pages[pn].orient & ORIENT_REVERSE)
	pageView->reverse(FALSE);
    pageView->update();
    setLabel();
    endSlowOperation();
    updateOrient(pages[pn].orient);
}

static XtCallbackProc SelectProc(Widget, caddr_t, caddr_t);

void
FAXViewer::open()
{
    Widget widget_list[3];
    Arg args[5];

    /*
     * Send visual, colormap, depth and iconPixmap to shellWidget.
     * Sending the visual to the shell is only possible with the advent of R4.
     */
    XtSetArg(args[0], XtNvisual, xVisual);
    XtSetArg(args[1], XtNcolormap, xColormap);
    XtSetArg(args[2], XtNdepth, 8);
#ifdef notdef
    XtSetArg(args[3], XtNiconPixmap,
        XCreateBitmapFromData(xDisplay, RootWindow(xDisplay, xScreen),
            xtifficon_bits, xtifficon_width, xtifficon_height));
    XtSetArg(args[4], XtNallowShellResize, True);
    XtSetValues(shellWidget, args, 5);
#else
    XtSetArg(args[3], XtNallowShellResize, True);
    XtSetValues(shellWidget, args, 4);
#endif

    /*
     * Craft widget instance hierarchy
     */
    formWidget = XtCreateManagedWidget("form", formWidgetClass,
        shellWidget, formArgs, XtNumber(formArgs));
        widget_list[0] = listWidget = XtCreateWidget("list",
            listWidgetClass, formWidget, listArgs, XtNumber(listArgs));
        widget_list[1] = labelWidget = XtCreateWidget("label",
            labelWidgetClass, formWidget, labelArgs, XtNumber(labelArgs));
        widget_list[2] = imageWidget = XtCreateWidget("image",
            widgetClass, formWidget, imageArgs, XtNumber(imageArgs));
    XtManageChildren(widget_list, XtNumber(widget_list));

    pageView = new ImageView(appData.small, imageWidget);

    /*
     * formWidget uses these constraints but they are stored in the children.
     */
    XtSetArg(args[0], XtNfromVert, listWidget);
    XtSetValues(imageWidget, args, 1);
    XtSetArg(args[0], XtNfromHoriz, listWidget);
    XtSetValues(labelWidget, args, 1);

    XtAddCallback(listWidget, XtNcallback, (XtCallbackProc) SelectProc,
        (XtPointer) NULL);

    XtAddActions(actionsTable, XtNumber(actionsTable));
    XtSetArg(args[0], XtNtranslations,
        XtParseTranslationTable(translationsTable));
    XtSetValues(formWidget, &args[0], 1);
    XtSetValues(imageWidget, &args[0], 1);

    XtRealizeWidget(shellWidget);

    window_attributes.cursor = XCreateFontCursor(xDisplay, XC_fleur);
    XChangeWindowAttributes(xDisplay, XtWindow(imageWidget),
        CWCursor, &window_attributes);

    pageView->open();

    setExpContrast();
    gotoDirectory(0);
}

void
FAXViewer::close()
{
    if (removeFile)
	(void) unlink(filename);
    exit(0);
}

void
FAXViewer::expose(XEvent* ev)
{
    pageView->expose(ev);
}

void
FAXViewer::setLabel()
{
    char buffer[BUFSIZ];
    Arg args[1];

    sprintf(buffer, "%s (page %d)",
	(char*) filename, TIFFCurrentDirectory(tif));
    XtSetArg(args[0], XtNlabel, buffer);
    XtSetValues(labelWidget, args, 1);
}

void
FAXViewer::printPage()
{
    u_short pn, total;
    if (!TIFFGetField(tif, TIFFTAG_PAGENUMBER, &pn, &total))
	pn = curDirnum;			// makeup page number
    char buf[1024];
    sprintf(buf, "fax2ps -p %d %s | lp -s", pn, (char*) filename);
    printCmd = buf;
    print();
}

void
FAXViewer::printAll()
{
    printCmd = "fax2ps " | filename | " | lp -s";
    print();
}

void
FAXViewer::print()
{
    beginSlowOperation();
    if (system(printCmd))
	fprintf(stderr, "Print command problem.\n");
    endSlowOperation();
}

void
FAXViewer::setOrientation(int o)
{
    beginSlowOperation();
    if (o & ORIENT_FLIP)
	pageView->flip(FALSE);
    if (o & ORIENT_REVERSE)
	pageView->reverse(FALSE);
    endSlowOperation();
    pageView->update();
    updateOrient(pages[curPageNum].orient ^= (o & ORIENT_OPS));
}

void
FAXViewer::updateOrient(int)
{
#ifdef notdef
    switch (o) {
    case ORIENT_NONE:
	orientItems[ORIENT_FLIP]->setState(0);
	orientItems[ORIENT_REVERSE]->setState(0);
	orientItems[ORIENT_FLIP+ORIENT_REVERSE]->setState(0);
	break;
    case ORIENT_FLIP:
	orientItems[ORIENT_FLIP]->setState(1);
	orientItems[ORIENT_REVERSE]->setState(0);
	orientItems[ORIENT_FLIP+ORIENT_REVERSE]->setState(0);
	break;
    case ORIENT_REVERSE:
	orientItems[ORIENT_FLIP]->setState(0);
	orientItems[ORIENT_REVERSE]->setState(1);
	orientItems[ORIENT_FLIP+ORIENT_REVERSE]->setState(0);
	break;
    case ORIENT_FLIP+ORIENT_REVERSE:
	orientItems[ORIENT_FLIP]->setState(0);
	orientItems[ORIENT_REVERSE]->setState(0);
	orientItems[ORIENT_FLIP+ORIENT_REVERSE]->setState(1);
	break;
    }
#endif
}

void FAXViewer::setLinearContrast()	{ setContrast(ImageView::LINEAR); }
void FAXViewer::setExpContrast()	{ setContrast(ImageView::EXP); }
void FAXViewer::setExp50Contrast()	{ setContrast(ImageView::EXP50); }
void FAXViewer::setExp60Contrast()	{ setContrast(ImageView::EXP60); }
void FAXViewer::setExp70Contrast()	{ setContrast(ImageView::EXP70); }

void
FAXViewer::setContrast(Contrast c)
{
    pageView->setContrast(c);
}

void
FAXViewer::beginSlowOperation()
{
    prevCursor = window_attributes.cursor;
    window_attributes.cursor = XCreateFontCursor(xDisplay, XC_watch);
    XChangeWindowAttributes(xDisplay, XtWindow(imageWidget),
        CWCursor, &window_attributes);
    XFlush(xDisplay);
}

void
FAXViewer::endSlowOperation()
{
    window_attributes.cursor = prevCursor;
    XChangeWindowAttributes(xDisplay, XtWindow(imageWidget),
        CWCursor, &window_attributes);
    XFlush(xDisplay);
}

static XtCallbackProc
SelectProc(Widget w, caddr_t arg, caddr_t)
{
    XawListReturnStruct* list_return = XawListShowCurrent(w);
    FAXViewer* app = (FAXViewer*) arg;
#ifdef notdef
    switch (list_return->list_index) {
    case FAXViewer::ButtonQuit:
	app->close();
        break;
    case FAXViewer::ButtonPage:
        app->gotoPrevPage();
        break;
    case FAXViewer::ButtonPage:
        app->gotoNextPage();
        break;
    default:
        fprintf(stderr, "error in SelectProc\n");
        exit(0);
    }
#endif
    XawListUnhighlight(w);
    return ((XtCallbackProc) SelectProc);
}

static void
ResizeProc(Widget, XEvent* ev, String*, Cardinal) { app->expose(ev); }

static void NextProc(Widget, XEvent*, String*, Cardinal) {app->gotoNextPage();}
static void PrevProc(Widget, XEvent*, String*, Cardinal) {app->gotoPrevPage();}
static void QuitProc(Widget, XEvent*, String*, Cardinal) {app->close();}

static int
XTiffErrorHandler(Display* display, XErrorEvent* ev)
{
    char message[80];
    /*
     * Some X servers limit the size of pixmaps.
     */
    if (ev->error_code == BadAlloc && ev->request_code == X_CreatePixmap)
        fprintf(stderr, "xtiff: requested pixmap is too large for display\n");
    else {
        XGetErrorText(display, ev->error_code, message, 80);
        fprintf(stderr, "xtiff: error code %s\n", message);
    }
    exit(0);
    return (0);
}

extern "C" int
main(int argc, char** argv)
{
    app = new FAXViewer;
    app->initialize(argc, argv);
    app->open();
    XtMainLoop();
    delete app;
    return 0;
}
